(function() {
  var CSON, CompositeDisposable, Disposable, Emitter, File, ScopedPropertyStore, Snippet, SnippetExpansion, _, async, fs, path, ref,
    slice = [].slice;

  path = require('path');

  ref = require('atom'), Emitter = ref.Emitter, Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable, File = ref.File;

  _ = require('underscore-plus');

  async = require('async');

  CSON = require('season');

  fs = require('fs-plus');

  ScopedPropertyStore = require('scoped-property-store');

  Snippet = require('./snippet');

  SnippetExpansion = require('./snippet-expansion');

  module.exports = {
    loaded: false,
    activate: function() {
      var snippets;
      this.userSnippetsPath = null;
      this.snippetIdCounter = 0;
      this.parsedSnippetsById = new Map;
      this.scopedPropertyStore = new ScopedPropertyStore;
      this.subscriptions = new CompositeDisposable;
      this.subscriptions.add(atom.workspace.addOpener((function(_this) {
        return function(uri) {
          if (uri === 'atom://.atom/snippets') {
            return atom.workspace.openTextFile(_this.getUserSnippetsPath());
          }
        };
      })(this)));
      this.loadAll();
      this.watchUserSnippets((function(_this) {
        return function(watchDisposable) {
          return _this.subscriptions.add(watchDisposable);
        };
      })(this));
      snippets = this;
      this.subscriptions.add(atom.commands.add('atom-text-editor', {
        'snippets:expand': function(event) {
          var editor;
          editor = this.getModel();
          if (snippets.snippetToExpandUnderCursor(editor)) {
            snippets.clearExpansions(editor);
            return snippets.expandSnippetsUnderCursors(editor);
          } else {
            return event.abortKeyBinding();
          }
        },
        'snippets:next-tab-stop': function(event) {
          var editor;
          editor = this.getModel();
          if (!snippets.goToNextTabStop(editor)) {
            return event.abortKeyBinding();
          }
        },
        'snippets:previous-tab-stop': function(event) {
          var editor;
          editor = this.getModel();
          if (!snippets.goToPreviousTabStop(editor)) {
            return event.abortKeyBinding();
          }
        },
        'snippets:available': function(event) {
          var SnippetsAvailable, editor;
          editor = this.getModel();
          SnippetsAvailable = require('./snippets-available');
          if (snippets.availableSnippetsView == null) {
            snippets.availableSnippetsView = new SnippetsAvailable(snippets);
          }
          return snippets.availableSnippetsView.toggle(editor);
        }
      }));
      return this.subscriptions.add(atom.workspace.observeTextEditors((function(_this) {
        return function(editor) {
          return _this.clearExpansions(editor);
        };
      })(this)));
    },
    deactivate: function() {
      var ref1;
      if ((ref1 = this.emitter) != null) {
        ref1.dispose();
      }
      this.emitter = null;
      this.editorSnippetExpansions = null;
      return atom.config.transact((function(_this) {
        return function() {
          return _this.subscriptions.dispose();
        };
      })(this));
    },
    getUserSnippetsPath: function() {
      if (this.userSnippetsPath != null) {
        return this.userSnippetsPath;
      }
      this.userSnippetsPath = CSON.resolve(path.join(atom.getConfigDirPath(), 'snippets'));
      if (this.userSnippetsPath == null) {
        this.userSnippetsPath = path.join(atom.getConfigDirPath(), 'snippets.cson');
      }
      return this.userSnippetsPath;
    },
    loadAll: function(callback) {
      return this.loadBundledSnippets((function(_this) {
        return function(bundledSnippets) {
          return _this.loadPackageSnippets(function(packageSnippets) {
            return _this.loadUserSnippets(function(userSnippets) {
              atom.config.transact(function() {
                var filepath, i, len, ref1, results1, snippetSet, snippetsBySelector;
                ref1 = [bundledSnippets, packageSnippets, userSnippets];
                results1 = [];
                for (i = 0, len = ref1.length; i < len; i++) {
                  snippetSet = ref1[i];
                  results1.push((function() {
                    var results2;
                    results2 = [];
                    for (filepath in snippetSet) {
                      snippetsBySelector = snippetSet[filepath];
                      results2.push(this.add(filepath, snippetsBySelector));
                    }
                    return results2;
                  }).call(_this));
                }
                return results1;
              });
              return _this.doneLoading();
            });
          });
        };
      })(this));
    },
    loadBundledSnippets: function(callback) {
      var bundledSnippetsPath;
      bundledSnippetsPath = CSON.resolve(path.join(__dirname, 'snippets'));
      return this.loadSnippetsFile(bundledSnippetsPath, function(snippets) {
        var snippetsByPath;
        snippetsByPath = {};
        snippetsByPath[bundledSnippetsPath] = snippets;
        return callback(snippetsByPath);
      });
    },
    loadUserSnippets: function(callback) {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return fs.stat(userSnippetsPath, (function(_this) {
        return function(error, stat) {
          if (stat != null ? stat.isFile() : void 0) {
            return _this.loadSnippetsFile(userSnippetsPath, function(snippets) {
              var result;
              result = {};
              result[userSnippetsPath] = snippets;
              return callback(result);
            });
          } else {
            return callback({});
          }
        };
      })(this));
    },
    watchUserSnippets: function(callback) {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return fs.stat(userSnippetsPath, (function(_this) {
        return function(error, stat) {
          var e, message, userSnippetsFile, userSnippetsFileDisposable;
          if (stat != null ? stat.isFile() : void 0) {
            userSnippetsFileDisposable = new CompositeDisposable();
            userSnippetsFile = new File(userSnippetsPath);
            try {
              userSnippetsFileDisposable.add(userSnippetsFile.onDidChange(function() {
                return _this.handleUserSnippetsDidChange();
              }));
              userSnippetsFileDisposable.add(userSnippetsFile.onDidDelete(function() {
                return _this.handleUserSnippetsDidChange();
              }));
              userSnippetsFileDisposable.add(userSnippetsFile.onDidRename(function() {
                return _this.handleUserSnippetsDidChange();
              }));
            } catch (error1) {
              e = error1;
              message = "Unable to watch path: `snippets.cson`. Make sure you have permissions\nto the `~/.atom` directory and `" + userSnippetsPath + "`.\n\nOn linux there are currently problems with watch sizes. See\n[this document][watches] for more info.\n[watches]:https://github.com/atom/atom/blob/master/docs/build-instructions/linux.md#typeerror-unable-to-watch-path";
              atom.notifications.addError(message, {
                dismissable: true
              });
            }
            return callback(userSnippetsFileDisposable);
          } else {
            return callback(new Disposable(function() {}));
          }
        };
      })(this));
    },
    handleUserSnippetsDidChange: function() {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return atom.config.transact((function(_this) {
        return function() {
          _this.clearSnippetsForPath(userSnippetsPath);
          return _this.loadSnippetsFile(userSnippetsPath, function(result) {
            return _this.add(userSnippetsPath, result);
          });
        };
      })(this));
    },
    loadPackageSnippets: function(callback) {
      var pack, packages, snippetsDirPaths;
      packages = atom.packages.getLoadedPackages();
      snippetsDirPaths = ((function() {
        var i, len, results1;
        results1 = [];
        for (i = 0, len = packages.length; i < len; i++) {
          pack = packages[i];
          results1.push(path.join(pack.path, 'snippets'));
        }
        return results1;
      })()).sort(function(a, b) {
        if (/\/app\.asar\/node_modules\//.test(a)) {
          return -1;
        } else {
          return 1;
        }
      });
      return async.map(snippetsDirPaths, this.loadSnippetsDirectory.bind(this), function(error, results) {
        return callback(_.extend.apply(_, [{}].concat(slice.call(results))));
      });
    },
    doneLoading: function() {
      this.loaded = true;
      return this.getEmitter().emit('did-load-snippets');
    },
    onDidLoadSnippets: function(callback) {
      return this.getEmitter().on('did-load-snippets', callback);
    },
    getEmitter: function() {
      return this.emitter != null ? this.emitter : this.emitter = new Emitter;
    },
    loadSnippetsDirectory: function(snippetsDirPath, callback) {
      return fs.isDirectory(snippetsDirPath, (function(_this) {
        return function(isDirectory) {
          if (!isDirectory) {
            return callback(null, {});
          }
          return fs.readdir(snippetsDirPath, function(error, entries) {
            if (error) {
              console.warn("Error reading snippets directory " + snippetsDirPath, error);
              return callback(null, {});
            }
            return async.map(entries, function(entry, done) {
              var filePath;
              filePath = path.join(snippetsDirPath, entry);
              return _this.loadSnippetsFile(filePath, function(snippets) {
                return done(null, {
                  filePath: filePath,
                  snippets: snippets
                });
              });
            }, function(error, results) {
              var filePath, i, len, ref1, snippets, snippetsByPath;
              snippetsByPath = {};
              for (i = 0, len = results.length; i < len; i++) {
                ref1 = results[i], filePath = ref1.filePath, snippets = ref1.snippets;
                snippetsByPath[filePath] = snippets;
              }
              return callback(null, snippetsByPath);
            });
          });
        };
      })(this));
    },
    loadSnippetsFile: function(filePath, callback) {
      if (!CSON.isObjectPath(filePath)) {
        return callback({});
      }
      return CSON.readFile(filePath, function(error, object) {
        var ref1, ref2;
        if (object == null) {
          object = {};
        }
        if (error != null) {
          console.warn("Error reading snippets file '" + filePath + "': " + ((ref1 = error.stack) != null ? ref1 : error));
          if ((ref2 = atom.notifications) != null) {
            ref2.addError("Failed to load snippets from '" + filePath + "'", {
              detail: error.message,
              dismissable: true
            });
          }
        }
        return callback(object);
      });
    },
    add: function(filePath, snippetsBySelector) {
      var attributes, body, name, prefix, selector, snippetsByName, unparsedSnippetsByPrefix;
      for (selector in snippetsBySelector) {
        snippetsByName = snippetsBySelector[selector];
        unparsedSnippetsByPrefix = {};
        for (name in snippetsByName) {
          attributes = snippetsByName[name];
          prefix = attributes.prefix, body = attributes.body;
          attributes.name = name;
          attributes.id = this.snippetIdCounter++;
          if (typeof body === 'string') {
            unparsedSnippetsByPrefix[prefix] = attributes;
          } else if (body == null) {
            unparsedSnippetsByPrefix[prefix] = null;
          }
        }
        this.storeUnparsedSnippets(unparsedSnippetsByPrefix, filePath, selector);
      }
    },
    getScopeChain: function(object) {
      var scopesArray;
      scopesArray = object != null ? typeof object.getScopesArray === "function" ? object.getScopesArray() : void 0 : void 0;
      if (scopesArray == null) {
        scopesArray = object;
      }
      return scopesArray.map(function(scope) {
        if (scope[0] !== '.') {
          scope = "." + scope;
        }
        return scope;
      }).join(' ');
    },
    storeUnparsedSnippets: function(value, path, selector) {
      var unparsedSnippets;
      unparsedSnippets = {};
      unparsedSnippets[selector] = {
        "snippets": value
      };
      return this.scopedPropertyStore.addProperties(path, unparsedSnippets, {
        priority: this.priorityForSource(path)
      });
    },
    clearSnippetsForPath: function(path) {
      var attributes, prefix, ref1, results1, scopeSelector;
      results1 = [];
      for (scopeSelector in this.scopedPropertyStore.propertiesForSource(path)) {
        ref1 = this.scopedPropertyStore.propertiesForSourceAndSelector(path, scopeSelector);
        for (prefix in ref1) {
          attributes = ref1[prefix];
          this.parsedSnippetsById["delete"](attributes.id);
        }
        results1.push(this.scopedPropertyStore.removePropertiesForSourceAndSelector(path, scopeSelector));
      }
      return results1;
    },
    parsedSnippetsForScopes: function(scopeDescriptor) {
      var attributes, body, bodyTree, description, descriptionMoreURL, id, leftLabel, leftLabelHTML, name, prefix, rightLabelHTML, snippet, snippets, unparsedSnippetsByPrefix;
      unparsedSnippetsByPrefix = this.scopedPropertyStore.getPropertyValue(this.getScopeChain(scopeDescriptor), "snippets");
      if (unparsedSnippetsByPrefix == null) {
        unparsedSnippetsByPrefix = {};
      }
      snippets = {};
      for (prefix in unparsedSnippetsByPrefix) {
        attributes = unparsedSnippetsByPrefix[prefix];
        if (typeof (attributes != null ? attributes.body : void 0) !== 'string') {
          continue;
        }
        id = attributes.id, name = attributes.name, body = attributes.body, bodyTree = attributes.bodyTree, description = attributes.description, descriptionMoreURL = attributes.descriptionMoreURL, rightLabelHTML = attributes.rightLabelHTML, leftLabel = attributes.leftLabel, leftLabelHTML = attributes.leftLabelHTML;
        if (!this.parsedSnippetsById.has(id)) {
          if (bodyTree == null) {
            bodyTree = this.getBodyParser().parse(body);
          }
          snippet = new Snippet({
            id: id,
            name: name,
            prefix: prefix,
            bodyTree: bodyTree,
            description: description,
            descriptionMoreURL: descriptionMoreURL,
            rightLabelHTML: rightLabelHTML,
            leftLabel: leftLabel,
            leftLabelHTML: leftLabelHTML,
            bodyText: body
          });
          this.parsedSnippetsById.set(id, snippet);
        }
        snippets[prefix] = this.parsedSnippetsById.get(id);
      }
      return snippets;
    },
    priorityForSource: function(source) {
      if (source === this.getUserSnippetsPath()) {
        return 1000;
      } else {
        return 0;
      }
    },
    getBodyParser: function() {
      return this.bodyParser != null ? this.bodyParser : this.bodyParser = require('./snippet-body-parser');
    },
    getPrefixText: function(snippets, editor) {
      var cursor, cursorSnippetPrefix, cursorWordPrefix, i, len, position, prefixStart, ref1, ref2, snippetPrefix, wordPrefix, wordRegex, wordStart;
      wordRegex = this.wordRegexForSnippets(snippets);
      ref1 = [], snippetPrefix = ref1[0], wordPrefix = ref1[1];
      ref2 = editor.getCursors();
      for (i = 0, len = ref2.length; i < len; i++) {
        cursor = ref2[i];
        position = cursor.getBufferPosition();
        prefixStart = cursor.getBeginningOfCurrentWordBufferPosition({
          wordRegex: wordRegex
        });
        cursorSnippetPrefix = editor.getTextInRange([prefixStart, position]);
        if ((snippetPrefix != null) && cursorSnippetPrefix !== snippetPrefix) {
          return null;
        }
        snippetPrefix = cursorSnippetPrefix;
        wordStart = cursor.getBeginningOfCurrentWordBufferPosition();
        cursorWordPrefix = editor.getTextInRange([wordStart, position]);
        if ((wordPrefix != null) && cursorWordPrefix !== wordPrefix) {
          return null;
        }
        wordPrefix = cursorWordPrefix;
      }
      return {
        snippetPrefix: snippetPrefix,
        wordPrefix: wordPrefix
      };
    },
    wordRegexForSnippets: function(snippets) {
      var character, i, len, prefix, prefixCharacters, prefixes;
      prefixes = {};
      for (prefix in snippets) {
        for (i = 0, len = prefix.length; i < len; i++) {
          character = prefix[i];
          prefixes[character] = true;
        }
      }
      prefixCharacters = Object.keys(prefixes).join('');
      return new RegExp("[" + (_.escapeRegExp(prefixCharacters)) + "]+");
    },
    snippetForPrefix: function(snippets, prefix, wordPrefix) {
      var longestPrefixMatch, snippet, snippetPrefix;
      longestPrefixMatch = null;
      for (snippetPrefix in snippets) {
        snippet = snippets[snippetPrefix];
        if (_.endsWith(prefix, snippetPrefix) && wordPrefix.length <= snippetPrefix.length) {
          if ((longestPrefixMatch == null) || snippetPrefix.length > longestPrefixMatch.prefix.length) {
            longestPrefixMatch = snippet;
          }
        }
      }
      return longestPrefixMatch;
    },
    getSnippets: function(editor) {
      return this.parsedSnippetsForScopes(editor.getLastCursor().getScopeDescriptor());
    },
    snippetToExpandUnderCursor: function(editor) {
      var prefixData, snippets;
      if (!editor.getLastSelection().isEmpty()) {
        return false;
      }
      snippets = this.getSnippets(editor);
      if (_.isEmpty(snippets)) {
        return false;
      }
      if (prefixData = this.getPrefixText(snippets, editor)) {
        return this.snippetForPrefix(snippets, prefixData.snippetPrefix, prefixData.wordPrefix);
      }
    },
    expandSnippetsUnderCursors: function(editor) {
      var snippet;
      if (!(snippet = this.snippetToExpandUnderCursor(editor))) {
        return false;
      }
      editor.transact((function(_this) {
        return function() {
          var cursor, cursorPosition, cursors, i, len, results1, startPoint;
          cursors = editor.getCursors();
          results1 = [];
          for (i = 0, len = cursors.length; i < len; i++) {
            cursor = cursors[i];
            cursorPosition = cursor.getBufferPosition();
            startPoint = cursorPosition.translate([0, -snippet.prefix.length], [0, 0]);
            cursor.selection.setBufferRange([startPoint, cursorPosition]);
            results1.push(_this.insert(snippet, editor, cursor));
          }
          return results1;
        };
      })(this));
      return true;
    },
    goToNextTabStop: function(editor) {
      var expansion, i, len, nextTabStopVisited, ref1;
      nextTabStopVisited = false;
      ref1 = this.getExpansions(editor);
      for (i = 0, len = ref1.length; i < len; i++) {
        expansion = ref1[i];
        if (expansion != null ? expansion.goToNextTabStop() : void 0) {
          nextTabStopVisited = true;
        }
      }
      return nextTabStopVisited;
    },
    goToPreviousTabStop: function(editor) {
      var expansion, i, len, previousTabStopVisited, ref1;
      previousTabStopVisited = false;
      ref1 = this.getExpansions(editor);
      for (i = 0, len = ref1.length; i < len; i++) {
        expansion = ref1[i];
        if (expansion != null ? expansion.goToPreviousTabStop() : void 0) {
          previousTabStopVisited = true;
        }
      }
      return previousTabStopVisited;
    },
    getExpansions: function(editor) {
      var ref1, ref2;
      return (ref1 = (ref2 = this.editorSnippetExpansions) != null ? ref2.get(editor) : void 0) != null ? ref1 : [];
    },
    clearExpansions: function(editor) {
      if (this.editorSnippetExpansions == null) {
        this.editorSnippetExpansions = new WeakMap();
      }
      return this.editorSnippetExpansions.set(editor, []);
    },
    addExpansion: function(editor, snippetExpansion) {
      return this.getExpansions(editor).push(snippetExpansion);
    },
    insert: function(snippet, editor, cursor) {
      var bodyTree;
      if (editor == null) {
        editor = atom.workspace.getActiveTextEditor();
      }
      if (cursor == null) {
        cursor = editor.getLastCursor();
      }
      if (typeof snippet === 'string') {
        bodyTree = this.getBodyParser().parse(snippet);
        snippet = new Snippet({
          name: '__anonymous',
          prefix: '',
          bodyTree: bodyTree,
          bodyText: snippet
        });
      }
      return new SnippetExpansion(snippet, editor, cursor, this);
    },
    getUnparsedSnippets: function() {
      return _.deepClone(this.scopedPropertyStore.propertySets);
    },
    provideSnippets: function() {
      return {
        bundledSnippetsLoaded: (function(_this) {
          return function() {
            return _this.loaded;
          };
        })(this),
        insertSnippet: this.insert.bind(this),
        snippetsForScopes: this.parsedSnippetsForScopes.bind(this),
        getUnparsedSnippets: this.getUnparsedSnippets.bind(this)
      };
    }
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
