(function() {
  var BufferedProcess, CompositeDisposable, Disposable, PackageGeneratorView, TextEditor, _, fs, path, ref,
    slice = [].slice;

  path = require('path');

  _ = require('underscore-plus');

  ref = require('atom'), TextEditor = ref.TextEditor, BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Disposable = ref.Disposable;

  fs = require('fs-plus');

  module.exports = PackageGeneratorView = (function() {
    PackageGeneratorView.prototype.previouslyFocusedElement = null;

    PackageGeneratorView.prototype.mode = null;

    function PackageGeneratorView() {
      var blurHandler;
      this.disposables = new CompositeDisposable;
      this.element = document.createElement('div');
      this.element.classList.add('package-generator');
      this.miniEditor = new TextEditor({
        mini: true
      });
      this.element.appendChild(this.miniEditor.element);
      this.error = document.createElement('div');
      this.error.classList.add('error');
      this.element.appendChild(this.error);
      this.message = document.createElement('div');
      this.message.classList.add('message');
      this.element.appendChild(this.message);
      this.disposables.add(atom.commands.add('atom-workspace', {
        'package-generator:generate-package': (function(_this) {
          return function() {
            return _this.attach('package');
          };
        })(this),
        'package-generator:generate-syntax-theme': (function(_this) {
          return function() {
            return _this.attach('theme');
          };
        })(this)
      }));
      blurHandler = (function(_this) {
        return function() {
          return _this.close();
        };
      })(this);
      this.miniEditor.element.addEventListener('blur', blurHandler);
      this.disposables.add(new Disposable((function(_this) {
        return function() {
          return _this.miniEditor.element.removeEventListener('blur', blurHandler);
        };
      })(this)));
      this.disposables.add(atom.commands.add(this.element, {
        'core:confirm': (function(_this) {
          return function() {
            return _this.confirm();
          };
        })(this),
        'core:cancel': (function(_this) {
          return function() {
            return _this.close();
          };
        })(this)
      }));
    }

    PackageGeneratorView.prototype.destroy = function() {
      var ref1;
      if ((ref1 = this.panel) != null) {
        ref1.destroy();
      }
      return this.disposables.dispose();
    };

    PackageGeneratorView.prototype.attach = function(mode) {
      this.mode = mode;
      if (this.panel == null) {
        this.panel = atom.workspace.addModalPanel({
          item: this,
          visible: false
        });
      }
      this.previouslyFocusedElement = document.activeElement;
      this.panel.show();
      this.message.textContent = "Enter " + this.mode + " path";
      if (this.isInPackageMode()) {
        this.setPathText("my-package");
      } else {
        this.setPathText("my-theme-syntax", [0, 8]);
      }
      return this.miniEditor.element.focus();
    };

    PackageGeneratorView.prototype.setPathText = function(placeholderName, rangeToSelect) {
      var endOfDirectoryIndex, packagesDirectory, pathLength;
      if (rangeToSelect == null) {
        rangeToSelect = [0, placeholderName.length];
      }
      packagesDirectory = this.getPackagesDirectory();
      this.miniEditor.setText(path.join(packagesDirectory, placeholderName));
      pathLength = this.miniEditor.getText().length;
      endOfDirectoryIndex = pathLength - placeholderName.length;
      return this.miniEditor.setSelectedBufferRange([[0, endOfDirectoryIndex + rangeToSelect[0]], [0, endOfDirectoryIndex + rangeToSelect[1]]]);
    };

    PackageGeneratorView.prototype.close = function() {
      var ref1;
      if (!this.panel.isVisible()) {
        return;
      }
      this.panel.hide();
      return (ref1 = this.previouslyFocusedElement) != null ? ref1.focus() : void 0;
    };

    PackageGeneratorView.prototype.confirm = function() {
      if (this.validPackagePath()) {
        return this.createPackageFiles((function(_this) {
          return function() {
            var packagePath;
            packagePath = _this.getPackagePath();
            atom.open({
              pathsToOpen: [packagePath]
            });
            return _this.close();
          };
        })(this));
      }
    };

    PackageGeneratorView.prototype.getPackagePath = function() {
      var packageName, packagePath;
      packagePath = fs.normalize(this.miniEditor.getText().trim());
      packageName = _.dasherize(path.basename(packagePath));
      return path.join(path.dirname(packagePath), packageName);
    };

    PackageGeneratorView.prototype.getPackagesDirectory = function() {
      return atom.config.get('core.projectHome') || process.env.ATOM_REPOS_HOME || path.join(fs.getHomeDirectory(), 'github');
    };

    PackageGeneratorView.prototype.validPackagePath = function() {
      if (fs.existsSync(this.getPackagePath())) {
        this.error.textContent = "Path already exists at '" + (this.getPackagePath()) + "'";
        this.error.style.display = 'block';
        return false;
      } else {
        return true;
      }
    };

    PackageGeneratorView.prototype.getInitOptions = function(packagePath) {
      var options;
      options = ["--" + this.mode, packagePath];
      if (this.isInPackageMode()) {
        return slice.call(options).concat(['--syntax'], [atom.config.get('package-generator.packageSyntax')]);
      } else {
        return options;
      }
    };

    PackageGeneratorView.prototype.initPackage = function(packagePath, callback) {
      var command;
      command = ['init'].concat(slice.call(this.getInitOptions(packagePath)));
      return this.runCommand(atom.packages.getApmPath(), command, callback);
    };

    PackageGeneratorView.prototype.linkPackage = function(packagePath, callback) {
      var args;
      args = ['link'];
      if (atom.config.get('package-generator.createInDevMode')) {
        args.push('--dev');
      }
      args.push(packagePath.toString());
      return this.runCommand(atom.packages.getApmPath(), args, callback);
    };

    PackageGeneratorView.prototype.isInPackageMode = function() {
      return this.mode === 'package';
    };

    PackageGeneratorView.prototype.isStoredInDotAtom = function(packagePath) {
      var devPackagesPath, packagesPath;
      packagesPath = path.join(atom.getConfigDirPath(), 'packages', path.sep);
      if (packagePath.indexOf(packagesPath) === 0) {
        return true;
      }
      devPackagesPath = path.join(atom.getConfigDirPath(), 'dev', 'packages', path.sep);
      return packagePath.indexOf(devPackagesPath) === 0;
    };

    PackageGeneratorView.prototype.createPackageFiles = function(callback) {
      var packagePath;
      packagePath = this.getPackagePath();
      if (this.isStoredInDotAtom(packagePath)) {
        return this.initPackage(packagePath, callback);
      } else {
        return this.initPackage(packagePath, (function(_this) {
          return function() {
            return _this.linkPackage(packagePath, callback);
          };
        })(this));
      }
    };

    PackageGeneratorView.prototype.runCommand = function(command, args, exit) {
      return new BufferedProcess({
        command: command,
        args: args,
        exit: exit
      });
    };

    return PackageGeneratorView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
