(function() {
  var CompositeDisposable, Disposable, Emitter, File, MarkdownPreviewView, _, fs, path, ref, renderer;

  path = require('path');

  ref = require('atom'), Emitter = ref.Emitter, Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable, File = ref.File;

  _ = require('underscore-plus');

  fs = require('fs-plus');

  renderer = require('./renderer');

  module.exports = MarkdownPreviewView = (function() {
    MarkdownPreviewView.deserialize = function(params) {
      return new MarkdownPreviewView(params);
    };

    function MarkdownPreviewView(arg) {
      this.editorId = arg.editorId, this.filePath = arg.filePath;
      this.element = document.createElement('div');
      this.element.classList.add('markdown-preview', 'native-key-bindings');
      this.element.tabIndex = -1;
      this.emitter = new Emitter;
      this.loaded = false;
      this.disposables = new CompositeDisposable;
      this.registerScrollCommands();
      if (this.editorId != null) {
        this.resolveEditor(this.editorId);
      } else if (atom.workspace != null) {
        this.subscribeToFilePath(this.filePath);
      } else {
        this.disposables.add(atom.packages.onDidActivateInitialPackages((function(_this) {
          return function() {
            return _this.subscribeToFilePath(_this.filePath);
          };
        })(this)));
      }
    }

    MarkdownPreviewView.prototype.serialize = function() {
      var ref1;
      return {
        deserializer: 'MarkdownPreviewView',
        filePath: (ref1 = this.getPath()) != null ? ref1 : this.filePath,
        editorId: this.editorId
      };
    };

    MarkdownPreviewView.prototype.copy = function() {
      var ref1;
      return new MarkdownPreviewView({
        editorId: this.editorId,
        filePath: (ref1 = this.getPath()) != null ? ref1 : this.filePath
      });
    };

    MarkdownPreviewView.prototype.destroy = function() {
      this.disposables.dispose();
      return this.element.remove();
    };

    MarkdownPreviewView.prototype.registerScrollCommands = function() {
      this.disposables.add(atom.commands.add(this.element, {
        'core:move-up': (function(_this) {
          return function() {
            _this.element.scrollTop -= document.body.offsetHeight / 20;
          };
        })(this),
        'core:move-down': (function(_this) {
          return function() {
            _this.element.scrollTop += document.body.offsetHeight / 20;
          };
        })(this),
        'core:page-up': (function(_this) {
          return function() {
            _this.element.scrollTop -= _this.element.offsetHeight;
          };
        })(this),
        'core:page-down': (function(_this) {
          return function() {
            _this.element.scrollTop += _this.element.offsetHeight;
          };
        })(this),
        'core:move-to-top': (function(_this) {
          return function() {
            _this.element.scrollTop = 0;
          };
        })(this),
        'core:move-to-bottom': (function(_this) {
          return function() {
            _this.element.scrollTop = _this.element.scrollHeight;
          };
        })(this)
      }));
    };

    MarkdownPreviewView.prototype.onDidChangeTitle = function(callback) {
      return this.emitter.on('did-change-title', callback);
    };

    MarkdownPreviewView.prototype.onDidChangeModified = function(callback) {
      return new Disposable;
    };

    MarkdownPreviewView.prototype.onDidChangeMarkdown = function(callback) {
      return this.emitter.on('did-change-markdown', callback);
    };

    MarkdownPreviewView.prototype.subscribeToFilePath = function(filePath) {
      this.file = new File(filePath);
      this.emitter.emit('did-change-title');
      this.disposables.add(this.file.onDidRename((function(_this) {
        return function() {
          return _this.emitter.emit('did-change-title');
        };
      })(this)));
      this.handleEvents();
      return this.renderMarkdown();
    };

    MarkdownPreviewView.prototype.resolveEditor = function(editorId) {
      var resolve;
      resolve = (function(_this) {
        return function() {
          _this.editor = _this.editorForId(editorId);
          if (_this.editor != null) {
            _this.emitter.emit('did-change-title');
            _this.disposables.add(_this.editor.onDidDestroy(function() {
              return _this.subscribeToFilePath(_this.getPath());
            }));
            _this.handleEvents();
            return _this.renderMarkdown();
          } else {
            return _this.subscribeToFilePath(_this.filePath);
          }
        };
      })(this);
      if (atom.workspace != null) {
        return resolve();
      } else {
        return this.disposables.add(atom.packages.onDidActivateInitialPackages(resolve));
      }
    };

    MarkdownPreviewView.prototype.editorForId = function(editorId) {
      var editor, i, len, ref1, ref2;
      ref1 = atom.workspace.getTextEditors();
      for (i = 0, len = ref1.length; i < len; i++) {
        editor = ref1[i];
        if (((ref2 = editor.id) != null ? ref2.toString() : void 0) === editorId.toString()) {
          return editor;
        }
      }
      return null;
    };

    MarkdownPreviewView.prototype.handleEvents = function() {
      var changeHandler;
      this.disposables.add(atom.grammars.onDidAddGrammar((function(_this) {
        return function() {
          return _.debounce((function() {
            return _this.renderMarkdown();
          }), 250);
        };
      })(this)));
      this.disposables.add(atom.grammars.onDidUpdateGrammar(_.debounce(((function(_this) {
        return function() {
          return _this.renderMarkdown();
        };
      })(this)), 250)));
      atom.commands.add(this.element, {
        'core:save-as': (function(_this) {
          return function(event) {
            event.stopPropagation();
            return _this.saveAs();
          };
        })(this),
        'core:copy': (function(_this) {
          return function(event) {
            if (_this.copyToClipboard()) {
              return event.stopPropagation();
            }
          };
        })(this),
        'markdown-preview:zoom-in': (function(_this) {
          return function() {
            var zoomLevel;
            zoomLevel = parseFloat(getComputedStyle(_this.element).zoom);
            return _this.element.style.zoom = zoomLevel + 0.1;
          };
        })(this),
        'markdown-preview:zoom-out': (function(_this) {
          return function() {
            var zoomLevel;
            zoomLevel = parseFloat(getComputedStyle(_this.element).zoom);
            return _this.element.style.zoom = zoomLevel - 0.1;
          };
        })(this),
        'markdown-preview:reset-zoom': (function(_this) {
          return function() {
            return _this.element.style.zoom = 1;
          };
        })(this)
      });
      changeHandler = (function(_this) {
        return function() {
          var pane;
          _this.renderMarkdown();
          pane = atom.workspace.paneForItem(_this);
          if ((pane != null) && pane !== atom.workspace.getActivePane()) {
            return pane.activateItem(_this);
          }
        };
      })(this);
      if (this.file != null) {
        this.disposables.add(this.file.onDidChange(changeHandler));
      } else if (this.editor != null) {
        this.disposables.add(this.editor.getBuffer().onDidStopChanging(function() {
          if (atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
        this.disposables.add(this.editor.onDidChangePath((function(_this) {
          return function() {
            return _this.emitter.emit('did-change-title');
          };
        })(this)));
        this.disposables.add(this.editor.getBuffer().onDidSave(function() {
          if (!atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
        this.disposables.add(this.editor.getBuffer().onDidReload(function() {
          if (!atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
      }
      this.disposables.add(atom.config.onDidChange('markdown-preview.breakOnSingleNewline', changeHandler));
      return this.disposables.add(atom.config.observe('markdown-preview.useGitHubStyle', (function(_this) {
        return function(useGitHubStyle) {
          if (useGitHubStyle) {
            return _this.element.setAttribute('data-use-github-style', '');
          } else {
            return _this.element.removeAttribute('data-use-github-style');
          }
        };
      })(this)));
    };

    MarkdownPreviewView.prototype.renderMarkdown = function() {
      if (!this.loaded) {
        this.showLoading();
      }
      return this.getMarkdownSource().then((function(_this) {
        return function(source) {
          if (source != null) {
            return _this.renderMarkdownText(source);
          }
        };
      })(this))["catch"]((function(_this) {
        return function(reason) {
          return _this.showError({
            message: reason
          });
        };
      })(this));
    };

    MarkdownPreviewView.prototype.getMarkdownSource = function() {
      var ref1;
      if ((ref1 = this.file) != null ? ref1.getPath() : void 0) {
        return this.file.read().then((function(_this) {
          return function(source) {
            if (source === null) {
              return Promise.reject((_this.file.getBaseName()) + " could not be found");
            } else {
              return Promise.resolve(source);
            }
          };
        })(this))["catch"](function(reason) {
          return Promise.reject(reason);
        });
      } else if (this.editor != null) {
        return Promise.resolve(this.editor.getText());
      } else {
        return Promise.reject();
      }
    };

    MarkdownPreviewView.prototype.getHTML = function(callback) {
      return this.getMarkdownSource().then((function(_this) {
        return function(source) {
          if (source == null) {
            return;
          }
          return renderer.toHTML(source, _this.getPath(), _this.getGrammar(), callback);
        };
      })(this));
    };

    MarkdownPreviewView.prototype.renderMarkdownText = function(text) {
      return renderer.toDOMFragment(text, this.getPath(), this.getGrammar(), (function(_this) {
        return function(error, domFragment) {
          if (error) {
            return _this.showError(error);
          } else {
            _this.loading = false;
            _this.loaded = true;
            _this.element.textContent = '';
            _this.element.appendChild(domFragment);
            return _this.emitter.emit('did-change-markdown');
          }
        };
      })(this));
    };

    MarkdownPreviewView.prototype.getTitle = function() {
      if ((this.file != null) && (this.getPath() != null)) {
        return (path.basename(this.getPath())) + " Preview";
      } else if (this.editor != null) {
        return (this.editor.getTitle()) + " Preview";
      } else {
        return "Markdown Preview";
      }
    };

    MarkdownPreviewView.prototype.getIconName = function() {
      return "markdown";
    };

    MarkdownPreviewView.prototype.getURI = function() {
      if (this.file != null) {
        return "markdown-preview://" + (this.getPath());
      } else {
        return "markdown-preview://editor/" + this.editorId;
      }
    };

    MarkdownPreviewView.prototype.getPath = function() {
      if (this.file != null) {
        return this.file.getPath();
      } else if (this.editor != null) {
        return this.editor.getPath();
      }
    };

    MarkdownPreviewView.prototype.getGrammar = function() {
      var ref1;
      return (ref1 = this.editor) != null ? ref1.getGrammar() : void 0;
    };

    MarkdownPreviewView.prototype.getDocumentStyleSheets = function() {
      return document.styleSheets;
    };

    MarkdownPreviewView.prototype.getTextEditorStyles = function() {
      var textEditorStyles;
      textEditorStyles = document.createElement("atom-styles");
      textEditorStyles.initialize(atom.styles);
      textEditorStyles.setAttribute("context", "atom-text-editor");
      document.body.appendChild(textEditorStyles);
      return Array.prototype.slice.apply(textEditorStyles.childNodes).map(function(styleElement) {
        return styleElement.innerText;
      });
    };

    MarkdownPreviewView.prototype.getMarkdownPreviewCSS = function() {
      var cssUrlRegExp, i, j, len, len1, markdownPreviewRules, ref1, ref2, ref3, rule, ruleRegExp, stylesheet;
      markdownPreviewRules = [];
      ruleRegExp = /\.markdown-preview/;
      cssUrlRegExp = /url\(atom:\/\/markdown-preview\/assets\/(.*)\)/;
      ref1 = this.getDocumentStyleSheets();
      for (i = 0, len = ref1.length; i < len; i++) {
        stylesheet = ref1[i];
        if (stylesheet.rules != null) {
          ref2 = stylesheet.rules;
          for (j = 0, len1 = ref2.length; j < len1; j++) {
            rule = ref2[j];
            if (((ref3 = rule.selectorText) != null ? ref3.match(ruleRegExp) : void 0) != null) {
              markdownPreviewRules.push(rule.cssText);
            }
          }
        }
      }
      return markdownPreviewRules.concat(this.getTextEditorStyles()).join('\n').replace(/atom-text-editor/g, 'pre.editor-colors').replace(/:host/g, '.host').replace(cssUrlRegExp, function(match, assetsName, offset, string) {
        var assetPath, base64Data, originalData;
        assetPath = path.join(__dirname, '../assets', assetsName);
        originalData = fs.readFileSync(assetPath, 'binary');
        base64Data = new Buffer(originalData, 'binary').toString('base64');
        return "url('data:image/jpeg;base64," + base64Data + "')";
      });
    };

    MarkdownPreviewView.prototype.showError = function(result) {
      var failureMessage, h2, h3;
      this.element.textContent = '';
      h2 = document.createElement('h2');
      h2.textContent = 'Previewing Markdown Failed';
      this.element.appendChild(h2);
      if (failureMessage = result != null ? result.message : void 0) {
        h3 = document.createElement('h3');
        h3.textContent = failureMessage;
        return this.element.appendChild(h3);
      }
    };

    MarkdownPreviewView.prototype.showLoading = function() {
      var div;
      this.loading = true;
      this.element.textContent = '';
      div = document.createElement('div');
      div.classList.add('markdown-spinner');
      div.textContent = 'Loading Markdown\u2026';
      return this.element.appendChild(div);
    };

    MarkdownPreviewView.prototype.copyToClipboard = function() {
      var selectedNode, selectedText, selection;
      if (this.loading) {
        return false;
      }
      selection = window.getSelection();
      selectedText = selection.toString();
      selectedNode = selection.baseNode;
      if (selectedText && (selectedNode != null) && (this.element === selectedNode || this.element.contains(selectedNode))) {
        return false;
      }
      this.getHTML(function(error, html) {
        if (error != null) {
          return console.warn('Copying Markdown as HTML failed', error);
        } else {
          return atom.clipboard.write(html);
        }
      });
      return true;
    };

    MarkdownPreviewView.prototype.saveAs = function() {
      var filePath, htmlFilePath, projectPath, title;
      if (this.loading) {
        return;
      }
      filePath = this.getPath();
      title = 'Markdown to HTML';
      if (filePath) {
        title = path.parse(filePath).name;
        filePath += '.html';
      } else {
        filePath = 'untitled.md.html';
        if (projectPath = atom.project.getPaths()[0]) {
          filePath = path.join(projectPath, filePath);
        }
      }
      if (htmlFilePath = atom.showSaveDialogSync(filePath)) {
        return this.getHTML((function(_this) {
          return function(error, htmlBody) {
            var html;
            if (error != null) {
              return console.warn('Saving Markdown as HTML failed', error);
            } else {
              html = ("<!DOCTYPE html>\n<html>\n  <head>\n      <meta charset=\"utf-8\" />\n      <title>" + title + "</title>\n      <style>" + (_this.getMarkdownPreviewCSS()) + "</style>\n  </head>\n  <body class='markdown-preview' data-use-github-style>" + htmlBody + "</body>\n</html>") + "\n";
              fs.writeFileSync(htmlFilePath, html);
              return atom.workspace.open(htmlFilePath);
            }
          };
        })(this));
      }
    };

    return MarkdownPreviewView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
