'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _listSelection = require('./list-selection');

var _listSelection2 = _interopRequireDefault(_listSelection);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const COPY = {};

let FilePatchSelection = class FilePatchSelection {
  constructor(hunks) {
    if (hunks._copy !== COPY) {
      // Initialize a new selection
      this.mode = 'hunk';

      this.hunksByLine = new Map();
      const lines = [];
      for (const hunk of hunks) {
        for (const line of hunk.lines) {
          lines.push(line);
          this.hunksByLine.set(line, hunk);
        }
      }

      this.hunksSelection = new _listSelection2.default({ items: hunks });
      this.linesSelection = new _listSelection2.default({ items: lines, isItemSelectable: line => line.isChanged() });
      this.resolveNextUpdatePromise = () => {};
    } else {
      // Copy from options. *Only* reachable from the copy() method because no other module has visibility to
      // the COPY object without shenanigans.
      const options = hunks;

      this.mode = options.mode;
      this.hunksSelection = options.hunksSelection;
      this.linesSelection = options.linesSelection;
      this.resolveNextUpdatePromise = options.resolveNextUpdatePromise;
      this.hunksByLine = options.hunksByLine;
    }
  }

  copy() {
    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

    const mode = options.mode || this.mode;
    const hunksSelection = options.hunksSelection || this.hunksSelection.copy();
    const linesSelection = options.linesSelection || this.linesSelection.copy();

    let hunksByLine = null;
    if (options.hunks) {
      // Update hunks
      const oldHunks = this.hunksSelection.getItems();
      const newHunks = options.hunks;

      let wasChanged = false;
      if (newHunks.length !== oldHunks.length) {
        wasChanged = true;
      } else {
        for (let i = 0; i < oldHunks.length; i++) {
          if (oldHunks[i] !== newHunks[i]) {
            wasChanged = true;
            break;
          }
        }
      }

      // Update hunks, preserving selection index
      hunksSelection.setItems(newHunks);

      const oldLines = this.linesSelection.getItems();
      const newLines = [];

      hunksByLine = new Map();
      for (const hunk of newHunks) {
        for (const line of hunk.lines) {
          newLines.push(line);
          hunksByLine.set(line, hunk);
        }
      }

      // Update lines, preserving selection index in *changed* lines
      let newSelectedLine;
      if (oldLines.length > 0 && newLines.length > 0) {
        const oldSelectionStartIndex = this.linesSelection.getMostRecentSelectionStartIndex();
        let changedLineCount = 0;
        for (let i = 0; i < oldSelectionStartIndex; i++) {
          if (oldLines[i].isChanged()) {
            changedLineCount++;
          }
        }

        for (let i = 0; i < newLines.length; i++) {
          const line = newLines[i];
          if (line.isChanged()) {
            newSelectedLine = line;
            if (changedLineCount === 0) {
              break;
            }
            changedLineCount--;
          }
        }
      }

      linesSelection.setItems(newLines);
      if (newSelectedLine) {
        linesSelection.selectItem(newSelectedLine);
      }
      if (wasChanged) {
        this.resolveNextUpdatePromise();
      }
    } else {
      // Hunks are unchanged. Don't recompute hunksByLine.
      hunksByLine = this.hunksByLine;
    }

    return new FilePatchSelection({
      _copy: COPY,
      mode,
      hunksSelection,
      linesSelection,
      hunksByLine,
      resolveNextUpdatePromise: options.resolveNextUpdatePromise || this.resolveNextUpdatePromise
    });
  }

  toggleMode() {
    if (this.mode === 'hunk') {
      const firstLineOfSelectedHunk = this.getHeadHunk().lines[0];
      const selection = this.selectLine(firstLineOfSelectedHunk);
      if (!firstLineOfSelectedHunk.isChanged()) {
        return selection.selectNextLine();
      } else {
        return selection;
      }
    } else {
      const selectedLine = this.getHeadLine();
      const hunkContainingSelectedLine = this.hunksByLine.get(selectedLine);
      return this.selectHunk(hunkContainingSelectedLine);
    }
  }

  getMode() {
    return this.mode;
  }

  selectNext() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    if (this.mode === 'hunk') {
      return this.selectNextHunk(preserveTail);
    } else {
      return this.selectNextLine(preserveTail);
    }
  }

  selectPrevious() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    if (this.mode === 'hunk') {
      return this.selectPreviousHunk(preserveTail);
    } else {
      return this.selectPreviousLine(preserveTail);
    }
  }

  selectAll() {
    if (this.mode === 'hunk') {
      return this.selectAllHunks();
    } else {
      return this.selectAllLines();
    }
  }

  selectFirst(preserveTail) {
    if (this.mode === 'hunk') {
      return this.selectFirstHunk(preserveTail);
    } else {
      return this.selectFirstLine(preserveTail);
    }
  }

  selectLast(preserveTail) {
    if (this.mode === 'hunk') {
      return this.selectLastHunk(preserveTail);
    } else {
      return this.selectLastLine(preserveTail);
    }
  }

  selectHunk(hunk) {
    let preserveTail = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectItem(hunk, preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  addOrSubtractHunkSelection(hunk) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.addOrSubtractSelection(hunk);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  selectAllHunks() {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectAllItems();

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  selectFirstHunk(preserveTail) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectFirstItem(preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  selectLastHunk(preserveTail) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectLastItem(preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  jumpToNextHunk() {
    const next = this.selectNextHunk();
    return next.getMode() !== this.mode ? next.toggleMode() : next;
  }

  jumpToPreviousHunk() {
    const next = this.selectPreviousHunk();
    return next.getMode() !== this.mode ? next.toggleMode() : next;
  }

  selectNextHunk(preserveTail) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectNextItem(preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  selectPreviousHunk(preserveTail) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectPreviousItem(preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  getSelectedHunks() {
    if (this.mode === 'line') {
      const selectedHunks = new Set();
      const selectedLines = this.getSelectedLines();
      selectedLines.forEach(line => selectedHunks.add(this.hunksByLine.get(line)));
      return selectedHunks;
    } else {
      return this.hunksSelection.getSelectedItems();
    }
  }

  getHeadHunk() {
    return this.mode === 'hunk' ? this.hunksSelection.getHeadItem() : null;
  }

  selectLine(line) {
    let preserveTail = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

    const linesSelection = this.linesSelection.copy();
    linesSelection.selectItem(line, preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  addOrSubtractLineSelection(line) {
    const linesSelection = this.linesSelection.copy();
    linesSelection.addOrSubtractSelection(line);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectAllLines(preserveTail) {
    const linesSelection = this.linesSelection.copy();
    linesSelection.selectAllItems(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectFirstLine(preserveTail) {
    const linesSelection = this.linesSelection.copy();
    linesSelection.selectFirstItem(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectLastLine(preserveTail) {
    const linesSelection = this.linesSelection.copy();
    linesSelection.selectLastItem(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectNextLine() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    const linesSelection = this.linesSelection.copy();
    linesSelection.selectNextItem(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectPreviousLine() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    const linesSelection = this.linesSelection.copy();
    linesSelection.selectPreviousItem(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  getSelectedLines() {
    if (this.mode === 'hunk') {
      const selectedLines = new Set();
      this.getSelectedHunks().forEach(hunk => {
        for (const line of hunk.lines) {
          if (line.isChanged()) {
            selectedLines.add(line);
          }
        }
      });
      return selectedLines;
    } else {
      return this.linesSelection.getSelectedItems();
    }
  }

  getHeadLine() {
    return this.mode === 'line' ? this.linesSelection.getHeadItem() : null;
  }

  updateHunks(newHunks) {
    return this.copy({ hunks: newHunks });
  }

  coalesce() {
    const hunksSelection = this.hunksSelection.copy();
    const linesSelection = this.linesSelection.copy();

    hunksSelection.coalesce();
    linesSelection.coalesce();

    return this.copy({ hunksSelection, linesSelection });
  }

  getNextUpdatePromise() {
    return new Promise((resolve, reject) => {
      this.resolveNextUpdatePromise = resolve;
    });
  }

  getLineSelectionTailIndex() {
    return this.linesSelection.getTailIndex();
  }

  goToDiffLine(lineNumber) {
    const lines = this.linesSelection.getItems();

    let closestLine;
    let closestLineDistance = Infinity;

    for (let i = 0; i < lines.length; i++) {
      const line = lines[i];
      if (!this.linesSelection.isItemSelectable(line)) {
        continue;
      }
      if (line.newLineNumber === lineNumber) {
        return this.selectLine(line);
      } else {
        const newDistance = Math.abs(line.newLineNumber - lineNumber);
        if (newDistance < closestLineDistance) {
          closestLineDistance = newDistance;
          closestLine = line;
        } else {
          return this.selectLine(closestLine);
        }
      }
    }

    return this.selectLine(closestLine);
  }
};
exports.default = FilePatchSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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