(function() {
  var COMPLETIONS, attributePattern, firstCharsEqual, tagPattern;

  COMPLETIONS = require('../completions.json');

  attributePattern = /\s+([a-zA-Z][-a-zA-Z]*)\s*=\s*$/;

  tagPattern = /<([a-zA-Z][-a-zA-Z]*)(?:\s|$)/;

  module.exports = {
    selector: '.text.html',
    disableForSelector: '.text.html .comment',
    filterSuggestions: true,
    completions: COMPLETIONS,
    getSuggestions: function(request) {
      if (this.isAttributeValueStart(request)) {
        return this.getAttributeValueCompletions(request);
      } else if (this.isAttributeStart(request)) {
        return this.getAttributeNameCompletions(request);
      } else if (this.isTagStart(request)) {
        return this.getTagNameCompletions(request);
      } else {
        return [];
      }
    },
    onDidInsertSuggestion: function(arg) {
      var editor, suggestion;
      editor = arg.editor, suggestion = arg.suggestion;
      if (suggestion.type === 'attribute') {
        return setTimeout(this.triggerAutocomplete.bind(this, editor), 1);
      }
    },
    triggerAutocomplete: function(editor) {
      return atom.commands.dispatch(atom.views.getView(editor), 'autocomplete-plus:activate', {
        activatedManually: false
      });
    },
    isTagStart: function(arg) {
      var bufferPosition, editor, prefix, scopeDescriptor, scopes;
      prefix = arg.prefix, scopeDescriptor = arg.scopeDescriptor, bufferPosition = arg.bufferPosition, editor = arg.editor;
      if (prefix.trim() && prefix.indexOf('<') === -1) {
        return this.hasTagScope(scopeDescriptor.getScopesArray());
      }
      prefix = editor.getTextInRange([[bufferPosition.row, bufferPosition.column - 1], bufferPosition]);
      scopes = scopeDescriptor.getScopesArray();
      return prefix === '<' && scopes[0] === 'text.html.basic' && scopes.length === 1;
    },
    isAttributeStart: function(arg) {
      var bufferPosition, editor, prefix, previousBufferPosition, previousScopes, previousScopesArray, scopeDescriptor, scopes;
      prefix = arg.prefix, scopeDescriptor = arg.scopeDescriptor, bufferPosition = arg.bufferPosition, editor = arg.editor;
      scopes = scopeDescriptor.getScopesArray();
      if (!this.getPreviousAttribute(editor, bufferPosition) && prefix && !prefix.trim()) {
        return this.hasTagScope(scopes);
      }
      previousBufferPosition = [bufferPosition.row, Math.max(0, bufferPosition.column - 1)];
      previousScopes = editor.scopeDescriptorForBufferPosition(previousBufferPosition);
      previousScopesArray = previousScopes.getScopesArray();
      if (previousScopesArray.indexOf('entity.other.attribute-name.html') !== -1) {
        return true;
      }
      if (!this.hasTagScope(scopes)) {
        return false;
      }
      return scopes.indexOf('punctuation.definition.tag.end.html') !== -1 && previousScopesArray.indexOf('punctuation.definition.tag.end.html') === -1;
    },
    isAttributeValueStart: function(arg) {
      var bufferPosition, editor, previousBufferPosition, previousScopes, previousScopesArray, scopeDescriptor, scopes;
      scopeDescriptor = arg.scopeDescriptor, bufferPosition = arg.bufferPosition, editor = arg.editor;
      scopes = scopeDescriptor.getScopesArray();
      previousBufferPosition = [bufferPosition.row, Math.max(0, bufferPosition.column - 1)];
      previousScopes = editor.scopeDescriptorForBufferPosition(previousBufferPosition);
      previousScopesArray = previousScopes.getScopesArray();
      return this.hasStringScope(scopes) && this.hasStringScope(previousScopesArray) && previousScopesArray.indexOf('punctuation.definition.string.end.html') === -1 && this.hasTagScope(scopes) && (this.getPreviousAttribute(editor, bufferPosition) != null);
    },
    hasTagScope: function(scopes) {
      return scopes.indexOf('meta.tag.any.html') !== -1 || scopes.indexOf('meta.tag.other.html') !== -1 || scopes.indexOf('meta.tag.block.any.html') !== -1 || scopes.indexOf('meta.tag.inline.any.html') !== -1 || scopes.indexOf('meta.tag.structure.any.html') !== -1;
    },
    hasStringScope: function(scopes) {
      return scopes.indexOf('string.quoted.double.html') !== -1 || scopes.indexOf('string.quoted.single.html') !== -1;
    },
    getTagNameCompletions: function(arg) {
      var bufferPosition, completions, editor, ignorePrefix, options, prefix, ref, tag;
      prefix = arg.prefix, editor = arg.editor, bufferPosition = arg.bufferPosition;
      ignorePrefix = editor.getTextInRange([[bufferPosition.row, bufferPosition.column - 1], bufferPosition]) === '<';
      completions = [];
      ref = this.completions.tags;
      for (tag in ref) {
        options = ref[tag];
        if (ignorePrefix || firstCharsEqual(tag, prefix)) {
          completions.push(this.buildTagCompletion(tag, options));
        }
      }
      return completions;
    },
    buildTagCompletion: function(tag, arg) {
      var description;
      description = arg.description;
      return {
        text: tag,
        type: 'tag',
        description: description != null ? description : "HTML <" + tag + "> tag",
        descriptionMoreURL: description ? this.getTagDocsURL(tag) : null
      };
    },
    getAttributeNameCompletions: function(arg) {
      var attribute, bufferPosition, completions, editor, i, len, options, prefix, ref, tag, tagAttributes;
      prefix = arg.prefix, editor = arg.editor, bufferPosition = arg.bufferPosition;
      completions = [];
      tag = this.getPreviousTag(editor, bufferPosition);
      tagAttributes = this.getTagAttributes(tag);
      for (i = 0, len = tagAttributes.length; i < len; i++) {
        attribute = tagAttributes[i];
        if (!prefix.trim() || firstCharsEqual(attribute, prefix)) {
          completions.push(this.buildLocalAttributeCompletion(attribute, tag, this.completions.attributes[attribute]));
        }
      }
      ref = this.completions.attributes;
      for (attribute in ref) {
        options = ref[attribute];
        if (!prefix.trim() || firstCharsEqual(attribute, prefix)) {
          if (options.global) {
            completions.push(this.buildGlobalAttributeCompletion(attribute, options));
          }
        }
      }
      return completions;
    },
    buildLocalAttributeCompletion: function(attribute, tag, options) {
      return {
        snippet: (options != null ? options.type : void 0) === 'flag' ? attribute : attribute + "=\"$1\"$0",
        displayText: attribute,
        type: 'attribute',
        rightLabel: "<" + tag + ">",
        description: attribute + " attribute local to <" + tag + "> tags",
        descriptionMoreURL: this.getLocalAttributeDocsURL(attribute, tag)
      };
    },
    buildGlobalAttributeCompletion: function(attribute, arg) {
      var description, type;
      description = arg.description, type = arg.type;
      return {
        snippet: type === 'flag' ? attribute : attribute + "=\"$1\"$0",
        displayText: attribute,
        type: 'attribute',
        description: description != null ? description : "Global " + attribute + " attribute",
        descriptionMoreURL: description ? this.getGlobalAttributeDocsURL(attribute) : null
      };
    },
    getAttributeValueCompletions: function(arg) {
      var attribute, bufferPosition, completions, editor, i, len, prefix, ref, tag, value, values;
      prefix = arg.prefix, editor = arg.editor, bufferPosition = arg.bufferPosition;
      completions = [];
      tag = this.getPreviousTag(editor, bufferPosition);
      attribute = this.getPreviousAttribute(editor, bufferPosition);
      values = this.getAttributeValues(tag, attribute);
      for (i = 0, len = values.length; i < len; i++) {
        value = values[i];
        if (!prefix || firstCharsEqual(value, prefix)) {
          completions.push(this.buildAttributeValueCompletion(tag, attribute, value));
        }
      }
      if (completions.length === 0 && ((ref = this.completions.attributes[attribute]) != null ? ref.type : void 0) === 'boolean') {
        completions.push(this.buildAttributeValueCompletion(tag, attribute, 'true'));
        completions.push(this.buildAttributeValueCompletion(tag, attribute, 'false'));
      }
      return completions;
    },
    buildAttributeValueCompletion: function(tag, attribute, value) {
      if (this.completions.attributes[attribute].global) {
        return {
          text: value,
          type: 'value',
          description: value + " value for global " + attribute + " attribute",
          descriptionMoreURL: this.getGlobalAttributeDocsURL(attribute)
        };
      } else {
        return {
          text: value,
          type: 'value',
          rightLabel: "<" + tag + ">",
          description: value + " value for " + attribute + " attribute local to <" + tag + ">",
          descriptionMoreURL: this.getLocalAttributeDocsURL(attribute, tag)
        };
      }
    },
    getPreviousTag: function(editor, bufferPosition) {
      var ref, row, tag;
      row = bufferPosition.row;
      while (row >= 0) {
        tag = (ref = tagPattern.exec(editor.lineTextForBufferRow(row))) != null ? ref[1] : void 0;
        if (tag) {
          return tag;
        }
        row--;
      }
    },
    getPreviousAttribute: function(editor, bufferPosition) {
      var quoteIndex, ref, scopes, scopesArray;
      quoteIndex = bufferPosition.column - 1;
      while (quoteIndex) {
        scopes = editor.scopeDescriptorForBufferPosition([bufferPosition.row, quoteIndex]);
        scopesArray = scopes.getScopesArray();
        if (!this.hasStringScope(scopesArray) || scopesArray.indexOf('punctuation.definition.string.begin.html') !== -1) {
          break;
        }
        quoteIndex--;
      }
      return (ref = attributePattern.exec(editor.getTextInRange([[bufferPosition.row, 0], [bufferPosition.row, quoteIndex]]))) != null ? ref[1] : void 0;
    },
    getAttributeValues: function(tag, attribute) {
      var ref, ref1, ref2, ref3;
      return (ref = (ref1 = (ref2 = this.completions.attributes[attribute]) != null ? ref2.attribOption : void 0) != null ? ref1 : (ref3 = this.completions.attributes[tag + "/" + attribute]) != null ? ref3.attribOption : void 0) != null ? ref : [];
    },
    getTagAttributes: function(tag) {
      var ref, ref1;
      return (ref = (ref1 = this.completions.tags[tag]) != null ? ref1.attributes : void 0) != null ? ref : [];
    },
    getTagDocsURL: function(tag) {
      return "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/" + tag;
    },
    getLocalAttributeDocsURL: function(attribute, tag) {
      return (this.getTagDocsURL(tag)) + "#attr-" + attribute;
    },
    getGlobalAttributeDocsURL: function(attribute) {
      return "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/" + attribute;
    }
  };

  firstCharsEqual = function(str1, str2) {
    return str1[0].toLowerCase() === str2[0].toLowerCase();
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
