/*$
mlk in apdtool
Copyright (c) 2020 Azel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
$*/

/*****************************************
 * mFile を使ったファイル操作
 *****************************************/

#include "mlk.h"
#include "mlk_file.h"
#include "mlk_string.h"


//==============================
// 単体関数
//==============================


/**@ ファイルの内容を、確保したメモリ上にすべて読み込む
 *
 * @g:単体関数
 *
 * @d:ファイルが空の場合、デフォルトでエラーになる。\
 * 空ファイルを許可して空だった場合は、常に終端に 0 (1byte) が追加される。\
 * \
 * また、ファイルサイズが INT32_MAX (2GB) を超える場合もエラーになる。
 *
 * @p:filename ファイル名 (UTF-8)
 * @p:psize NULL 以外で、確保したバッファのサイズが入る。\
 * 0 を追加した場合、それも含む。
 * @p:flags フラグ。\
 * \
 * @tbl>
 * |||MREADFILEFULL_APPEND_0||終端に 1byte の 0 を追加する||
 * |||MREADFILEFULL_ACCEPT_EMPTY||空ファイルをエラーにせず受け入れる||
 * @tbl<
 * 
 * @r:確保されたバッファ (NULL で失敗) */

void *mReadFileFull_alloc(const char *filename,int32_t *psize,uint32_t flags)
{
	mFile file;
	mlkfoff fsize;
	uint8_t *buf = NULL,fappend = FALSE;
	int32_t bufsize;

	if(!mFileOpen_read(&file, filename)) return NULL;

	//ファイルサイズ

	fsize = mFileGetSize(file);
	
	if((fsize == 0 && !(flags & MREADFILEFULL_ACCEPT_EMPTY))
		|| fsize > INT32_MAX)
		goto ERR;

	//バッファサイズ

	bufsize = fsize;

	if(fsize == 0 || (flags & MREADFILEFULL_APPEND_0))
	{
		bufsize++;
		fappend = TRUE;
	}

	//確保 & 読み込み

	buf = (uint8_t *)mMalloc(bufsize);
	if(!buf) goto ERR;

	if(!mFileRead_full(file, buf, fsize)) goto ERR;

	//0 追加

	if(fappend)
		buf[fsize] = 0;

	mFileClose(file);

	if(psize) *psize = bufsize;

	return buf;

ERR:
	mFree(buf);
	mFileClose(file);
	return NULL;
}

/**@ ファイルの先頭から指定サイズを読み込み
 *
 * @p:buf 読み込み先
 * @p:size 読み込むサイズ
 * @r:TRUE で、エラーがなく指定サイズ分を読み込めた */

mlkbool mReadFileHead(const char *filename,void *buf,int32_t size)
{
	mFile file;
	mlkbool ret;

	if(!mFileOpen_read(&file, filename))
		return FALSE;

	ret = mFileRead_full(file, buf, size);

	mFileClose(file);

	return ret;
}

/**@ ファイルをコピー
 *
 * @d:ファイルを読み込みつつ書き込んでいく。 */

mlkbool mCopyFile(const char *srcpath,const char *dstpath)
{
	mFile fsrc,fdst;
	uint8_t *buf;
	int32_t size;
	mlkbool ret = FALSE;

	//バッファ確保

	buf = (uint8_t *)mMalloc(16 * 1024);
	if(!buf) return FALSE;

	//開く

	fsrc = fdst = MFILE_NONE;

	if(!mFileOpen_read(&fsrc, srcpath)
		|| !mFileOpen_write(&fdst, dstpath, -1))
		goto ERR;

	//コピー

	while(1)
	{
		size = mFileRead(fsrc, buf, 16 * 1024);

		if(size > 0)
		{
			if(!mFileWrite_full(fdst, buf, size))
				break;
		}
		else
		{
			//エラーまたは終端
			if(size == 0) ret = TRUE;
			break;
		}
	}

ERR:
	mFileClose(fdst);
	mFileClose(fsrc);
	mFree(buf);

	return ret;
}


//==============================
// テキスト読み込み
//==============================


struct _mFileText
{
	char *buf,
		*cur,		//現在位置
		*next;		//次の行の位置 (NULL で終了)
	int32_t size;	//バッファサイズ
};


/**@ ファイルから読み込んで作成
 *
 * @g:テキスト読み込み
 *
 * @d:ファイル内容はすべてメモリに読み込まれる。 */

mFileText *mFileText_readFile(const char *filename)
{
	mFileText *p;
	char *buf;
	int32_t size;

	//読み込み

	buf = (char *)mReadFileFull_alloc(filename, &size,
			MREADFILEFULL_ACCEPT_EMPTY | MREADFILEFULL_APPEND_0);

	if(!buf) return NULL;

	//

	p = (mFileText *)mMalloc0(sizeof(mFileText));

	if(!p)
		mFree(buf);
	else
	{
		p->buf = p->cur = p->next = buf;
		p->size = size;
	}

	return p;
}

/**@ 終了 */

void mFileText_end(mFileText *p)
{
	if(p)
	{
		mFree(p->buf);
		mFree(p);
	}
}

/**@ 次の一行を取得
 *
 * @d:改行文字を 0 に変換して、行頭のポインタ (読み込んだバッファ内) を返す。\
 * 戻り値のポインタからヌル文字までの範囲は、内容を変更しても構わない。
 * 
 * @r:NULL で終了 */

char *mFileText_nextLine(mFileText *p)
{
	if(!p->next)
		return NULL;
	else
	{
		p->cur = p->next;
		p->next = mStringGetNextLine(p->cur, TRUE);

		return p->cur;
	}
}

/**@ 次の一行を取得 (空行はスキップ)
 *
 * @d:すべての空行をスキップして、文字がある次の行を取得する。 */

char *mFileText_nextLine_skip(mFileText *p)
{
	while(p->next)
	{
		p->cur = p->next;
		p->next = mStringGetNextLine(p->cur, TRUE);

		//行頭に文字がある
		if(*(p->cur)) break;
	}

	return (p->next)? p->cur: NULL;
}
