/*$
apdtool
Copyright (c) 2020 Azel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
$*/

/*****************************************
 * APD ver3 読み込み
 *****************************************/

#include <stdio.h>

#include "mlk.h"
#include "mlk_zlib.h"
#include "mlk_stdio.h"
#include "mlk_imagebuf.h"
#include "mlk_util.h"

#include "def.h"


//------------------

typedef struct
{
	FILE *fp;
	mZlib *zdec;
	mImageBuf2 *img;
	uint8_t *tilebuf;
	LayerData *layer;

	off_t thunk_fpos;
	uint32_t thunk_size;
	
	int w,h;
	uint32_t fcoltype;
}APDv3Data;

//------------------

enum
{
	_COLTYPE_F_8BIT = 1<<0,
	_COLTYPE_F_1BIT = 1<<1,
	_COLTYPE_F_F15BIT = 1<<2
};

static uint32_t g_blendid[] = {
	MLK_MAKE32_4('n','o','r','m'), MLK_MAKE32_4('m','u','l',' '), MLK_MAKE32_4('a','d','d',' '), MLK_MAKE32_4('s','u','b',' '),
	MLK_MAKE32_4('s','c','r','n'), MLK_MAKE32_4('o','v','r','l'), MLK_MAKE32_4('h','d','l','g'), MLK_MAKE32_4('s','t','l','g'),
	MLK_MAKE32_4('d','o','d','g'), MLK_MAKE32_4('b','u','r','n'), MLK_MAKE32_4('l','b','u','n'), MLK_MAKE32_4('v','i','v','l'),
	MLK_MAKE32_4('l','i','n','l'), MLK_MAKE32_4('p','i','n','l'), MLK_MAKE32_4('l','i','g','t'), MLK_MAKE32_4('d','a','r','k'),
	MLK_MAKE32_4('d','i','f','f'), MLK_MAKE32_4('l','m','a','d'), MLK_MAKE32_4('l','m','d','g'), 0
};

static const uint32_t g_tilesize_tbl[5] = {64*64*2*4, 64*64*2*2, 64*64*2, 64*64/8, 64*64*4};

static const func_set_tileimage g_funcs_settile[5] = {
	set_tileimage_rgba16, set_tileimage_graya16, set_tileimage_a16, set_tileimage_a1, set_tileimage_rgba8
};

static const func_blendtile g_funcs_blendtile[5] = {
	blendtile_rgba16, blendtile_graya16, blendtile_a16, blendtile_a1, blendtile_rgba8
};

//------------------


/** レイヤの色とテクスチャアイテム取得 */

static void _getlayer_info(LayerData *pl,int *pcol,TexItem **pitem)
{
	TexItem *item;
	int col;

	item = pl->texitem;
	col = pl->col;

	if((g_work.flags & FLAGS_LAYERTEX_TO_COL)
		&& pl->texpath
		&& (pl->coltype == 2 || pl->coltype == 3))
	{
		col = TexturePathList_getColor(&g_work.listTexPath, pl->texpath);

		if(col == -1)
			col = pl->col;
		else
			item = NULL;
	}

	*pcol = col;
	*pitem = item;
}

/** レイヤイメージ読み込み (レイヤ抽出用) */

static mlkerr _read_layer_image(LayerData *pl,void *ptr)
{
	APDv3Data *p = (APDv3Data *)ptr;
	TexItem *texitem;
	uint8_t *tilebuf,tilepos[4];
	uint32_t tilesize,i;
	int tx,ty,col;
	func_set_tileimage func;
	mlkerr ret;

	fseeko(p->fp, pl->fpos, SEEK_SET);

	image_fill_zero(p->img);

	tilebuf = p->tilebuf;
	tilesize = g_tilesize_tbl[pl->coltype];
	func = g_funcs_settile[pl->coltype];

	_getlayer_info(pl, &col, &texitem);

	//

	mZlibDecReset(p->zdec);
	mZlibDecSetSize(p->zdec, pl->imgparam2);

	for(i = pl->imgparam; i; i--)
	{
		//タイル位置

		ret = mZlibDecRead(p->zdec, tilepos, 4);
		if(ret) return ret;

		tx = mGetBufBE16(tilepos);
		ty = mGetBufBE16(tilepos + 2);

		//タイルデータ読み込み

		ret = mZlibDecRead(p->zdec, tilebuf, tilesize);
		if(ret) return ret;

		//イメージセット

		(func)(p->img, tilebuf,
			pl->offx + tx * 64, pl->offy + ty * 64, col, texitem);
	}

	return MLKERR_OK;
}

/** レイヤイメージ合成 */

static mlkerr _blend_layer(mImageBuf2 *img,LayerData *pl,int opacity,void *ptr)
{
	APDv3Data *p = (APDv3Data *)ptr;
	TexItem *texitem;
	uint8_t *tilebuf,tilepos[4];
	uint32_t tilesize,i;
	int tx,ty,col;
	func_blendtile func;
	mlkerr ret;

	fseeko(p->fp, pl->fpos, SEEK_SET);

	tilebuf = p->tilebuf;
	tilesize = g_tilesize_tbl[pl->coltype];
	func = g_funcs_blendtile[pl->coltype];

	_getlayer_info(pl, &col, &texitem);

	//

	mZlibDecReset(p->zdec);
	mZlibDecSetSize(p->zdec, pl->imgparam2);

	for(i = pl->imgparam; i; i--)
	{
		//タイル位置

		ret = mZlibDecRead(p->zdec, tilepos, 4);
		if(ret) return ret;

		tx = mGetBufBE16(tilepos);
		ty = mGetBufBE16(tilepos + 2);

		//タイルデータ読み込み

		ret = mZlibDecRead(p->zdec, tilebuf, tilesize);
		if(ret) return ret;

		//合成

		(func)(img, tilebuf,
			pl->offx + tx * 64, pl->offy + ty * 64,
			pl->blendmode, opacity, col, texitem);
	}

	return MLKERR_OK;
}

/** 指定チャンクを探す
 *
 * return: 0 で見つかった */

static int _goto_thunk(FILE *fp,APDv3Data *p,uint32_t thunk_id)
{
	uint32_t id,size;

	fseeko(fp, p->thunk_fpos, SEEK_SET);

	while(1)
	{
		//ID、チャンクサイズ
		
		if(mFILEreadFormatBE(fp, "ii", &id, &size))
			break;
	
		if(id == thunk_id)
		{
			p->thunk_size = size;
			return 0;
		}
		else if(id == MLK_MAKE32_4('E','N','D',' '))
			break;

		if(fseek(fp, size, SEEK_CUR))
			break;
	}

	return 1;
}

/* UTF-8 文字列をバッファに追加 */

static const char *_add_namebuf(FILE *fp,int len)
{
	char *buf;
	const char *retbuf = NULL;

	buf = (char *)mMalloc(len);
	if(buf)
	{
		if(fread(buf, 1, len, fp) == len)
			retbuf = add_name_utf8(buf, len);

		mFree(buf);
	}

	return retbuf;
}

/** レイヤ情報読み込み */

static mlkerr _read_layerinfo(FILE *fp,APDv3Data *p,int layernum)
{
	int32_t x1,y1,x2,y2;
	uint32_t j,blendid,flags,exid;
	uint16_t ver,exsize;
	uint8_t coltype,chnum,opacity,amask,r,g,b,blendmode;
	LayerData *pl;

	if(_goto_thunk(fp, p, MLK_MAKE32_4('L','Y','I','F'))
		|| mFILEreadBE16(fp, &ver))
		return MLKERR_DAMAGED;

	if(ver != 0)
		return MLKERR_UNSUPPORTED;

	//

	pl = p->layer;

	for(; layernum; layernum--, pl++)
	{
		//基本データ
		
		if(mFILEreadFormatBE(fp, "bbiiiibiibbbb",
			&coltype, &chnum, &x1, &y1, &x2, &y2,
			&opacity, &blendid, &flags, &amask,
			&r, &g, &b))
			return MLKERR_DAMAGED;

		blendmode = 0;

		//フォルダ以外の場合

		if(coltype != 255)
		{
			if(coltype >= 4
				|| (chnum != 1 && chnum != 2 && chnum != 4))
				return MLKERR_INVALID_VALUE;

			//合成モード (フォルダの場合 id = 0)

			for(j = 0; g_blendid[j]; j++)
			{
				if(blendid == g_blendid[j])
				{
					blendmode = j;
					break;
				}
			}
		}

		//拡張データ

		while(1)
		{
			if(mFILEreadBE32(fp, &exid))
				return MLKERR_DAMAGED;

			if(exid == 0) break;

			if(mFILEreadBE16(fp, &exsize))
				return MLKERR_DAMAGED;

			//

			if(exid == MLK_MAKE32_4('n','a','m','e'))
			{
				//名前

				pl->name = _add_namebuf(fp, exsize);
			}
			else if(exid == MLK_MAKE32_4('t','e','x','r'))
			{
				//レイヤテクスチャパス

				pl->texpath = _add_namebuf(fp, exsize);
			}
			else
			{
				if(fseek(fp, exsize, SEEK_CUR))
					return MLKERR_DAMAGED;
			}
		}

		//データをセット

		pl->parent = 0xffff;
		pl->blendmode = blendmode;
		pl->opacity = opacity;
		pl->col = MLK_RGB(r, g, b);

		if(coltype == 255) pl->flags |= LAYER_FLAGS_FOLDER;
		if(!(flags & 1)) pl->flags |= LAYER_FLAGS_HIDE;

		pl->offx = x1;
		pl->offy = y1;

	/*
		if(x1 >= x2 && y1 >= y2)
			pl->tilew = pl->tileh = 0;
		else
		{
			pl->tilew = (x2 - x1) / 64;
			pl->tileh = (y2 - y1) / 64;

			if(pl->tilew == 0) pl->tilew = 1;
			if(pl->tileh == 0) pl->tileh = 1;
		}
	*/

		//coltype = 0:8bit, 1:1bit, 2:16bit(固定小数点数15bit), 255:folder

		if(coltype != 255)
		{
			if(coltype == 2 && chnum == 4) //RGBA-16
				pl->coltype = 0;
			else if(coltype == 2 && chnum == 2) //GRAY+A-16
				pl->coltype = 1;
			else if(coltype == 2 && chnum == 1) //A16
				pl->coltype = 2;
			else if(coltype == 1 && chnum == 1) //A1
				pl->coltype = 3;
			else if(coltype == 0 && chnum == 4) //RGBA-8
				pl->coltype = 4;
			else
				return MLKERR_UNSUPPORTED;

			//

			p->fcoltype |= 1 << coltype;
		}
	}

	return MLKERR_OK;
}

/** レイヤツリー情報読み込み */

static mlkerr _read_layertree(FILE *fp,APDv3Data *p,int layernum)
{
	LayerData *pl;
	uint16_t parent;

	//データがなければ、すべてルート

	if(_goto_thunk(fp, p, MLK_MAKE32_4('L','Y','T','R')))
		return MLKERR_OK;

	//

	if(p->thunk_size != layernum * 2)
		return MLKERR_DAMAGED;

	for(pl = p->layer; layernum; layernum--, pl++)
	{
		if(mFILEreadBE16(fp, &parent))
			return MLKERR_DAMAGED;

		pl->parent = parent;
	}

	return MLKERR_OK;
}

/** レイヤタイル情報読み込み */

static mlkerr _read_layertile(FILE *fp,APDv3Data *p,int layernum)
{
	LayerData *pl;
	uint32_t tilenum,size;
	uint16_t tilesize;
	uint8_t comp;

	if(_goto_thunk(fp, p, MLK_MAKE32_4('L','Y','T','L'))
		|| p->thunk_size < 3
		|| mFILEreadBE16(fp, &tilesize)
		|| mFILEreadByte(fp, &comp))
		return MLKERR_DAMAGED;

	if(tilesize != 64 || comp != 0)
		return MLKERR_INVALID_VALUE;

	//各レイヤごと

	pl = p->layer;

	for(; layernum; layernum--, pl++)
	{
		if(mFILEreadBE32(fp, &tilenum)
			|| mFILEreadBE32(fp, &size))
			return MLKERR_DAMAGED;

		pl->fpos = ftello(fp);
		pl->imgparam = tilenum;
		pl->imgparam2 = size;

		if(fseek(fp, size, SEEK_CUR))
			return MLKERR_DAMAGED;
	}

	return MLKERR_OK;
}

/** 読み込みメイン */

static mlkerr _main_proc(FILE *fp,APDv3Data *p)
{
	int32_t imgw,imgh,dpi;
	uint16_t layernum;
	uint8_t unit;
	mlkerr ret;

	//情報

	if(mFILEreadFormatBE(fp, "iibi", &imgw, &imgh, &unit, &dpi))
		return MLKERR_DAMAGED;

	p->w = imgw;
	p->h = imgh;
	p->thunk_fpos = ftello(fp);

	//dpi

	if(unit != 1 || dpi < 1)
		dpi = 96;

	g_work.src_dpi = dpi;

	//レイヤヘッダ

	if(_goto_thunk(fp, p, MLK_MAKE32_4('L','Y','H','D'))
		|| p->thunk_size < 4
		|| mFILEreadBE16(fp, &layernum))
		return MLKERR_DAMAGED;

	//

	printf(" (%dx%d) layer:%d, %d dpi\n", imgw, imgh, layernum, dpi);

	//レイヤ情報

	p->layer = (LayerData *)mMalloc0(sizeof(LayerData) * layernum);
	if(!p->layer) return MLKERR_ALLOC;

	//レイヤ情報読み込み

	ret = _read_layerinfo(fp, p, layernum);
	if(ret) return ret;

	//レイヤツリー読み込み

	ret = _read_layertree(fp, p, layernum);
	if(ret) return ret;

	//

	if(ISNOT_PROC_INFO)
	{
		//タイル情報読み込み

		ret = _read_layertile(fp, p, layernum);
		if(ret) return ret;

		//レイヤテクスチャ読み込み

		read_layertexture(p->layer, layernum);

		//レイヤイメージ
		
		if(IS_PROC_LAYER)
		{
			p->img = mImageBuf2_new(imgw, imgh, 32, 0);
			if(!p->img) return MLKERR_ALLOC;
		}

		//タイルバッファ

		p->tilebuf = (uint8_t *)mMalloc(64 * 64 * 2 * 4);
		if(!p->tilebuf) return MLKERR_ALLOC;
	}

	//処理

	switch(g_work.proctype)
	{
		case PROCTYPE_BLEND:
			//8bit は 8bit レイヤしか存在してはいけない
			if((p->fcoltype & _COLTYPE_F_8BIT) && (p->fcoltype != _COLTYPE_F_8BIT))
				return MLKERR_INVALID_VALUE;
			
			//8bit RGBA のレイヤのみであれば、8bit
			return output_blendimage_tile(imgw, imgh, p->layer, layernum, _blend_layer, p,
				(p->fcoltype == _COLTYPE_F_8BIT)? 24: 48);

		case PROCTYPE_LAYER:
			return output_layers(p->img, p->layer, layernum, _read_layer_image, p);

		case PROCTYPE_INFO:
			put_layerinfo_texpath(p->layer, layernum);
			break;
	}

	return MLKERR_OK;
}

/** APD v3 読み込み (BigEndian) */

mlkerr load_apd_v3(FILE *fp)
{
	APDv3Data dat;
	mlkerr ret;

	mMemset0(&dat, sizeof(APDv3Data));
	dat.fp = fp;

	//zlib

	if(ISNOT_PROC_INFO)
	{
		dat.zdec = mZlibDecNew(8 * 1024, MZLIB_WINDOWBITS_ZLIB);
		if(!dat.zdec) return MLKERR_UNKNOWN;

		mZlibSetIO_stdio(dat.zdec, fp);
	}

	//

	ret = _main_proc(fp, &dat);

	//

	mZlibFree(dat.zdec);
	mImageBuf2_free(dat.img);
	mFree(dat.tilebuf);
	mFree(dat.layer);

	return ret;
}
