/*$
apdtool
Copyright (c) 2020 Azel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
$*/

/*****************************************
 * ADW ver 1 読み込み
 *****************************************/

#include <stdio.h>

#include "mlk.h"
#include "mlk_zlib.h"
#include "mlk_stdio.h"
#include "mlk_imagebuf.h"

#include "def.h"


//-----------------

typedef struct
{
	FILE *fp;
	mZlib *zdec;
	mImageBuf2 *img;
	uint8_t *readbuf;	//Y1行読み込みバッファ
	LayerData *layer;
	
	int w,h,pitch;
}ADWv1Data;

//-----------------


/** レイヤイメージ読み込み */

static mlkerr _read_layer_image(LayerData *pl,void *ptr)
{
	ADWv1Data *p = (ADWv1Data *)ptr;
	uint8_t **ppbuf,*readbuf,*pd,*ps,r,g,b;
	int i,ix,pitch;
	mlkerr ret;

	fseek(p->fp, pl->fpos, SEEK_SET);

	//

	ppbuf = p->img->ppbuf;
	readbuf = p->readbuf;
	pitch = p->pitch;

	r = MLK_RGB_R(pl->col);
	g = MLK_RGB_G(pl->col);
	b = MLK_RGB_B(pl->col);

	mZlibDecReset(p->zdec);
	mZlibDecSetSize(p->zdec, pl->imgparam);

	//

	for(i = p->h; i; i--, ppbuf++)
	{
		//8bit Y1行読み込み
		
		ret = mZlibDecRead(p->zdec, readbuf, pitch);
		if(ret) return ret;

		//RGBA に変換

		ps = readbuf;
		pd = *ppbuf;

		for(ix = p->w; ix; ix--, pd += 4)
		{
			pd[0] = r;
			pd[1] = g;
			pd[2] = b;
			pd[3] = *(ps++);
		}
	}

	return mZlibDecFinish(p->zdec);
}

/** レイヤ情報読み込み */

static mlkerr _read_layerinfo(FILE *fp,ADWv1Data *p,int layernum)
{
	LayerData *pl;
	char name[32];
	uint32_t size,col;
	uint8_t opacity,flags;

	pl = p->layer + layernum - 1;

	for(; layernum; layernum--, pl--)
	{
		//レイヤ情報

		if(fread(name, 1, 32, fp) != 32
			|| mFILEreadFormatLE(fp, "bbii", &opacity, &flags, &col, &size))
			return MLKERR_DAMAGED;

		//レイヤ名

		name[31] = 0;
		pl->name = add_name_checkcode(name, -1);

		//セット

		pl->fpos = ftello(fp);
		pl->parent = 0xffff;
		pl->opacity = opacity;
		pl->col = col;
		pl->imgparam = size;

		if(!(flags & 1)) pl->flags |= LAYER_FLAGS_HIDE;

		//イメージスキップ

		if(fseek(fp, size, SEEK_CUR))
			return MLKERR_DAMAGED;
	}

	return MLKERR_OK;
}

/** 読み込みメイン */

static mlkerr _main_proc(FILE *fp,ADWv1Data *p)
{
	uint32_t size;
	uint16_t imgw,imgh,layernum,layercnt,layersel;
	uint8_t comptype;
	mlkerr ret;

	//メイン情報

	if(mFILEreadFormatLE(fp, "hhhhhb",
		&imgw, &imgh, &layernum, &layercnt, &layersel, &comptype))
		return MLKERR_DAMAGED;

	if(comptype != 0)
		return MLKERR_INVALID_VALUE;
	
	p->w = imgw;
	p->h = imgh;
	p->pitch = (imgw + 3) & (~3);

	printf(" (%dx%d) layer:%d\n", imgw, imgh, layernum);

	//プレビューイメージをスキップ

	if(mFILEreadLE32(fp, &size)
		|| fseek(fp, size, SEEK_CUR))
		return MLKERR_DAMAGED;
	
	//レイヤデータ

	p->layer = (LayerData *)mMalloc0(sizeof(LayerData) * layernum);
	if(!p->layer) return MLKERR_ALLOC;

	//レイヤ情報読み込み

	ret = _read_layerinfo(fp, p, layernum);
	if(ret) return ret;

	//確保

	if(ISNOT_PROC_INFO)
	{
		//レイヤイメージ
	
		p->img = mImageBuf2_new(imgw, imgh, 32, 0);
		if(!p->img) return MLKERR_ALLOC;

		//読み込みバッファ

		p->readbuf = (uint8_t *)mMalloc(p->pitch);
		if(!p->readbuf) return MLKERR_ALLOC;
	}

	//処理

	switch(g_work.proctype)
	{
		case PROCTYPE_BLEND:
			return output_blendimage8(imgw, imgh, p->img,
				p->layer, layernum, _read_layer_image, p);

		case PROCTYPE_LAYER:
			return output_layers(p->img, p->layer, layernum, _read_layer_image, p);

		case PROCTYPE_INFO:
			put_layerinfo(p->layer, layernum);
			break;
	}

	return MLKERR_OK;
}

/** ADW v1 読み込み */

mlkerr load_adw_v1(FILE *fp)
{
	ADWv1Data dat;
	mlkerr ret;

	mMemset0(&dat, sizeof(ADWv1Data));
	dat.fp = fp;

	g_work.src_dpi = 0;

	//zlib

	if(ISNOT_PROC_INFO)
	{
		dat.zdec = mZlibDecNew(8 * 1024, MZLIB_WINDOWBITS_ZLIB);
		if(!dat.zdec) return MLKERR_UNKNOWN;

		mZlibSetIO_stdio(dat.zdec, fp);
	}

	//

	ret = _main_proc(fp, &dat);

	//

	mZlibFree(dat.zdec);
	mImageBuf2_free(dat.img);
	mFree(dat.readbuf);
	mFree(dat.layer);

	return ret;
}

