/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

// Copyright 2012 the V8 project authors. All rights reserved.
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
//       copyright notice, this list of conditions and the following
//       disclaimer in the documentation and/or other materials provided
//       with the distribution.
//     * Neither the name of Google Inc. nor the names of its
//       contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#include "irregexp/RegExpCharRanges.h"

#include "ds/LifoAlloc.h"
#include "mozilla/Unused.h"
#include "unicode/uniset.h"
#include "vm/Unicode.h"

// Generated table
#include "irregexp/RegExpCharacters-inl.h"

using namespace js::irregexp;

using mozilla::ArrayLength;

void
CharacterRange::AddCaseEquivalents(bool is_ascii, bool unicode, CharacterRangeVector* ranges)
{
    char16_t bottom = from();
    char16_t top = to();

    if (is_ascii && !RangeContainsLatin1Equivalents(*this, unicode)) {
        if (bottom > kMaxOneByteCharCode)
            return;
        if (top > kMaxOneByteCharCode)
            top = kMaxOneByteCharCode;
    }

    for (char16_t c = bottom;; c++) {
        char16_t chars[kEcma262UnCanonicalizeMaxWidth];
        size_t length = GetCaseIndependentLetters(c, is_ascii, unicode, chars);

        for (size_t i = 0; i < length; i++) {
            char16_t other = chars[i];
            if (other == c)
                continue;

            // Try to combine with an existing range.
            bool found = false;
            for (size_t i = 0; i < ranges->length(); i++) {
                CharacterRange& range = (*ranges)[i];
                if (range.Contains(other)) {
                    found = true;
                    break;
                } else if (other == range.from() - 1) {
                    range.set_from(other);
                    found = true;
                    break;
                } else if (other == range.to() + 1) {
                    range.set_to(other);
                    found = true;
                    break;
                }
            }

            if (!found)
                ranges->append(CharacterRange::Singleton(other));
        }

        if (c == top)
            break;
    }
}

/* static */
void
CharacterRange::AddClass(const int* elmv, int elmc, CharacterRangeVector* ranges)
{
    elmc--;
    MOZ_ASSERT(elmv[elmc] == 0x10000);
    for (int i = 0; i < elmc; i += 2) {
        MOZ_ASSERT(elmv[i] < elmv[i + 1]);
        ranges->append(CharacterRange(elmv[i], elmv[i + 1] - 1));
    }
}

/* static */ void
CharacterRange::AddClassNegated(const int* elmv, int elmc, CharacterRangeVector* ranges)
{
    elmc--;
    MOZ_ASSERT(elmv[elmc] == 0x10000);
    MOZ_ASSERT(elmv[0] != 0x0000);
    MOZ_ASSERT(elmv[elmc-1] != kMaxUtf16CodeUnit);
    char16_t last = 0x0000;
    for (int i = 0; i < elmc; i += 2) {
        MOZ_ASSERT(last <= elmv[i] - 1);
        MOZ_ASSERT(elmv[i] < elmv[i + 1]);
        ranges->append(CharacterRange(last, elmv[i] - 1));
        last = elmv[i + 1];
    }
    ranges->append(CharacterRange(last, kMaxUtf16CodeUnit));
}

/* static */ void
CharacterRange::AddClassEscape(LifoAlloc* alloc, char16_t type,
			       CharacterRangeVector* ranges)
{
    switch (type) {
      case 's':
        AddClass(kSpaceRanges, kSpaceRangeCount, ranges);
        break;
      case 'S':
        AddClassNegated(kSpaceRanges, kSpaceRangeCount, ranges);
        break;
      case 'w':
        AddClass(kWordRanges, kWordRangeCount, ranges);
        break;
      case 'W':
        AddClassNegated(kWordRanges, kWordRangeCount, ranges);
        break;
      case 'd':
        AddClass(kDigitRanges, kDigitRangeCount, ranges);
        break;
      case 'D':
        AddClassNegated(kDigitRanges, kDigitRangeCount, ranges);
        break;
      case '.':
        AddClassNegated(kLineTerminatorRanges, kLineTerminatorRangeCount, ranges);
        break;
        // This is not a character range as defined by the spec but a
        // convenient shorthand for a character class that matches any
        // character.
      case '*':
        ranges->append(CharacterRange::Everything());
        break;
        // This is the set of characters matched by the $ and ^ symbols
        // in multiline mode.
      case 'n':
        AddClass(kLineTerminatorRanges, kLineTerminatorRangeCount, ranges);
        break;
      default:
        MOZ_CRASH("Bad character class escape");
    }
}

// Add class escape, excluding surrogate pair range.
/* static */ void
CharacterRange::AddClassEscapeUnicode(LifoAlloc* alloc, char16_t type,
                                      CharacterRangeVector* ranges, bool ignore_case)
{
    switch (type) {
      case 's':
      case 'd':
        return AddClassEscape(alloc, type, ranges);
        break;
      case 'S':
        AddClassNegated(kSpaceAndSurrogateRanges, kSpaceAndSurrogateRangeCount, ranges);
        break;
      case 'w':
        if (ignore_case)
            AddClass(kIgnoreCaseWordRanges, kIgnoreCaseWordRangeCount, ranges);
        else
            AddClassEscape(alloc, type, ranges);
        break;
      case 'W':
        if (ignore_case) {
            AddClass(kNegatedIgnoreCaseWordAndSurrogateRanges,
                     kNegatedIgnoreCaseWordAndSurrogateRangeCount, ranges);
        } else {
            AddClassNegated(kWordAndSurrogateRanges, kWordAndSurrogateRangeCount, ranges);
        }
        break;
      case 'D':
        AddClassNegated(kDigitAndSurrogateRanges, kDigitAndSurrogateRangeCount, ranges);
        break;
      default:
        MOZ_CRASH("Bad type!");
    }
}

/* static */ void
CharacterRange::AddCharOrEscape(LifoAlloc* alloc, CharacterRangeVector* ranges,
                                char16_t char_class, widechar c)
{
    if (char_class != kNoCharClass)
        AddClassEscape(alloc, char_class, ranges);
    else
        ranges->append(CharacterRange::Singleton(c));
}

/* static */ void
CharacterRange::AddCharOrEscapeUnicode(LifoAlloc* alloc,
                                       CharacterRangeVector* ranges,
                                       CharacterRangeVector* lead_ranges,
                                       CharacterRangeVector* trail_ranges,
                                       WideCharRangeVector* wide_ranges,
                                       char16_t char_class,
                                       widechar c,
                                       bool ignore_case)
{
    if (char_class != kNoCharClass) {
        AddClassEscapeUnicode(alloc, char_class, ranges, ignore_case);
        switch (char_class) {
          case 'S':
          case 'W':
          case 'D':
            lead_ranges->append(CharacterRange::LeadSurrogate());
            trail_ranges->append(CharacterRange::TrailSurrogate());
            wide_ranges->append(WideCharRange::NonBMP());
            break;
          case '.':
            MOZ_CRASH("Bad char_class!");
        }
        return;
    }

    if (unicode::IsLeadSurrogate(c))
        lead_ranges->append(CharacterRange::Singleton(c));
    else if (unicode::IsTrailSurrogate(c))
        trail_ranges->append(CharacterRange::Singleton(c));
    else if (c >= unicode::NonBMPMin)
        wide_ranges->append(WideCharRange::Singleton(c));
    else
        ranges->append(CharacterRange::Singleton(c));
}

/* static */ void
CharacterRange::AddCharUnicode(LifoAlloc* alloc,
                               CharacterRangeVector* ranges,
                               CharacterRangeVector* lead_ranges,
                               CharacterRangeVector* trail_ranges,
                               WideCharRangeVector* wide_ranges,
                               widechar c)
{
    if (unicode::IsLeadSurrogate(c))
        lead_ranges->append(CharacterRange::Singleton(c));
    else if (unicode::IsTrailSurrogate(c))
        trail_ranges->append(CharacterRange::Singleton(c));
    else if (c >= unicode::NonBMPMin)
        wide_ranges->append(WideCharRange::Singleton(c));
    else
        ranges->append(CharacterRange::Singleton(c));
}

/* static */ void
CharacterRange::AddUnicodeRange(LifoAlloc* alloc,
                                CharacterRangeVector* ranges,
                                CharacterRangeVector* lead_ranges,
                                CharacterRangeVector* trail_ranges,
                                WideCharRangeVector* wide_ranges,
                                widechar first,
                                widechar next)
{
    MOZ_ASSERT(first <= next);
    if (first < unicode::LeadSurrogateMin) {
        if (next < unicode::LeadSurrogateMin) {
            ranges->append(CharacterRange::Range(first, next));
            return;
        }
        ranges->append(CharacterRange::Range(first, unicode::LeadSurrogateMin - 1));
        first = unicode::LeadSurrogateMin;
    }
    if (first <= unicode::LeadSurrogateMax) {
        if (next <= unicode::LeadSurrogateMax) {
            lead_ranges->append(CharacterRange::Range(first, next));
            return;
        }
        lead_ranges->append(CharacterRange::Range(first, unicode::LeadSurrogateMax));
        first = unicode::LeadSurrogateMax + 1;
    }
    MOZ_ASSERT(unicode::LeadSurrogateMax + 1 == unicode::TrailSurrogateMin);
    if (first <= unicode::TrailSurrogateMax) {
        if (next <= unicode::TrailSurrogateMax) {
            trail_ranges->append(CharacterRange::Range(first, next));
            return;
        }
        trail_ranges->append(CharacterRange::Range(first, unicode::TrailSurrogateMax));
        first = unicode::TrailSurrogateMax + 1;
    }
    if (first <= unicode::UTF16Max) {
        if (next <= unicode::UTF16Max) {
            ranges->append(CharacterRange::Range(first, next));
            return;
        }
        ranges->append(CharacterRange::Range(first, unicode::UTF16Max));
        first = unicode::NonBMPMin;
    }
    MOZ_ASSERT(unicode::UTF16Max + 1 == unicode::NonBMPMin);
    wide_ranges->append(WideCharRange::Range(first, next));
}

/* static */ bool
CharacterRange::RangesContainLatin1Equivalents(const CharacterRangeVector& ranges, bool unicode)
{
    for (size_t i = 0; i < ranges.length(); i++) {
        // TODO(dcarney): this could be a lot more efficient.
        if (RangeContainsLatin1Equivalents(ranges[i], unicode))
            return true;
    }
    return false;
}

/* static */ bool
CharacterRange::CompareRanges(const CharacterRangeVector& ranges, const int* special_class, size_t length)
{
    length--;  // Remove final 0x10000.
    MOZ_ASSERT(special_class[length] == 0x10000);
    if (ranges.length() * 2 != length)
        return false;
    for (size_t i = 0; i < length; i += 2) {
        CharacterRange range = ranges[i >> 1];
        if (range.from() != special_class[i] || range.to() != special_class[i + 1] - 1)
            return false;
    }
    return true;
}

/* static */ bool
CharacterRange::CompareInverseRanges(const CharacterRangeVector& ranges, const int* special_class, size_t length)
{
    length--;  // Remove final 0x10000.
    MOZ_ASSERT(special_class[length] == 0x10000);
    MOZ_ASSERT(ranges.length() != 0);
    MOZ_ASSERT(length != 0);
    MOZ_ASSERT(special_class[0] != 0);
    if (ranges.length() != (length >> 1) + 1)
        return false;
    CharacterRange range = ranges[0];
    if (range.from() != 0)
        return false;
    for (size_t i = 0; i < length; i += 2) {
        if (special_class[i] != (range.to() + 1))
            return false;
        range = ranges[(i >> 1) + 1];
        if (special_class[i+1] != range.from())
            return false;
    }
    if (range.to() != 0xffff)
        return false;
    return true;
}

template <typename RangeType>
/* static */ void
CharacterRange::NegateUnicodeRanges(LifoAlloc* alloc, InfallibleVector<RangeType, 1>** ranges,
                                    RangeType full_range)
{
    typedef InfallibleVector<RangeType, 1> RangeVector;
    RangeVector* tmp_ranges = alloc->newInfallible<RangeVector>(*alloc);
    tmp_ranges->append(full_range);
    RangeVector* result_ranges = alloc->newInfallible<RangeVector>(*alloc);

    // Perform the following calculation:
    //   result_ranges = tmp_ranges - ranges
    // with the following steps:
    //   result_ranges = tmp_ranges - ranges[0]
    //   SWAP(result_ranges, tmp_ranges)
    //   result_ranges = tmp_ranges - ranges[1]
    //   SWAP(result_ranges, tmp_ranges)
    //   ...
    //   result_ranges = tmp_ranges - ranges[N-1]
    //   SWAP(result_ranges, tmp_ranges)
    // The last SWAP is just for simplicity of the loop.
    for (size_t i = 0; i < (*ranges)->length(); i++) {
        result_ranges->clear();

        const RangeType& range = (**ranges)[i];
        for (size_t j = 0; j < tmp_ranges->length(); j++) {
            const RangeType& tmpRange = (*tmp_ranges)[j];
            auto from1 = tmpRange.from();
            auto to1 = tmpRange.to();
            auto from2 = range.from();
            auto to2 = range.to();

            if (from1 < from2) {
                if (to1 < from2) {
                    result_ranges->append(tmpRange);
                } else if (to1 <= to2) {
                    result_ranges->append(RangeType::Range(from1, from2 - 1));
                } else {
                    result_ranges->append(RangeType::Range(from1, from2 - 1));
                    result_ranges->append(RangeType::Range(to2 + 1, to1));
                }
            } else if (from1 <= to2) {
                if (to1 > to2)
                    result_ranges->append(RangeType::Range(to2 + 1, to1));
            } else {
                result_ranges->append(tmpRange);
            }
        }

        auto tmp = tmp_ranges;
        tmp_ranges = result_ranges;
        result_ranges = tmp;
    }

    // After the loop, result is pointed at by tmp_ranges, instead of
    // result_ranges.
    *ranges = tmp_ranges;
}

// Explicit specialization for NegateUnicodeRanges
template void CharacterRange::NegateUnicodeRanges<CharacterRange>(LifoAlloc* alloc, InfallibleVector<CharacterRange, 1>** ranges, CharacterRange full_range);
template void CharacterRange::NegateUnicodeRanges<WideCharRange>(LifoAlloc* alloc, InfallibleVector<WideCharRange, 1>** ranges, WideCharRange full_range);


namespace {
// private namespace to not pollute js::irregexp

bool IsExactPropertyAlias(const std::string& property_name, UProperty property) {
    const char* short_name = u_getPropertyName(property, U_SHORT_PROPERTY_NAME);
    if (short_name != nullptr && short_name == property_name)
        return true;
    for (int i = 0;; i++) {
        const char* long_name = u_getPropertyName(property, static_cast<UPropertyNameChoice>(U_LONG_PROPERTY_NAME + i));
        if (long_name == nullptr) break;
        if (long_name == property_name) return true;
    }
    return false;
}

bool IsExactPropertyValueAlias(const std::string& property_value_name, UProperty property, int32_t property_value) {
    const char* short_name = u_getPropertyValueName(property, property_value, U_SHORT_PROPERTY_NAME);
    if (short_name != nullptr && short_name == property_value_name)
        return true;
    for (int i = 0;; i++) {
      const char* long_name = u_getPropertyValueName(property, property_value,
                                                     static_cast<UPropertyNameChoice>(U_LONG_PROPERTY_NAME + i));
      if (long_name == nullptr) break;
      if (long_name == property_value_name) return true;
    }
    return false;
}

bool LookupPropertyValueName(js::LifoAlloc* alloc,
                             UProperty property,
                             const std::string& property_value_name, bool negate,
                             CharacterRangeVector* ranges,
                             CharacterRangeVector* lead_ranges,
                             CharacterRangeVector* trail_ranges,
                             WideCharRangeVector* wide_ranges) {
    UProperty property_for_lookup = property;
    if (property_for_lookup == UCHAR_SCRIPT_EXTENSIONS) {
        // For the property Script_Extensions, we have to do the property value
        // name lookup as if the property is Script.
        property_for_lookup = UCHAR_SCRIPT;
    }
    int32_t property_value = u_getPropertyValueEnum(property_for_lookup, property_value_name.c_str());
    if (property_value == UCHAR_INVALID_CODE) return false;

    // We require the property name to match exactly to one of the property value
    // aliases. However, u_getPropertyValueEnum uses loose matching.
    if (!IsExactPropertyValueAlias(property_value_name, property_for_lookup, property_value)) {
        return false;
    }

    UErrorCode ec = U_ZERO_ERROR;
    icu::UnicodeSet set;
    set.applyIntPropertyValue(property, property_value, ec);
    bool success = ec == U_ZERO_ERROR && !set.isEmpty();

    if (success) {
        set.removeAllStrings();
        if (negate) set.complement();
        for (int i = 0; i < set.getRangeCount(); i++) {
            CharacterRange::AddUnicodeRange(alloc, ranges, lead_ranges, trail_ranges, wide_ranges,
                                            set.getRangeStart(i), set.getRangeEnd(i));
        }
    }
    return success;
}

bool LookupSpecialPropertyValueName(js::LifoAlloc* alloc,
                                    const std::string& name, bool negate,
                                    CharacterRangeVector* ranges,
                                    CharacterRangeVector* lead_ranges,
                                    CharacterRangeVector* trail_ranges,
                                    WideCharRangeVector* wide_ranges) {
    if (name == "Any") {
        if (negate) {
            // Leave the list of character ranges empty, since the negation of 'Any'
            // is the empty set.
        } else {
            CharacterRange::AddUnicodeRange(alloc, ranges, lead_ranges, trail_ranges, wide_ranges,
                                            0, js::unicode::NonBMPMax);
        }
    } else
    if (name == "ASCII") {
        if (negate) {
            // negative ASCII contains all planes
            CharacterRange::AddUnicodeRange(alloc, ranges, lead_ranges, trail_ranges, wide_ranges,
                                            0x80, js::unicode::NonBMPMax);
        } else {
            // positve ASCII is just low codepoints
            ranges->append(CharacterRange::Range(0x00, 0x7F));
        }
    } else
    if (name == "Assigned") {
        return LookupPropertyValueName(alloc, UCHAR_GENERAL_CATEGORY, "Unassigned", !negate,
                                       ranges, lead_ranges, trail_ranges, wide_ranges);
    } else {
        return false;
    }
    return true;
}

bool IsSupportedBinaryProperty(UProperty property) {
    // Explicitly allowlist supported binary properties. The spec forbids supporting
    // properties outside of this set to ensure interoperability.
    switch (property) {
        case UCHAR_ALPHABETIC:
        // 'Any' is not supported by ICU. See LookupSpecialPropertyValueName.
        // 'ASCII' is not supported by ICU. See LookupSpecialPropertyValueName.
        case UCHAR_ASCII_HEX_DIGIT:
        // 'Assigned' is not supported by ICU. See LookupSpecialPropertyValueName.
        case UCHAR_BIDI_CONTROL:
        case UCHAR_BIDI_MIRRORED:
        case UCHAR_CASE_IGNORABLE:
        case UCHAR_CASED:
        case UCHAR_CHANGES_WHEN_CASEFOLDED:
        case UCHAR_CHANGES_WHEN_CASEMAPPED:
        case UCHAR_CHANGES_WHEN_LOWERCASED:
        case UCHAR_CHANGES_WHEN_NFKC_CASEFOLDED:
        case UCHAR_CHANGES_WHEN_TITLECASED:
        case UCHAR_CHANGES_WHEN_UPPERCASED:
        case UCHAR_DASH:
        case UCHAR_DEFAULT_IGNORABLE_CODE_POINT:
        case UCHAR_DEPRECATED:
        case UCHAR_DIACRITIC:
        case UCHAR_EMOJI:
        case UCHAR_EMOJI_COMPONENT:
        case UCHAR_EMOJI_MODIFIER_BASE:
        case UCHAR_EMOJI_MODIFIER:
        case UCHAR_EMOJI_PRESENTATION:
        // case UCHAR_EXTENDED_PICTOGRAPHIC:
        case UCHAR_EXTENDER:
        case UCHAR_GRAPHEME_BASE:
        case UCHAR_GRAPHEME_EXTEND:
        case UCHAR_HEX_DIGIT:
        case UCHAR_ID_CONTINUE:
        case UCHAR_ID_START:
        case UCHAR_IDEOGRAPHIC:
        case UCHAR_IDS_BINARY_OPERATOR:
        case UCHAR_IDS_TRINARY_OPERATOR:
        case UCHAR_JOIN_CONTROL:
        case UCHAR_LOGICAL_ORDER_EXCEPTION:
        case UCHAR_LOWERCASE:
        case UCHAR_MATH:
        case UCHAR_NONCHARACTER_CODE_POINT:
        case UCHAR_PATTERN_SYNTAX:
        case UCHAR_PATTERN_WHITE_SPACE:
        case UCHAR_QUOTATION_MARK:
        case UCHAR_RADICAL:
        case UCHAR_REGIONAL_INDICATOR:
        case UCHAR_S_TERM:
        case UCHAR_SOFT_DOTTED:
        case UCHAR_TERMINAL_PUNCTUATION:
        case UCHAR_UNIFIED_IDEOGRAPH:
        case UCHAR_UPPERCASE:
        case UCHAR_VARIATION_SELECTOR:
        case UCHAR_WHITE_SPACE:
        case UCHAR_XID_CONTINUE:
        case UCHAR_XID_START:
            return true;
        default:
            break;
    }
    return false;
}

} // namespace

/* static */ bool
CharacterRange::AddPropertyClassRange(LifoAlloc* alloc, 
                                      const std::string& name, const std::string& value,
                                      bool negate, bool ignore_case,
                                      CharacterRangeVector* ranges,
                                      CharacterRangeVector* lead_ranges,
                                      CharacterRangeVector* trail_ranges,
                                      WideCharRangeVector* wide_ranges)
{
    MOZ_UNUSED(ignore_case); // Full support of unicodeSets flag will use it
    if (value.empty()) {
        // Only one name given. Check what it refers to.
        // First attempt to interpret as general category property value name.
        if (LookupPropertyValueName(alloc, UCHAR_GENERAL_CATEGORY_MASK, name, negate,
                                    ranges, lead_ranges, trail_ranges, wide_ranges)) {
            return true;
        }
        // Interpret hard-coded aliases defined by ES but not present in Unicode
        if (LookupSpecialPropertyValueName(alloc, name, negate, ranges, lead_ranges,
                                           trail_ranges, wide_ranges)) {
            return true;
        }
        // Then attempt to interpret as binary property name with value name 'Y'.
        UProperty property = u_getPropertyEnum(name.c_str());
        if (!IsSupportedBinaryProperty(property)) return false;
        if (!IsExactPropertyAlias(name, property)) return false;
        return LookupPropertyValueName(alloc, property, negate ? "N" : "Y", false,
                                       ranges, lead_ranges, trail_ranges, wide_ranges);
    } else {
        // Both property name and value name are specified. Attempt to interpret
        // the property name as enumerated property.
        UProperty property = u_getPropertyEnum(name.c_str());
        if (!IsExactPropertyAlias(name, property)) return false;
        if (property == UCHAR_GENERAL_CATEGORY) {
            // We want to allow aggregate value names such as "Letter".
            property = UCHAR_GENERAL_CATEGORY_MASK;
        } else if (property != UCHAR_SCRIPT &&
                   property != UCHAR_SCRIPT_EXTENSIONS) {
            // The only allowed property is Script=
            return false;
        }
        return LookupPropertyValueName(alloc, property, value, negate,
                                       ranges, lead_ranges, trail_ranges, wide_ranges);
    }
}

/* static */ bool
CharacterRange::IsCanonical(const CharacterRangeVector& ranges)
{
    int n = ranges.length();
    if (n <= 1)
        return true;

    int max = ranges[0].to();
    for (int i = 1; i < n; i++) {
        CharacterRange next_range = ranges[i];
        if (next_range.from() <= max + 1)
            return false;
        max = next_range.to();
    }
    return true;
}

/* static */ void
CharacterRange::Canonicalize(CharacterRangeVector& character_ranges)
{
    if (character_ranges.length() <= 1) return;
    // Check whether ranges are already canonical (increasing, non-overlapping,
    // non-adjacent).
    int n = character_ranges.length();
    int max = character_ranges[0].to();
    int i = 1;
    while (i < n) {
        CharacterRange current = character_ranges[i];
        if (current.from() <= max + 1) {
            break;
        }
        max = current.to();
        i++;
    }
    // Canonical until the i'th range. If that's all of them, we are done.
    if (i == n) return;

    // The ranges at index i and forward are not canonicalized. Make them so by
    // doing the equivalent of insertion sort (inserting each into the previous
    // list, in order).
    // Notice that inserting a range can reduce the number of ranges in the
    // result due to combining of adjacent and overlapping ranges.
    int read = i;  // Range to insert.
    size_t num_canonical = i;  // Length of canonicalized part of list.
    do {
        num_canonical = InsertRangeInCanonicalList(character_ranges,
                                                   num_canonical,
                                                   character_ranges[read]);
        read++;
    } while (read < n);

    while (character_ranges.length() > num_canonical)
        character_ranges.popBack();

    MOZ_ASSERT(IsCanonical(character_ranges));
}

/* static */ int
CharacterRange::InsertRangeInCanonicalList(CharacterRangeVector& list,
                                           int count,
                                           CharacterRange insert)
{
    // Inserts a range into list[0..count[, which must be sorted
    // by from value and non-overlapping and non-adjacent, using at most
    // list[0..count] for the result. Returns the number of resulting
    // canonicalized ranges. Inserting a range may collapse existing ranges into
    // fewer ranges, so the return value can be anything in the range 1..count+1.
    char16_t from = insert.from();
    char16_t to = insert.to();
    int start_pos = 0;
    int end_pos = count;
    for (int i = count - 1; i >= 0; i--) {
        CharacterRange current = list[i];
        if (current.from() > to + 1) {
            end_pos = i;
        } else if (current.to() + 1 < from) {
            start_pos = i + 1;
            break;
        }
    }

    // Inserted range overlaps, or is adjacent to, ranges at positions
    // [start_pos..end_pos[. Ranges before start_pos or at or after end_pos are
    // not affected by the insertion.
    // If start_pos == end_pos, the range must be inserted before start_pos.
    // if start_pos < end_pos, the entire range from start_pos to end_pos
    // must be merged with the insert range.

    if (start_pos == end_pos) {
        // Insert between existing ranges at position start_pos.
        if (start_pos < count) {
            list.moveReplace(start_pos, start_pos + 1, count - start_pos);
        }
        list[start_pos] = insert;
        return count + 1;
    }
    if (start_pos + 1 == end_pos) {
        // Replace single existing range at position start_pos.
        CharacterRange to_replace = list[start_pos];
        int new_from = Min(to_replace.from(), from);
        int new_to = Max(to_replace.to(), to);
        list[start_pos] = CharacterRange(new_from, new_to);
        return count;
    }
    // Replace a number of existing ranges from start_pos to end_pos - 1.
    // Move the remaining ranges down.

    int new_from = Min(list[start_pos].from(), from);
    int new_to = Max(list[end_pos - 1].to(), to);
    if (end_pos < count) {
        list.moveReplace(end_pos, start_pos + 1, count - end_pos);
    }
    list[start_pos] = CharacterRange(new_from, new_to);
    return count - (end_pos - start_pos) + 1;
}

int
js::irregexp::GetCaseIndependentLetters(char16_t character,
                                        bool ascii_subject,
                                        bool unicode,
                                        const char16_t* choices,
                                        size_t choices_length,
                                        char16_t* letters)
{
    size_t count = 0;
    for (size_t i = 0; i < choices_length; i++) {
        char16_t c = choices[i];

        // Skip characters that can't appear in one byte strings.
        if (!unicode && ascii_subject && c > kMaxOneByteCharCode)
            continue;

        // Watch for duplicates.
        bool found = false;
        for (size_t j = 0; j < count; j++) {
            if (letters[j] == c) {
                found = true;
                break;
            }
        }
        if (found)
            continue;

        letters[count++] = c;
    }

    return count;
}

int
js::irregexp::GetCaseIndependentLetters(char16_t character,
                                        bool ascii_subject,
                                        bool unicode,
                                        char16_t* letters)
{
    if (unicode) {
        const char16_t choices[] = {
            character,
            unicode::FoldCase(character),
            unicode::ReverseFoldCase1(character),
            unicode::ReverseFoldCase2(character),
            unicode::ReverseFoldCase3(character),
        };
        return GetCaseIndependentLetters(character, ascii_subject, unicode,
                                         choices, ArrayLength(choices), letters);
    }

    char16_t upper = unicode::ToUpperCase(character);
    unicode::CodepointsWithSameUpperCase others(character);
    char16_t other1 = others.other1();
    char16_t other2 = others.other2();
    char16_t other3 = others.other3();

    // ES 2017 draft 996af87b7072b3c3dd2b1def856c66f456102215 21.2.4.2
    // step 3.g.
    // The standard requires that non-ASCII characters cannot have ASCII
    // character codes in their equivalence class, even though this
    // situation occurs multiple times in the Unicode tables.
    static const unsigned kMaxAsciiCharCode = 127;
    if (upper <= kMaxAsciiCharCode) {
        if (character > kMaxAsciiCharCode) {
            // If Canonicalize(character) == character, all other characters
            // should be ignored.
            return GetCaseIndependentLetters(character, ascii_subject, unicode,
                                             &character, 1, letters);
        }

        if (other1 > kMaxAsciiCharCode)
            other1 = character;
        if (other2 > kMaxAsciiCharCode)
            other2 = character;
        if (other3 > kMaxAsciiCharCode)
            other3 = character;
    }

    const char16_t choices[] = {
        character,
        upper,
        other1,
        other2,
        other3
    };
    return GetCaseIndependentLetters(character, ascii_subject, unicode,
                                     choices, ArrayLength(choices), letters);
}

/*
Generated from following Node.js source:

package.json

```
{
  "private": true,
  "dependencies": {
    "unicode-12.0.0": "^0.7.9"
  }
}
```

generate-unicode-sequence-property-data.js

```
const toHex = (symbol) => {
  return '0x' + symbol.codePointAt(0).toString(16)
                      .toUpperCase().padStart(6, '0');
};

const generateData = (property) => {
  const sequences =
      require(`unicode-12.0.0/Sequence_Property/${ property }/index.js`);
  const id = property.replace(/_/g, '') + 's';
  const buffer = [];
  for (const sequence of sequences) {
    const symbols = [...sequence];
    const codePoints = symbols.map(symbol => toHex(symbol));
    buffer.push('    ' + codePoints.join(', ') + ', 0,');
  }
  const output =
      `const uc32 UnicodePropertySequences::k${ id }[] = {\n` +
      `${ buffer.join('\n') }\n    0  // null-terminating the list\n};\n`;
  return output;
};

const properties = [
  'Emoji_Flag_Sequence',
  'Emoji_Tag_Sequence',
  'Emoji_ZWJ_Sequence',
];

for (const property of properties) {
  console.log(generateData(property));
}
```
*/

const widechar js::irregexp::kEmojiFlagSequences[] = {
    0x01F1E6, 0x01F1E8, 0,
    0x01F1FF, 0x01F1FC, 0,
    0x01F1E6, 0x01F1EA, 0,
    0x01F1E6, 0x01F1EB, 0,
    0x01F1E6, 0x01F1EC, 0,
    0x01F1E6, 0x01F1EE, 0,
    0x01F1E6, 0x01F1F1, 0,
    0x01F1E6, 0x01F1F2, 0,
    0x01F1E6, 0x01F1F4, 0,
    0x01F1E6, 0x01F1F6, 0,
    0x01F1E6, 0x01F1F7, 0,
    0x01F1E6, 0x01F1F8, 0,
    0x01F1E6, 0x01F1F9, 0,
    0x01F1E6, 0x01F1FA, 0,
    0x01F1E6, 0x01F1FC, 0,
    0x01F1E6, 0x01F1FD, 0,
    0x01F1E6, 0x01F1FF, 0,
    0x01F1E7, 0x01F1E6, 0,
    0x01F1E7, 0x01F1E7, 0,
    0x01F1E7, 0x01F1E9, 0,
    0x01F1E7, 0x01F1EA, 0,
    0x01F1E7, 0x01F1EB, 0,
    0x01F1E7, 0x01F1EC, 0,
    0x01F1E7, 0x01F1ED, 0,
    0x01F1E7, 0x01F1EE, 0,
    0x01F1E7, 0x01F1EF, 0,
    0x01F1E7, 0x01F1F1, 0,
    0x01F1E7, 0x01F1F2, 0,
    0x01F1E7, 0x01F1F3, 0,
    0x01F1E7, 0x01F1F4, 0,
    0x01F1E7, 0x01F1F6, 0,
    0x01F1E7, 0x01F1F7, 0,
    0x01F1E7, 0x01F1F8, 0,
    0x01F1E7, 0x01F1F9, 0,
    0x01F1E7, 0x01F1FB, 0,
    0x01F1E7, 0x01F1FC, 0,
    0x01F1E7, 0x01F1FE, 0,
    0x01F1E7, 0x01F1FF, 0,
    0x01F1E8, 0x01F1E6, 0,
    0x01F1E8, 0x01F1E8, 0,
    0x01F1E8, 0x01F1E9, 0,
    0x01F1E8, 0x01F1EB, 0,
    0x01F1E8, 0x01F1EC, 0,
    0x01F1E8, 0x01F1ED, 0,
    0x01F1E8, 0x01F1EE, 0,
    0x01F1E8, 0x01F1F0, 0,
    0x01F1E8, 0x01F1F1, 0,
    0x01F1E8, 0x01F1F2, 0,
    0x01F1E8, 0x01F1F3, 0,
    0x01F1E8, 0x01F1F4, 0,
    0x01F1E8, 0x01F1F5, 0,
    0x01F1E8, 0x01F1F7, 0,
    0x01F1E8, 0x01F1FA, 0,
    0x01F1E8, 0x01F1FB, 0,
    0x01F1E8, 0x01F1FC, 0,
    0x01F1E8, 0x01F1FD, 0,
    0x01F1E8, 0x01F1FE, 0,
    0x01F1E8, 0x01F1FF, 0,
    0x01F1E9, 0x01F1EA, 0,
    0x01F1E9, 0x01F1EC, 0,
    0x01F1E9, 0x01F1EF, 0,
    0x01F1E9, 0x01F1F0, 0,
    0x01F1E9, 0x01F1F2, 0,
    0x01F1E9, 0x01F1F4, 0,
    0x01F1E9, 0x01F1FF, 0,
    0x01F1EA, 0x01F1E6, 0,
    0x01F1EA, 0x01F1E8, 0,
    0x01F1EA, 0x01F1EA, 0,
    0x01F1EA, 0x01F1EC, 0,
    0x01F1EA, 0x01F1ED, 0,
    0x01F1EA, 0x01F1F7, 0,
    0x01F1EA, 0x01F1F8, 0,
    0x01F1EA, 0x01F1F9, 0,
    0x01F1EA, 0x01F1FA, 0,
    0x01F1EB, 0x01F1EE, 0,
    0x01F1EB, 0x01F1EF, 0,
    0x01F1EB, 0x01F1F0, 0,
    0x01F1EB, 0x01F1F2, 0,
    0x01F1EB, 0x01F1F4, 0,
    0x01F1EB, 0x01F1F7, 0,
    0x01F1EC, 0x01F1E6, 0,
    0x01F1EC, 0x01F1E7, 0,
    0x01F1EC, 0x01F1E9, 0,
    0x01F1EC, 0x01F1EA, 0,
    0x01F1EC, 0x01F1EB, 0,
    0x01F1EC, 0x01F1EC, 0,
    0x01F1EC, 0x01F1ED, 0,
    0x01F1EC, 0x01F1EE, 0,
    0x01F1EC, 0x01F1F1, 0,
    0x01F1EC, 0x01F1F2, 0,
    0x01F1EC, 0x01F1F3, 0,
    0x01F1EC, 0x01F1F5, 0,
    0x01F1EC, 0x01F1F6, 0,
    0x01F1EC, 0x01F1F7, 0,
    0x01F1EC, 0x01F1F8, 0,
    0x01F1EC, 0x01F1F9, 0,
    0x01F1EC, 0x01F1FA, 0,
    0x01F1EC, 0x01F1FC, 0,
    0x01F1EC, 0x01F1FE, 0,
    0x01F1ED, 0x01F1F0, 0,
    0x01F1ED, 0x01F1F2, 0,
    0x01F1ED, 0x01F1F3, 0,
    0x01F1ED, 0x01F1F7, 0,
    0x01F1ED, 0x01F1F9, 0,
    0x01F1ED, 0x01F1FA, 0,
    0x01F1EE, 0x01F1E8, 0,
    0x01F1EE, 0x01F1E9, 0,
    0x01F1EE, 0x01F1EA, 0,
    0x01F1EE, 0x01F1F1, 0,
    0x01F1EE, 0x01F1F2, 0,
    0x01F1EE, 0x01F1F3, 0,
    0x01F1EE, 0x01F1F4, 0,
    0x01F1EE, 0x01F1F6, 0,
    0x01F1EE, 0x01F1F7, 0,
    0x01F1EE, 0x01F1F8, 0,
    0x01F1EE, 0x01F1F9, 0,
    0x01F1EF, 0x01F1EA, 0,
    0x01F1EF, 0x01F1F2, 0,
    0x01F1EF, 0x01F1F4, 0,
    0x01F1EF, 0x01F1F5, 0,
    0x01F1F0, 0x01F1EA, 0,
    0x01F1F0, 0x01F1EC, 0,
    0x01F1F0, 0x01F1ED, 0,
    0x01F1F0, 0x01F1EE, 0,
    0x01F1F0, 0x01F1F2, 0,
    0x01F1F0, 0x01F1F3, 0,
    0x01F1F0, 0x01F1F5, 0,
    0x01F1F0, 0x01F1F7, 0,
    0x01F1F0, 0x01F1FC, 0,
    0x01F1E6, 0x01F1E9, 0,
    0x01F1F0, 0x01F1FF, 0,
    0x01F1F1, 0x01F1E6, 0,
    0x01F1F1, 0x01F1E7, 0,
    0x01F1F1, 0x01F1E8, 0,
    0x01F1F1, 0x01F1EE, 0,
    0x01F1F1, 0x01F1F0, 0,
    0x01F1F1, 0x01F1F7, 0,
    0x01F1F1, 0x01F1F8, 0,
    0x01F1F1, 0x01F1F9, 0,
    0x01F1F1, 0x01F1FA, 0,
    0x01F1F1, 0x01F1FB, 0,
    0x01F1F1, 0x01F1FE, 0,
    0x01F1F2, 0x01F1E6, 0,
    0x01F1F2, 0x01F1E8, 0,
    0x01F1F2, 0x01F1E9, 0,
    0x01F1F2, 0x01F1EA, 0,
    0x01F1F2, 0x01F1EB, 0,
    0x01F1F2, 0x01F1EC, 0,
    0x01F1F2, 0x01F1ED, 0,
    0x01F1F2, 0x01F1F0, 0,
    0x01F1F2, 0x01F1F1, 0,
    0x01F1F2, 0x01F1F2, 0,
    0x01F1F2, 0x01F1F3, 0,
    0x01F1F2, 0x01F1F4, 0,
    0x01F1F2, 0x01F1F5, 0,
    0x01F1F2, 0x01F1F6, 0,
    0x01F1F2, 0x01F1F7, 0,
    0x01F1F2, 0x01F1F8, 0,
    0x01F1F2, 0x01F1F9, 0,
    0x01F1F2, 0x01F1FA, 0,
    0x01F1F2, 0x01F1FB, 0,
    0x01F1F2, 0x01F1FC, 0,
    0x01F1F2, 0x01F1FD, 0,
    0x01F1F2, 0x01F1FE, 0,
    0x01F1F2, 0x01F1FF, 0,
    0x01F1F3, 0x01F1E6, 0,
    0x01F1F3, 0x01F1E8, 0,
    0x01F1F3, 0x01F1EA, 0,
    0x01F1F3, 0x01F1EB, 0,
    0x01F1F3, 0x01F1EC, 0,
    0x01F1F3, 0x01F1EE, 0,
    0x01F1F3, 0x01F1F1, 0,
    0x01F1F3, 0x01F1F4, 0,
    0x01F1F3, 0x01F1F5, 0,
    0x01F1F3, 0x01F1F7, 0,
    0x01F1F3, 0x01F1FA, 0,
    0x01F1F3, 0x01F1FF, 0,
    0x01F1F4, 0x01F1F2, 0,
    0x01F1F5, 0x01F1E6, 0,
    0x01F1F5, 0x01F1EA, 0,
    0x01F1F5, 0x01F1EB, 0,
    0x01F1F5, 0x01F1EC, 0,
    0x01F1F5, 0x01F1ED, 0,
    0x01F1F5, 0x01F1F0, 0,
    0x01F1F5, 0x01F1F1, 0,
    0x01F1F5, 0x01F1F2, 0,
    0x01F1F5, 0x01F1F3, 0,
    0x01F1F5, 0x01F1F7, 0,
    0x01F1F5, 0x01F1F8, 0,
    0x01F1F5, 0x01F1F9, 0,
    0x01F1F5, 0x01F1FC, 0,
    0x01F1F5, 0x01F1FE, 0,
    0x01F1F6, 0x01F1E6, 0,
    0x01F1F7, 0x01F1EA, 0,
    0x01F1F7, 0x01F1F4, 0,
    0x01F1F7, 0x01F1F8, 0,
    0x01F1F7, 0x01F1FA, 0,
    0x01F1F7, 0x01F1FC, 0,
    0x01F1F8, 0x01F1E6, 0,
    0x01F1F8, 0x01F1E7, 0,
    0x01F1F8, 0x01F1E8, 0,
    0x01F1F8, 0x01F1E9, 0,
    0x01F1F8, 0x01F1EA, 0,
    0x01F1F8, 0x01F1EC, 0,
    0x01F1F8, 0x01F1ED, 0,
    0x01F1F8, 0x01F1EE, 0,
    0x01F1F8, 0x01F1EF, 0,
    0x01F1F8, 0x01F1F0, 0,
    0x01F1F8, 0x01F1F1, 0,
    0x01F1F8, 0x01F1F2, 0,
    0x01F1F8, 0x01F1F3, 0,
    0x01F1F8, 0x01F1F4, 0,
    0x01F1F8, 0x01F1F7, 0,
    0x01F1F8, 0x01F1F8, 0,
    0x01F1F8, 0x01F1F9, 0,
    0x01F1F8, 0x01F1FB, 0,
    0x01F1F8, 0x01F1FD, 0,
    0x01F1F8, 0x01F1FE, 0,
    0x01F1F8, 0x01F1FF, 0,
    0x01F1F9, 0x01F1E6, 0,
    0x01F1F9, 0x01F1E8, 0,
    0x01F1F9, 0x01F1E9, 0,
    0x01F1F9, 0x01F1EB, 0,
    0x01F1F9, 0x01F1EC, 0,
    0x01F1F9, 0x01F1ED, 0,
    0x01F1F9, 0x01F1EF, 0,
    0x01F1F9, 0x01F1F0, 0,
    0x01F1F9, 0x01F1F1, 0,
    0x01F1F9, 0x01F1F2, 0,
    0x01F1F9, 0x01F1F3, 0,
    0x01F1F9, 0x01F1F4, 0,
    0x01F1F9, 0x01F1F7, 0,
    0x01F1F9, 0x01F1F9, 0,
    0x01F1F9, 0x01F1FB, 0,
    0x01F1F9, 0x01F1FC, 0,
    0x01F1F9, 0x01F1FF, 0,
    0x01F1FA, 0x01F1E6, 0,
    0x01F1FA, 0x01F1EC, 0,
    0x01F1FA, 0x01F1F2, 0,
    0x01F1FA, 0x01F1F3, 0,
    0x01F1FA, 0x01F1F8, 0,
    0x01F1FA, 0x01F1FE, 0,
    0x01F1FA, 0x01F1FF, 0,
    0x01F1FB, 0x01F1E6, 0,
    0x01F1FB, 0x01F1E8, 0,
    0x01F1FB, 0x01F1EA, 0,
    0x01F1FB, 0x01F1EC, 0,
    0x01F1FB, 0x01F1EE, 0,
    0x01F1FB, 0x01F1F3, 0,
    0x01F1FB, 0x01F1FA, 0,
    0x01F1FC, 0x01F1EB, 0,
    0x01F1FC, 0x01F1F8, 0,
    0x01F1FD, 0x01F1F0, 0,
    0x01F1FE, 0x01F1EA, 0,
    0x01F1FE, 0x01F1F9, 0,
    0x01F1FF, 0x01F1E6, 0,
    0x01F1FF, 0x01F1F2, 0,
    0x01F1F0, 0x01F1FE, 0,
    0  // null-terminating the list
};

const widechar js::irregexp::kEmojiTagSequences[] = {
    0x01F3F4, 0x0E0067, 0x0E0062, 0x0E0065, 0x0E006E, 0x0E0067, 0x0E007F, 0,
    0x01F3F4, 0x0E0067, 0x0E0062, 0x0E0073, 0x0E0063, 0x0E0074, 0x0E007F, 0,
    0x01F3F4, 0x0E0067, 0x0E0062, 0x0E0077, 0x0E006C, 0x0E0073, 0x0E007F, 0,
    0  // null-terminating the list
};

const widechar js::irregexp::kEmojiZWJSequences[] = {
    0x01F468, 0x00200D, 0x002764, 0x00FE0F, 0x00200D, 0x01F468, 0,
    0x01F441, 0x00FE0F, 0x00200D, 0x01F5E8, 0x00FE0F, 0,
    0x01F468, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F466, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F467, 0,
    0x01F468, 0x00200D, 0x01F467, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F467, 0x00200D, 0x01F467, 0,
    0x01F468, 0x00200D, 0x01F468, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F468, 0x00200D, 0x01F466, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F468, 0x00200D, 0x01F467, 0,
    0x01F468, 0x00200D, 0x01F468, 0x00200D, 0x01F467, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F468, 0x00200D, 0x01F467, 0x00200D, 0x01F467, 0,
    0x01F468, 0x00200D, 0x01F469, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F469, 0x00200D, 0x01F466, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F469, 0x00200D, 0x01F467, 0,
    0x01F468, 0x00200D, 0x01F469, 0x00200D, 0x01F467, 0x00200D, 0x01F466, 0,
    0x01F468, 0x00200D, 0x01F469, 0x00200D, 0x01F467, 0x00200D, 0x01F467, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FB, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FB, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FC, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FB, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FC, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FD, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FB, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FC, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FD, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FE, 0,
    0x01F469, 0x00200D, 0x002764, 0x00FE0F, 0x00200D, 0x01F468, 0,
    0x01F469, 0x00200D, 0x002764, 0x00FE0F, 0x00200D, 0x01F469, 0,
    0x01F469, 0x00200D, 0x002764, 0x00FE0F, 0x00200D, 0x01F48B, 0x00200D,
        0x01F468, 0,
    0x01F469, 0x00200D, 0x002764, 0x00FE0F, 0x00200D, 0x01F48B, 0x00200D,
        0x01F469, 0,
    0x01F469, 0x00200D, 0x01F466, 0,
    0x01F469, 0x00200D, 0x01F466, 0x00200D, 0x01F466, 0,
    0x01F469, 0x00200D, 0x01F467, 0,
    0x01F469, 0x00200D, 0x01F467, 0x00200D, 0x01F466, 0,
    0x01F469, 0x00200D, 0x01F467, 0x00200D, 0x01F467, 0,
    0x01F469, 0x00200D, 0x01F469, 0x00200D, 0x01F466, 0,
    0x01F469, 0x00200D, 0x01F469, 0x00200D, 0x01F466, 0x00200D, 0x01F466, 0,
    0x01F469, 0x00200D, 0x01F469, 0x00200D, 0x01F467, 0,
    0x01F469, 0x00200D, 0x01F469, 0x00200D, 0x01F467, 0x00200D, 0x01F466, 0,
    0x01F469, 0x00200D, 0x01F469, 0x00200D, 0x01F467, 0x00200D, 0x01F467, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FC, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FD, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FE, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FF, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FB, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FD, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FE, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FF, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FB, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FB, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FC, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FE, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FF, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FB, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FC, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FB, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FC, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FD, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FF, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FB, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FC, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FD, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FB, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FC, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FD, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F468, 0x01F3FE, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FB, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FC, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FD, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F469, 0x01F3FE, 0,
    0x01F9D1, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0,
    0x01F9D1, 0x01F3FB, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FB, 0,
    0x01F9D1, 0x01F3FC, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FB, 0,
    0x01F9D1, 0x01F3FC, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FC, 0,
    0x01F9D1, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FB, 0,
    0x01F9D1, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FC, 0,
    0x01F9D1, 0x01F3FD, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FD, 0,
    0x01F9D1, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FB, 0,
    0x01F9D1, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FC, 0,
    0x01F9D1, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FD, 0,
    0x01F9D1, 0x01F3FE, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FE, 0,
    0x01F9D1, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FB, 0,
    0x01F9D1, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FC, 0,
    0x01F9D1, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FD, 0,
    0x01F9D1, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FE, 0,
    0x01F9D1, 0x01F3FF, 0x00200D, 0x01F91D, 0x00200D, 0x01F9D1, 0x01F3FF, 0,
    0x01F468, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F468, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F468, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F468, 0x00200D, 0x01F33E, 0,
    0x01F468, 0x00200D, 0x01F373, 0,
    0x01F468, 0x00200D, 0x01F393, 0,
    0x01F468, 0x00200D, 0x01F3A4, 0,
    0x01F468, 0x00200D, 0x01F3A8, 0,
    0x01F468, 0x00200D, 0x01F3EB, 0,
    0x01F468, 0x00200D, 0x01F3ED, 0,
    0x01F468, 0x00200D, 0x01F4BB, 0,
    0x01F468, 0x00200D, 0x01F4BC, 0,
    0x01F468, 0x00200D, 0x01F527, 0,
    0x01F468, 0x00200D, 0x01F52C, 0,
    0x01F468, 0x00200D, 0x01F680, 0,
    0x01F468, 0x00200D, 0x01F692, 0,
    0x01F468, 0x00200D, 0x01F9AF, 0,
    0x01F468, 0x00200D, 0x01F9BC, 0,
    0x01F468, 0x00200D, 0x01F9BD, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F33E, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F373, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F393, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F3A4, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F3A8, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F3EB, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F3ED, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F4BB, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F4BC, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F527, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F52C, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F680, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F692, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F9AF, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F9BC, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F9BD, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F33E, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F373, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F393, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F3A4, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F3A8, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F3EB, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F3ED, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F4BB, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F4BC, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F527, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F52C, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F680, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F692, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F9AF, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F9BC, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F9BD, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F33E, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F373, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F393, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F3A4, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F3A8, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F3EB, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F3ED, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F4BB, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F4BC, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F527, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F52C, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F680, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F692, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F9AF, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F9BC, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F9BD, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F33E, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F373, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F393, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F3A4, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F3A8, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F3EB, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F3ED, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F4BB, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F4BC, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F527, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F52C, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F680, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F692, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F9AF, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F9BC, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F9BD, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F33E, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F373, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F393, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F3A4, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F3A8, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F3EB, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F3ED, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F4BB, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F4BC, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F527, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F52C, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F680, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F692, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F9AF, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F9BC, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F9BD, 0,
    0x01F469, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F469, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F469, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F469, 0x00200D, 0x01F33E, 0,
    0x01F469, 0x00200D, 0x01F373, 0,
    0x01F469, 0x00200D, 0x01F393, 0,
    0x01F469, 0x00200D, 0x01F3A4, 0,
    0x01F469, 0x00200D, 0x01F3A8, 0,
    0x01F469, 0x00200D, 0x01F3EB, 0,
    0x01F469, 0x00200D, 0x01F3ED, 0,
    0x01F469, 0x00200D, 0x01F4BB, 0,
    0x01F469, 0x00200D, 0x01F4BC, 0,
    0x01F469, 0x00200D, 0x01F527, 0,
    0x01F469, 0x00200D, 0x01F52C, 0,
    0x01F469, 0x00200D, 0x01F680, 0,
    0x01F469, 0x00200D, 0x01F692, 0,
    0x01F469, 0x00200D, 0x01F9AF, 0,
    0x01F469, 0x00200D, 0x01F9BC, 0,
    0x01F469, 0x00200D, 0x01F9BD, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F33E, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F373, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F393, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F3A4, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F3A8, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F3EB, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F3ED, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F4BB, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F4BC, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F527, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F52C, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F680, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F692, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F9AF, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F9BC, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F9BD, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F33E, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F373, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F393, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F3A4, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F3A8, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F3EB, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F3ED, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F4BB, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F4BC, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F527, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F52C, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F680, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F692, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F9AF, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F9BC, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F9BD, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F33E, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F373, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F393, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F3A4, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F3A8, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F3EB, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F3ED, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F4BB, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F4BC, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F527, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F52C, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F680, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F692, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F9AF, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F9BC, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F9BD, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F33E, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F373, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F393, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F3A4, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F3A8, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F3EB, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F3ED, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F4BB, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F4BC, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F527, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F52C, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F680, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F692, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F9AF, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F9BC, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F9BD, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x002695, 0x00FE0F, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x002696, 0x00FE0F, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x002708, 0x00FE0F, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F33E, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F373, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F393, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F3A4, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F3A8, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F3EB, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F3ED, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F4BB, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F4BC, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F527, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F52C, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F680, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F692, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F9AF, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F9BC, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F9BD, 0,
    0x0026F9, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x0026F9, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x0026F9, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x0026F9, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x0026F9, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x0026F9, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x0026F9, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x0026F9, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x0026F9, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x0026F9, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x0026F9, 0x00FE0F, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x0026F9, 0x00FE0F, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C3, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C3, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C3, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C4, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C4, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3C4, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CA, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CA, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CA, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CB, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CB, 0x00FE0F, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CB, 0x00FE0F, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CC, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F3CC, 0x00FE0F, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F3CC, 0x00FE0F, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F46E, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F46E, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F46E, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F46E, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F46E, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F46E, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F46E, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F46E, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F46E, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F46E, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F46E, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F46E, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F46F, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F46F, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F471, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F471, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F471, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F471, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F471, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F471, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F471, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F471, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F471, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F471, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F471, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F471, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F473, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F473, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F473, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F473, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F473, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F473, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F473, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F473, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F473, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F473, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F473, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F473, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F477, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F477, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F477, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F477, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F477, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F477, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F477, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F477, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F477, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F477, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F477, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F477, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F481, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F481, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F481, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F481, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F481, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F481, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F481, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F481, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F481, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F481, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F481, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F481, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F482, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F482, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F468, 0x00200D, 0x002764, 0x00FE0F, 0x00200D, 0x01F48B, 0x00200D,
        0x01F468, 0,
    0x01F482, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F482, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F482, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F482, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F482, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F482, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F482, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F482, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F482, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F486, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F486, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F486, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F486, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F486, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F486, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F486, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F486, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F486, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F486, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F486, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F486, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F487, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F487, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F487, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F487, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F487, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F487, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F487, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F487, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F487, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F487, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F487, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F487, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F575, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F575, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F575, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F575, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F575, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F575, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F575, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F575, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F575, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F575, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F575, 0x00FE0F, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F575, 0x00FE0F, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F645, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F645, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F645, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F645, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F645, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F645, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F645, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F645, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F645, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F645, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F645, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F645, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F646, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F646, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F646, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F646, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F646, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F646, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F646, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F646, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F646, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F646, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F646, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F646, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F647, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F647, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F647, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F647, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F647, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F647, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F647, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F647, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F647, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F647, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F647, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F647, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64B, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64B, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64B, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64B, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64B, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64B, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64B, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64B, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64B, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64B, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64B, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64B, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64D, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64D, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64D, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64D, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64D, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64D, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64D, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64D, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64D, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64D, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64D, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64D, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64E, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64E, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64E, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64E, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64E, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64E, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64E, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64E, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64E, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64E, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F64E, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F64E, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6A3, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6A3, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6A3, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B4, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B4, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B4, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B5, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B5, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B5, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B6, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B6, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F6B6, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F926, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F926, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F926, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F926, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F926, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F926, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F926, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F926, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F926, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F926, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F926, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F926, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F937, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F937, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F937, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F937, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F937, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F937, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F937, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F937, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F937, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F937, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F937, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F937, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F938, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F938, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F938, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F938, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F938, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F938, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F938, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F938, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F938, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F938, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F938, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F938, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F939, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F939, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F939, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F939, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F939, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F939, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F939, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F939, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F939, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F939, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F939, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F939, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93C, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93C, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93D, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93D, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93D, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93D, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93D, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93D, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93D, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93D, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93D, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93D, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93D, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93D, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93E, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93E, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93E, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93E, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93E, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93E, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93E, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93E, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93E, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93E, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F93E, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F93E, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B8, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B8, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B8, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B9, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B9, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9B9, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CD, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CE, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9CF, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D6, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D6, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D6, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D7, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D7, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D7, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D8, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D8, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D8, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D9, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D9, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9D9, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DA, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DA, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DA, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DB, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DC, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FB, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FC, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FC, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FD, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FD, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DD, 0x01F3FF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DE, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DE, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F9DF, 0x00200D, 0x002640, 0x00FE0F, 0,
    0x01F9DF, 0x00200D, 0x002642, 0x00FE0F, 0,
    0x01F468, 0x00200D, 0x01F9B0, 0,
    0x01F468, 0x00200D, 0x01F9B1, 0,
    0x01F468, 0x00200D, 0x01F9B2, 0,
    0x01F468, 0x00200D, 0x01F9B3, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F9B0, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F9B1, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F9B2, 0,
    0x01F468, 0x01F3FB, 0x00200D, 0x01F9B3, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F9B0, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F9B1, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F9B2, 0,
    0x01F468, 0x01F3FC, 0x00200D, 0x01F9B3, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F9B0, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F9B1, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F9B2, 0,
    0x01F468, 0x01F3FD, 0x00200D, 0x01F9B3, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F9B0, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F9B1, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F9B2, 0,
    0x01F468, 0x01F3FE, 0x00200D, 0x01F9B3, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F9B0, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F9B1, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F9B2, 0,
    0x01F468, 0x01F3FF, 0x00200D, 0x01F9B3, 0,
    0x01F469, 0x00200D, 0x01F9B0, 0,
    0x01F469, 0x00200D, 0x01F9B1, 0,
    0x01F469, 0x00200D, 0x01F9B2, 0,
    0x01F469, 0x00200D, 0x01F9B3, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F9B0, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F9B1, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F9B2, 0,
    0x01F469, 0x01F3FB, 0x00200D, 0x01F9B3, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F9B0, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F9B1, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F9B2, 0,
    0x01F469, 0x01F3FC, 0x00200D, 0x01F9B3, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F9B0, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F9B1, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F9B2, 0,
    0x01F469, 0x01F3FD, 0x00200D, 0x01F9B3, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F9B0, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F9B1, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F9B2, 0,
    0x01F469, 0x01F3FE, 0x00200D, 0x01F9B3, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F9B0, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F9B1, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F9B2, 0,
    0x01F469, 0x01F3FF, 0x00200D, 0x01F9B3, 0,
    0x01F3F3, 0x00FE0F, 0x00200D, 0x01F308, 0,
    0x01F3F4, 0x00200D, 0x002620, 0x00FE0F, 0,
    0x01F415, 0x00200D, 0x01F9BA, 0,
    0x01F482, 0x01F3FB, 0x00200D, 0x002640, 0x00FE0F, 0,
    0  // null-terminating the list
};


