/*
 * Copyright (C) 2020 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(GPU_PROCESS) && ENABLE(WEBGL)

#include "Connection.h"
#include "GPUConnectionToWebProcess.h"
#include "GraphicsContextGLIdentifier.h"
#include "QualifiedRenderingResourceIdentifier.h"
#include "RemoteRenderingBackend.h"
#include "ScopedWebGLRenderingResourcesRequest.h"
#include "SharedVideoFrame.h"
#include "StreamMessageReceiver.h"
#include "StreamServerConnection.h"
#include <WebCore/NotImplemented.h>
#include <WebCore/PixelBuffer.h>
#include <WebCore/ProcessIdentity.h>
#include <wtf/ThreadAssertions.h>
#include <wtf/WeakPtr.h>

#if PLATFORM(COCOA)
#include <WebCore/GraphicsContextGLCocoa.h>
#elif USE(GBM)
#include <WebCore/GraphicsContextGLGBM.h>
#else
#include <WebCore/GraphicsContextGLTextureMapperANGLE.h>
#endif

#if PLATFORM(MAC)
#include <CoreGraphics/CGDisplayConfiguration.h>
#endif

#if USE(GRAPHICS_LAYER_WC)
#include "WCContentBufferIdentifier.h"
#endif

#if ENABLE(VIDEO)
#include "RemoteVideoFrameIdentifier.h"
#include "RemoteVideoFrameProxy.h"
#endif

#if PLATFORM(COCOA)
namespace WTF {
class MachSendRight;
}
#endif

namespace WebKit {
#if ENABLE(VIDEO)
class RemoteVideoFrameObjectHeap;
#endif

IPC::StreamConnectionWorkQueue& remoteGraphicsContextGLStreamWorkQueue();


// GPU process side implementation of that receives messages about GraphicsContextGL calls
// and issues real GraphicsContextGL calls based on the received messages.
// The implementation is largely generated by running Tools/Scripts/generate-gpup-webgl.
class RemoteGraphicsContextGL : private WebCore::GraphicsContextGL::Client, public IPC::StreamMessageReceiver {
    WTF_MAKE_FAST_ALLOCATED;
public:
    static Ref<RemoteGraphicsContextGL> create(GPUConnectionToWebProcess&, WebCore::GraphicsContextGLAttributes&&, GraphicsContextGLIdentifier, RemoteRenderingBackend&, IPC::StreamServerConnection::Handle&&);
    ~RemoteGraphicsContextGL() override;
    void stopListeningForIPC(Ref<RemoteGraphicsContextGL>&& refFromConnection);

    void didReceiveStreamMessage(IPC::StreamServerConnection&, IPC::Decoder&) final;
#if PLATFORM(MAC)
    void displayWasReconfigured();
#endif

protected:
    RemoteGraphicsContextGL(GPUConnectionToWebProcess&, GraphicsContextGLIdentifier, RemoteRenderingBackend&, IPC::StreamServerConnection::Handle&&);
    void initialize(WebCore::GraphicsContextGLAttributes&&);
    IPC::StreamConnectionWorkQueue& workQueue() const { return m_workQueue; }

    void workQueueInitialize(WebCore::GraphicsContextGLAttributes&&);
    virtual void platformWorkQueueInitialize(WebCore::GraphicsContextGLAttributes&&) { };
    void workQueueUninitialize();
    template<typename T>
    bool send(T&& message) const { return m_streamConnection->send(WTFMove(message), m_graphicsContextGLIdentifier); }

    // GraphicsContextGL::Client overrides.
    void didComposite() final;
    void forceContextLost() final;
    void dispatchContextChangedNotification() final;

    // Messages to be received.
    void ensureExtensionEnabled(String&&);
    void markContextChanged();
    void reshape(int32_t width, int32_t height);
#if PLATFORM(COCOA)
    virtual void prepareForDisplay(CompletionHandler<void(WTF::MachSendRight&&)>&&) = 0;
#elif USE(GRAPHICS_LAYER_WC)
    virtual void prepareForDisplay(CompletionHandler<void(std::optional<WCContentBufferIdentifier>)>&&) = 0;
#elif USE(GBM)
    virtual void prepareForDisplay(CompletionHandler<void(WebCore::DMABufObject&&)>&&) = 0;
#else
    void prepareForDisplay(CompletionHandler<void()>&&);
#endif
    void getError(CompletionHandler<void(uint32_t)>&&);
    void synthesizeGLError(uint32_t error);
    void paintRenderingResultsToCanvas(WebCore::RenderingResourceIdentifier, CompletionHandler<void()>&&);
    void paintCompositedResultsToCanvas(WebCore::RenderingResourceIdentifier, CompletionHandler<void()>&&);
#if ENABLE(MEDIA_STREAM)
    void paintCompositedResultsToVideoFrame(CompletionHandler<void(std::optional<WebKit::RemoteVideoFrameProxy::Properties>&&)>&&);
#endif
#if ENABLE(VIDEO) && PLATFORM(COCOA)
    void copyTextureFromVideoFrame(SharedVideoFrame&&, uint32_t texture, uint32_t target, int32_t level, uint32_t internalFormat, uint32_t format, uint32_t type, bool premultiplyAlpha, bool flipY, CompletionHandler<void(bool)>&&);
    void setSharedVideoFrameSemaphore(IPC::Semaphore&&);
    void setSharedVideoFrameMemory(const SharedMemory::Handle&);
#endif
    void simulateEventForTesting(WebCore::GraphicsContextGL::SimulatedEventForTesting);
    void readnPixels0(int32_t x, int32_t y, int32_t width, int32_t height, uint32_t format, uint32_t type, IPC::ArrayReference<uint8_t>&& data, CompletionHandler<void(IPC::ArrayReference<uint8_t>)>&&);
    void readnPixels1(int32_t x, int32_t y, int32_t width, int32_t height, uint32_t format, uint32_t type, uint64_t offset);
    void readnPixels2(int32_t x, int32_t y, int32_t width, int32_t height, uint32_t format, uint32_t type, SharedMemory::Handle, CompletionHandler<void(bool)>&&);
    void multiDrawArraysANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t>&& firstsAndCounts);
    void multiDrawArraysInstancedANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t, int32_t>&& firstsCountsAndInstanceCounts);
    void multiDrawElementsANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t>&& countsAndOffsets, uint32_t type);
    void multiDrawElementsInstancedANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t, int32_t>&& countsOffsetsAndInstanceCounts, uint32_t type);
    void multiDrawArraysInstancedBaseInstanceANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t, int32_t, uint32_t>&& firstsCountsInstanceCountsAndBaseInstances);
    void multiDrawElementsInstancedBaseVertexBaseInstanceANGLE(uint32_t mode, IPC::ArrayReferenceTuple<int32_t, int32_t, int32_t, int32_t, uint32_t>&& countsOffsetsInstanceCountsBaseVerticesAndBaseInstances, uint32_t type);

#include "RemoteGraphicsContextGLFunctionsGenerated.h" // NOLINT

private:
    void paintRenderingResultsToCanvasWithQualifiedIdentifier(QualifiedRenderingResourceIdentifier, CompletionHandler<void()>&&);
    void paintCompositedResultsToCanvasWithQualifiedIdentifier(QualifiedRenderingResourceIdentifier, CompletionHandler<void()>&&);
    void paintPixelBufferToImageBuffer(RefPtr<WebCore::PixelBuffer>&&, QualifiedRenderingResourceIdentifier, CompletionHandler<void()>&&);

protected:
    WeakPtr<GPUConnectionToWebProcess> m_gpuConnectionToWebProcess;
    Ref<IPC::StreamConnectionWorkQueue> m_workQueue;
    RefPtr<IPC::StreamServerConnection> m_streamConnection;
#if PLATFORM(COCOA)
    using GCGLContext = WebCore::GraphicsContextGLCocoa;
#elif USE(GBM)
    using GCGLContext = WebCore::GraphicsContextGLGBM;
#else
    using GCGLContext = WebCore::GraphicsContextGLTextureMapperANGLE;
#endif
    RefPtr<GCGLContext> m_context WTF_GUARDED_BY_CAPABILITY(workQueue());
    GraphicsContextGLIdentifier m_graphicsContextGLIdentifier;
    Ref<RemoteRenderingBackend> m_renderingBackend;
#if ENABLE(VIDEO)
    Ref<RemoteVideoFrameObjectHeap> m_videoFrameObjectHeap;
#if PLATFORM(COCOA)
    SharedVideoFrameReader m_sharedVideoFrameReader;
#endif
#endif
    ScopedWebGLRenderingResourcesRequest m_renderingResourcesRequest;
    WebCore::ProcessIdentifier m_webProcessIdentifier;
    const WebCore::ProcessIdentity m_resourceOwner;
};


} // namespace WebKit

#endif
