#!/usr/bin/env python
#coding:utf-8
# Author:  mozman --<mozman@gmx.at>
# Purpose: transform list parser
# Created: 10.10.2010
# Copyright (C) 2010, Manfred Moitzi
# License: MIT License

import unittest
import time

from svgwrite.data.svgparser import is_valid_pathdata

class TestPathDataParser(unittest.TestCase):
    def test_moveto(self):
        self.assertTrue(is_valid_pathdata("m 0, 0"))
        self.assertTrue(is_valid_pathdata("m 0,0"))
        self.assertTrue(is_valid_pathdata("m 0, 0"))
        self.assertTrue(is_valid_pathdata("m 0 ,0"))
        self.assertTrue(is_valid_pathdata("m 0 , 0"))
        self.assertTrue(is_valid_pathdata("m0,0"))
        self.assertTrue(is_valid_pathdata("M0,0,1,1"))
        self.assertTrue(is_valid_pathdata("M 0,0 1,1"))
        self.assertTrue(is_valid_pathdata("M 0 0 1 1"))
        self.assertTrue(is_valid_pathdata("M 0 0 , 1 1"))

    def test_moveto_errors(self):
        self.assertFalse(is_valid_pathdata("m m"))
        self.assertFalse(is_valid_pathdata("m 1 2 3"))

    def test_lineto(self):
        self.assertTrue(is_valid_pathdata("m0,0L0,0"))
        self.assertTrue(is_valid_pathdata("m0,0 L0,0"))
        self.assertTrue(is_valid_pathdata("m0,0 L 0.5 0.5"))
        self.assertTrue(is_valid_pathdata("m0,0 L 0.5 0.5 99,88"))
        self.assertTrue(is_valid_pathdata("m0,0l0,0"))
        self.assertTrue(is_valid_pathdata("m0,0 l0,0"))
        self.assertTrue(is_valid_pathdata("m0,0 l 0.5 0.5"))
        self.assertTrue(is_valid_pathdata("m0,0 l 0.5 0.5 99,88"))

    def test_lineto_errors(self):
        # error: 0 lineto args
        self.assertFalse(is_valid_pathdata("m0,0 l"))
        # error: 1 lineto args
        self.assertFalse(is_valid_pathdata("m0,0 l0"))
        # error: 3 lineto args
        self.assertFalse(is_valid_pathdata("m0,0 l0,0,0"))

    def test_horizontal_lineto(self):
        self.assertTrue(is_valid_pathdata("m0,0h1"))
        self.assertTrue(is_valid_pathdata("m0,0H1,2"))
        self.assertTrue(is_valid_pathdata("m0,0h1,2,3"))
        self.assertTrue(is_valid_pathdata("m0,0H1,2 3,4 5 6 7"))
        self.assertTrue(is_valid_pathdata("m0,0h1."))
        self.assertTrue(is_valid_pathdata("m0,0H1.,2."))
        self.assertTrue(is_valid_pathdata("m0,0h.1,.2,.3"))
        self.assertTrue(is_valid_pathdata("m0,0H1.,.2 3.,.4 .5 .6 7."))

    def test_horizontal_lineto_errors(self):
        # error: 0 horizontal-lineto args
        self.assertFalse(is_valid_pathdata("m0,0 h"))

    def test_vertical_lineto(self):
        self.assertTrue(is_valid_pathdata("m0,0v1"))
        self.assertTrue(is_valid_pathdata("m0,0V1,2"))
        self.assertTrue(is_valid_pathdata("m0,0v1,2,3"))
        self.assertTrue(is_valid_pathdata("m0,0V1,2 3,4 5 6 7"))
        self.assertTrue(is_valid_pathdata("m0,0v1."))
        self.assertTrue(is_valid_pathdata("m0,0V1.,2."))
        self.assertTrue(is_valid_pathdata("m0,0v.1,.2,.3"))
        self.assertTrue(is_valid_pathdata("m0,0V1.,.2 3.,.4 .5 .6 7."))

    def test_vertical_lineto_errors(self):
        # error: 0 vertical-lineto args
        self.assertFalse(is_valid_pathdata("m0,0 v"))

    def test_closepath(self):
        self.assertTrue(is_valid_pathdata("m0,0h1z"))
        self.assertTrue(is_valid_pathdata("m0,0h1Z"))
        self.assertTrue(is_valid_pathdata("m0,0v1 z"))
        self.assertTrue(is_valid_pathdata("m0,0v1 Z"))

    def test_closepath_errors(self):
        # error: 1 closepath arg
        self.assertFalse(is_valid_pathdata("m0,0 z 1"))

    def test_curveto(self):
        self.assertTrue(is_valid_pathdata("m0,0 c 1 2 3 4 5 6"))
        self.assertTrue(is_valid_pathdata("m0,0 C 1 2 3 4 5 6, 7 8 9 10 11 12"))

    def test_curveto_errors(self):
        self.assertFalse(is_valid_pathdata("m0,0 c 1 2 3 4 5"))
        self.assertFalse(is_valid_pathdata("m0,0 C 1 2 3 4 5 6, 7 8 9 10 11"))
        self.assertFalse(is_valid_pathdata("m0,0 C 1 2 3 4 5 6, 7 8 9 10"))
        self.assertFalse(is_valid_pathdata("m0,0 C 1 2 3 4 5 6, 7 8 9"))
        self.assertFalse(is_valid_pathdata("m0,0 C 1 2 3 4 5 6, 7 8"))
        self.assertFalse(is_valid_pathdata("m0,0 C 1 2 3 4 5 6, 7"))

    def test_smooth_curveto(self):
        self.assertTrue(is_valid_pathdata("m0,0 s 3 4 5 6"))
        self.assertTrue(is_valid_pathdata("m0,0 S 3 4 5 6, 9 10 11 12"))

    def test_smooth_curveto_errors(self):
        self.assertFalse(is_valid_pathdata("m0,0 s 3 4 5"))
        self.assertFalse(is_valid_pathdata("m0,0 S 1 2 3 4, 7 8 9"))
        self.assertFalse(is_valid_pathdata("m0,0 S 1 2 3 4, 7 8"))
        self.assertFalse(is_valid_pathdata("m0,0 S 1 2 3 4, 7"))

    def test_quadratic_bezier_curveto(self):
        self.assertTrue(is_valid_pathdata("m0,0 q 1 2 3 4"))
        self.assertTrue(is_valid_pathdata("m0,0 Q 1 2 3 4, 5 6 7 8"))

    def test_quadratic_bezier_curveto_errors(self):
        self.assertFalse(is_valid_pathdata("m0,0 q 1 2 3"))
        self.assertFalse(is_valid_pathdata("m0,0 q 1 2 3 4, 5 6 7"))
        self.assertFalse(is_valid_pathdata("m0,0 q 1 2 3 4, 5 6"))
        self.assertFalse(is_valid_pathdata("m0,0 q 1 2 3 4, 5"))

    def test_smooth_quadratic_bezier_curveto(self):
        self.assertTrue(is_valid_pathdata("m0,0 t 1 2"))
        self.assertTrue(is_valid_pathdata("m0,0 T 1 2 3 4 5 6"))

    def test_smooth_quadratic_bezier_curveto_errors(self):
        self.assertFalse(is_valid_pathdata("m0,0 t 1"))
        self.assertFalse(is_valid_pathdata("m0,0 t 1 2 3"))

    def test_elliptical_arc(self):
        self.assertTrue(is_valid_pathdata("m0,0 a 1 1 0 0 0 10 10"))
        self.assertTrue(is_valid_pathdata("m0,0 a 1 1 0 1 1 10 10"))
        self.assertTrue(is_valid_pathdata("m0,0 A 1 1 0 0 0 10 10, 1 1 0 0 0 10 10"))

    def test_elliptical_arc_errors(self):
        self.assertFalse(is_valid_pathdata("m0,0 a 1 1 45 0 0 10"))
        self.assertFalse(is_valid_pathdata("m0,0 a 1 1 45 0 0"))
        self.assertFalse(is_valid_pathdata("m0,0 a 1 1 45 0"))
        self.assertFalse(is_valid_pathdata("m0,0 a 1 1 45"))
        self.assertFalse(is_valid_pathdata("m0,0 a 1 1"))
        self.assertFalse(is_valid_pathdata("m0,0 a 1"))
        self.assertFalse(is_valid_pathdata("m0,0 a"))
        # flag errors flags != [01]
        self.assertFalse(is_valid_pathdata("m0,0 a 1 1 45 2 2 10 10"))

    def test_performance(self):
        start = time.time()
        self.assertTrue(is_valid_pathdata("M 0.85 9885.75 H 11.35 A 0.10,0.10,0,0,0,11.45 9885.65 L 11.45 9882.75 C 11.45 9882.65 11.35 9882.75 11.45 9882.75 H 17.25 C 17.35 9882.75 17.25 9882.65 17.25 9882.75 V 9885.65 A 0.10,0.10,0,0,0,17.35 9885.75 L 27.85 9885.75 M 29.38 9885.75 V 9869.55 M 29.38 9866.55 V 9837.15 M 29.38 9834.15 V 9804.75 M 29.38 9801.75 V 9785.55 M 30.88 9788.55 V 9817.95 M 30.88 9820.95 V 9850.35 M 30.88 9853.35 V 9882.75 M 32.38 9885.75 V 9869.55 M 32.38 9866.55 V 9837.15 M 32.38 9834.15 V 9804.75 M 32.38 9801.75 V 9785.55 M 33.88 9788.55 V 9817.95 M 33.88 9820.95 V 9850.35 M 33.88 9853.35 V 9882.75 M 35.38 9885.75 V 9869.55 M 35.38 9866.55 V 9837.15 M 35.38 9834.15 V 9804.75 M 35.38 9801.75 V 9785.55 M 36.88 9788.55 V 9817.95 M 36.88 9820.95 V 9850.35 M 36.88 9853.35 V 9882.75 M 38.38 9885.75 V 9869.55 M 38.38 9866.55 V 9837.15 M 38.38 9834.15 V 9804.75 M 38.38 9801.75 V 9785.55 M 39.88 9788.55 V 9817.95 M 39.88 9820.95 V 9850.35 M 39.88 9853.35 V 9882.75 M 41.38 9885.75 V 9869.55 M 41.38 9866.55 V 9837.15 M 41.38 9834.15 V 9804.75 M 41.38 9801.75 V 9785.55 M 42.88 9788.55 V 9817.95 M 42.88 9820.95 V 9850.35 M 42.88 9853.35 V 9882.75 M 44.38 9885.75 V 9869.55 M 44.38 9866.55 V 9837.15 M 44.38 9834.15 V 9804.75 M 44.38 9801.75 V 9785.55 M 45.88 9788.55 V 9817.95 M 45.88 9820.95 V 9850.35 M 45.88 9853.35 V 9882.75 M 47.38 9885.75 V 9869.55 M 47.38 9866.55 V 9837.15 M 47.38 9834.15 V 9804.75 M 47.38 9801.75 V 9785.55 M 48.88 9788.55 V 9817.95 M 48.88 9820.95 V 9850.35 M 48.88 9853.35 V 9882.75 M 50.38 9885.75 V 9869.55 M 50.38 9866.55 V 9837.15 M 50.38 9834.15 V 9804.75 M 50.38 9801.75 V 9785.55 M 51.88 9788.55 V 9817.95 M 51.88 9820.95 V 9850.35 M 51.88 9853.35 V 9882.75 M 53.38 9885.75 V 9869.55 M 53.38 9866.55 V 9837.15 M 53.38 9834.15 V 9804.75 M 53.38 9801.75 V 9785.55 M 54.88 9788.55 V 9817.95 M 54.88 9820.95 V 9850.35 M 54.88 9853.35 V 9882.75 M 56.38 9885.75 V 9869.55 M 56.38 9866.55 V 9837.15 M 56.38 9834.15 V 9804.75 M 56.38 9801.75 V 9785.55 M 27.85 9885.75 H 57.92 L 63.92 9885.75 A 0.10,0.10,0,0,0,64.02 9885.65 L 64.02 9882.75 C 64.02 9882.65 63.92 9882.75 64.02 9882.75 H 69.82 C 69.92 9882.75 69.82 9882.65 69.82 9882.75 V 9885.65 A 0.10,0.10,0,0,0,69.92 9885.75 L 75.92 9885.75 A 0.10,0.10,0,0,0,76.02 9885.65 L 76.02 9882.75 C 76.02 9882.65 75.92 9882.75 76.02 9882.75 H 81.82 C 81.92 9882.75 81.82 9882.65 81.82 9882.75 V 9885.65 A 0.10,0.10,0,0,0,81.92 9885.75 L 87.92 9885.75 A 0.10,0.10,0,0,0,88.02 9885.65 L 88.02 9882.75 C 88.02 9882.65 87.92 9882.75 88.02 9882.75 H 93.82 C 93.92 9882.75 93.82 9882.65 93.82 9882.75 V 9885.65 A 0.10,0.10,0,0,0,93.92 9885.75 L 99.92 9885.75 A 0.10,0.10,0,0,0,100.02 9885.65 L 100.02 9882.75 C 100.02 9882.65 99.92 9882.75 100.02 9882.75 H 105.82 C 105.92 9882.75 105.82 9882.65 105.82 9882.75 V 9885.65 A 0.10,0.10,0,0,0,105.92 9885.75 L 111.92 9885.75 M 113.45 9885.75 V 9869.55 M 113.45 9866.55 V 9837.15 M 113.45 9834.15 V 9804.75 M 113.45 9801.75 V 9785.55 M 114.95 9788.55 V 9817.95 M 114.95 9820.95 V 9850.35 M 114.95 9853.35 V 9882.75 M 116.45 9885.75 V 9869.55 M 116.45 9866.55 V 9837.15 M 116.45 9834.15 V 9804.75 M 116.45 9801.75 V 9785.55 M 117.95 9788.55 V 9817.95 M 117.95 9820.95 V 9850.35 M 117.95 9853.35 V 9882.75 M 119.45 9885.75 V 9869.55 M 119.45 9866.55 V 9837.15 M 119.45 9834.15 V 9804.75 M 119.45 9801.75 V 9785.55 M 120.95 9788.55 V 9817.95 M 120.95 9820.95 V 9850.35 M 120.95 9853.35 V 9882.75 M 122.45 9885.75 V 9869.55 M 122.45 9866.55 V 9837.15 M 122.45 9834.15 V 9804.75 M 122.45 9801.75 V 9785.55 M 123.95 9788.55 V 9817.95 M 123.95 9820.95 V 9850.35 M 123.95 9853.35 V 9882.75 M 125.45 9885.75 V 9869.55 M 125.45 9866.55 V 9837.15 M 125.45 9834.15 V 9804.75 M 125.45 9801.75 V 9785.55 M 126.95 9788.55 V 9817.95 M 126.95 9820.95 V 9850.35 M 126.95 9853.35 V 9882.75 M 128.45 9885.75 V 9869.55 M 128.45 9866.55 V 9837.15 M 128.45 9834.15 V 9804.75 M 128.45 9801.75 V 9785.55 M 129.95 9788.55 V 9817.95 M 129.95 9820.95 V 9850.35 M 129.95 9853.35 V 9882.75 M 131.45 9885.75 V 9869.55 M 131.45 9866.55 V 9837.15 M 131.45 9834.15 V 9804.75 M 131.45 9801.75 V 9785.55 M 132.95 9788.55 V 9817.95 M 132.95 9820.95 V 9850.35 M 132.95 9853.35 V 9882.75 M 134.45 9885.75 V 9869.55 M 134.45 9866.55 V 9837.15 M 134.45 9834.15 V 9804.75 M 134.45 9801.75 V 9785.55 M 135.95 9788.55 V 9817.95 M 135.95 9820.95 V 9850.35 M 135.95 9853.35 V 9882.75 M 137.45 9885.75 V 9869.55 M 137.45 9866.55 V 9837.15 M 137.45 9834.15 V 9804.75 M 137.45 9801.75 V 9785.55 M 138.95 9788.55 V 9817.95 M 138.95 9820.95 V 9850.35 M 138.95 9853.35 V 9882.75 M 140.45 9885.75 V 9869.55 M 140.45 9866.55 V 9837.15 M 140.45 9834.15 V 9804.75 M 140.45 9801.75 V 9785.55 M 111.92 9885.75 H 141.99 L 147.99 9885.75 A 0.10,0.10,0,0,0,148.09 9885.65 L 148.09 9882.75 C 148.09 9882.65 147.99 9882.75 148.09 9882.75 H 153.89 C 153.99 9882.75 153.89 9882.65 153.89 9882.75 V 9885.65 A 0.10,0.10,0,0,0,153.99 9885.75 L 159.99 9885.75 A 0.10,0.10,0,0,0,160.09 9885.65 L 160.09 9882.75 C 160.09 9882.65 159.99 9882.75 160.09 9882.75 H 165.89 C 165.99 9882.75 165.89 9882.65 165.89 9882.75 V 9885.65 A 0.10,0.10,0,0,0,165.99 9885.75 L 171.99 9885.75 A 0.10,0.10,0,0,0,172.09 9885.65 L 172.09 9882.75 C 172.09 9882.65 171.99 9882.75 172.09 9882.75 H 177.89 C 177.99 9882.75 177.89 9882.65 177.89 9882.75 V 9885.65 A 0.10,0.10,0,0,0,177.99 9885.75 L 183.99 9885.75 A 0.10,0.10,0,0,0,184.09 9885.65 L 184.09 9882.75 C 184.09 9882.65 183.99 9882.75 184.09 9882.75 H 189.89 C 189.99 9882.75 189.89 9882.65 189.89 9882.75 V 9885.65 A 0.10,0.10,0,0,0,189.99 9885.75 L 195.99 9885.75 M 197.52 9885.75 V 9869.55 M 197.52 9866.55 V 9837.15 M 197.52 9834.15 V 9804.75 M 197.52 9801.75 V 9785.55 M 199.02 9788.55 V 9817.95 M 199.02 9820.95 V 9850.35 M 199.02 9853.35 V 9882.75 M 200.52 9885.75 V 9869.55 M 200.52 9866.55 V 9837.15 M 200.52 9834.15 V 9804.75 M 200.52 9801.75 V 9785.55 M 202.02 9788.55 V 9817.95 M 202.02 9820.95 V 9850.35 M 202.02 9853.35 V 9882.75 M 203.52 9885.75 V 9869.55 M 203.52 9866.55 V 9837.15 M 203.52 9834.15 V 9804.75 M 203.52 9801.75 V 9785.55 M 205.02 9788.55 V 9817.95 M 205.02 9820.95 V 9850.35 M 205.02 9853.35 V 9882.75 M 206.52 9885.75 V 9869.55 M 206.52 9866.55 V 9837.15 M 206.52 9834.15 V 9804.75 M 206.52 9801.75 V 9785.55 M 208.02 9788.55 V 9817.95 M 208.02 9820.95 V 9850.35 M 208.02 9853.35 V 9882.75 M 209.52 9885.75 V 9869.55 M 209.52 9866.55 V 9837.15 M 209.52 9834.15 V 9804.75 M 209.52 9801.75 V 9785.55 M 211.02 9788.55 V 9817.95 M 211.02 9820.95 V 9850.35 M 211.02 9853.35 V 9882.75 M 212.52 9885.75 V 9869.55 M 212.52 9866.55 V 9837.15 M 212.52 9834.15 V 9804.75 M 212.52 9801.75 V 9785.55 M 214.02 9788.55 V 9817.95 M 214.02 9820.95 V 9850.35 M 214.02 9853.35 V 9882.75 M 215.52 9885.75 V 9869.55 M 215.52 9866.55 V 9837.15 M 215.52 9834.15 V 9804.75 M 215.52 9801.75 V 9785.55 M 217.02 9788.55 V 9817.95 M 217.02 9820.95 V 9850.35 M 217.02 9853.35 V 9882.75 M 218.52 9885.75 V 9869.55 M 218.52 9866.55 V 9837.15 M 218.52 9834.15 V 9804.75 M 218.52 9801.75 V 9785.55 M 220.02 9788.55 V 9817.95 M 220.02 9820.95 V 9850.35 M 220.02 9853.35 V 9882.75 M 221.52 9885.75 V 9869.55 M 221.52 9866.55 V 9837.15 M 221.52 9834.15 V 9804.75 M 221.52 9801.75 V 9785.55 M 223.02 9788.55 V 9817.95 M 223.02 9820.95 V 9850.35 M 223.02 9853.35 V 9882.75 M 224.52 9885.75 V 9869.55 M 224.52 9866.55 V 9837.15 M 224.52 9834.15 V 9804.75 M 224.52 9801.75 V 9785.55 M 195.99 9885.75 H 226.06 L 232.06 9885.75 A 0.10,0.10,0,0,0,232.16 9885.65 L 232.16 9882.75 C 232.16 9882.65 232.06 9882.75 232.16 9882.75 H 237.96 C 238.06 9882.75 237.96 9882.65 237.96 9882.75 V 9885.65 A 0.10,0.10,0,0,0,238.06 9885.75 L 244.06 9885.75 A 0.10,0.10,0,0,0,244.16 9885.65 L 244.16 9882.75 C 244.16 9882.65 244.06 9882.75 244.16 9882.75 H 249.96 C 250.06 9882.75 249.96 9882.65 249.96 9882.75 V 9885.65 A 0.10,0.10,0,0,0,250.06 9885.75 L 256.06 9885.75 A 0.10,0.10,0,0,0,256.16 9885.65 L 256.16 9882.75 C 256.16 9882.65 256.06 9882.75 256.16 9882.75 H 261.96 C 262.06 9882.75 261.96 9882.65 261.96 9882.75 V 9885.65 A 0.10,0.10,0,0,0,262.06 9885.75 L 268.06 9885.75 A 0.10,0.10,0,0,0,268.16 9885.65 L 268.16 9882.75 C 268.16 9882.65 268.06 9882.75 268.16 9882.75 H 273.96 C 274.06 9882.75 273.96 9882.65 273.96 9882.75 V 9885.65 A 0.10,0.10,0,0,0,274.06 9885.75 L 280.06 9885.75 M 281.59 9885.75 V 9869.55 M 281.59 9866.55 V 9837.15 M 281.59 9834.15 V 9804.75 M 281.59 9801.75 V 9785.55 M 283.09 9788.55 V 9817.95 M 283.09 9820.95 V 9850.35 M 283.09 9853.35 V 9882.75 M 284.59 9885.75 V 9869.55 M 284.59 9866.55 V 9837.15 M 284.59 9834.15 V 9804.75 M 284.59 9801.75 V 9785.55 M 286.09 9788.55 V 9817.95 M 286.09 9820.95 V 9850.35 M 286.09 9853.35 V 9882.75 M 287.59 9885.75 V 9869.55 M 287.59 9866.55 V 9837.15 M 287.59 9834.15 V 9804.75 M 287.59 9801.75 V 9785.55 M 289.09 9788.55 V 9817.95 M 289.09 9820.95 V 9850.35 M 289.09 9853.35 V 9882.75 M 290.59 9885.75 V 9869.55 M 290.59 9866.55 V 9837.15 M 290.59 9834.15 V 9804.75 M 290.59 9801.75 V 9785.55 M 292.09 9788.55 V 9817.95 M 292.09 9820.95 V 9850.35 M 292.09 9853.35 V 9882.75 M 293.59 9885.75 V 9869.55 M 293.59 9866.55 V 9837.15 M 293.59 9834.15 V 9804.75 M 293.59 9801.75 V 9785.55 M 295.09 9788.55 V 9817.95 M 295.09 9820.95 V 9850.35 M 295.09 9853.35 V 9882.75 M 296.59 9885.75 V 9869.55 M 296.59 9866.55 V 9837.15 M 296.59 9834.15 V 9804.75 M 296.59 9801.75 V 9785.55 M 298.09 9788.55 V 9817.95 M 298.09 9820.95 V 9850.35 M 298.09 9853.35 V 9882.75 M 299.59 9885.75 V 9869.55 M 299.59 9866.55 V 9837.15 M 299.59 9834.15 V 9804.75 M 299.59 9801.75 V 9785.55 M 301.09 9788.55 V 9817.95 M 301.09 9820.95 V 9850.35 M 301.09 9853.35 V 9882.75 M 302.59 9885.75 V 9869.55 M 302.59 9866.55 V 9837.15 M 302.59 9834.15 V 9804.75 M 302.59 9801.75 V 9785.55 M 304.09 9788.55 V 9817.95 M 304.09 9820.95 V 9850.35 M 304.09 9853.35 V 9882.75 M 305.59 9885.75 V 9869.55 M 305.59 9866.55 V 9837.15 M 305.59 9834.15 V 9804.75 M 305.59 9801.75 V 9785.55 M 307.09 9788.55 V 9817.95 M 307.09 9820.95 V 9850.35 M 307.09 9853.35 V 9882.75 M 308.59 9885.75 V 9869.55 M 308.59 9866.55 V 9837.15 M 308.59 9834.15 V 9804.75 M 308.59 9801.75 V 9785.55 M 280.06 9885.75 H 310.13 L 320.63 9885.75 A 0.10,0.10,0,0,0,320.73 9885.65 L 320.73 9882.75 C 320.73 9882.65 320.63 9882.75 320.73 9882.75 H 326.53 C 326.63 9882.75 326.53 9882.65 326.53 9882.75 V 9885.65 A 0.10,0.10,0,0,0,326.63 9885.75 L 337.13 9885.75 A 0.10,0.10,0,0,0,337.23 9885.65 L 337.23 9882.65 V 9875.12 A 0.50,0.50,0,0,1,338.11 9874.80 L 340.49 9877.63 A 0.70,0.70,0,0,0,341.73 9877.18 L 341.73 9866.12 A 0.70,0.70,0,0,0,340.49 9865.67 L 338.11 9868.50 A 0.50,0.50,0,0,1,337.23 9868.18 L 337.23 9857.12 A 0.50,0.50,0,0,1,338.11 9856.80 L 340.49 9859.63 A 0.70,0.70,0,0,0,341.73 9859.18 L 341.73 9848.12 A 0.70,0.70,0,0,0,340.49 9847.67 L 338.11 9850.50 A 0.50,0.50,0,0,1,337.23 9850.18 L 337.23 9839.12 A 0.50,0.50,0,0,1,338.11 9838.80 L 340.49 9841.63 A 0.70,0.70,0,0,0,341.73 9841.18 L 341.73 9830.12 A 0.70,0.70,0,0,0,340.49 9829.67 L 338.11 9832.50 A 0.50,0.50,0,0,1,337.23 9832.18 L 337.23 9821.12 A 0.50,0.50,0,0,1,338.11 9820.80 L 340.49 9823.63 A 0.70,0.70,0,0,0,341.73 9823.18 L 341.73 9812.12 A 0.70,0.70,0,0,0,340.49 9811.67 L 338.11 9814.50 A 0.50,0.50,0,0,1,337.23 9814.18 L 337.23 9803.12 A 0.50,0.50,0,0,1,338.11 9802.80 L 340.49 9805.63 A 0.70,0.70,0,0,0,341.73 9805.18 L 341.73 9794.12 A 0.70,0.70,0,0,0,340.49 9793.67 L 338.11 9796.50 A 0.50,0.50,0,0,1,337.23 9796.18 L 337.23 9788.65 V 9785.65 A 0.10,0.10,0,0,0,337.13 9785.55 L 326.63 9785.55 A 0.10,0.10,0,0,0,326.53 9785.65 L 326.53 9788.55 C 326.53 9788.65 326.63 9788.55 326.53 9788.55 H 320.73 C 320.63 9788.55 320.73 9788.65 320.73 9788.55 V 9785.65 A 0.10,0.10,0,0,0,320.63 9785.55 L 310.13 9785.55 H 280.06 L 274.06 9785.55 A 0.10,0.10,0,0,0,273.96 9785.65 L 273.96 9788.55 C 273.96 9788.65 274.06 9788.55 273.96 9788.55 H 268.16 C 268.06 9788.55 268.16 9788.65 268.16 9788.55 V 9785.65 A 0.10,0.10,0,0,0,268.06 9785.55 L 262.06 9785.55 A 0.10,0.10,0,0,0,261.96 9785.65 L 261.96 9788.55 C 261.96 9788.65 262.06 9788.55 261.96 9788.55 H 256.16 C 256.06 9788.55 256.16 9788.65 256.16 9788.55 V 9785.65 A 0.10,0.10,0,0,0,256.06 9785.55 L 250.06 9785.55 A 0.10,0.10,0,0,0,249.96 9785.65 L 249.96 9788.55 C 249.96 9788.65 250.06 9788.55 249.96 9788.55 H 244.16 C 244.06 9788.55 244.16 9788.65 244.16 9788.55 V 9785.65 A 0.10,0.10,0,0,0,244.06 9785.55 L 238.06 9785.55 A 0.10,0.10,0,0,0,237.96 9785.65 L 237.96 9788.55 C 237.96 9788.65 238.06 9788.55 237.96 9788.55 H 232.16 C 232.06 9788.55 232.16 9788.65 232.16 9788.55 V 9785.65 A 0.10,0.10,0,0,0,232.06 9785.55 L 226.06 9785.55 H 195.99 L 189.99 9785.55 A 0.10,0.10,0,0,0,189.89 9785.65 L 189.89 9788.55 C 189.89 9788.65 189.99 9788.55 189.89 9788.55 H 184.09 C 183.99 9788.55 184.09 9788.65 184.09 9788.55 V 9785.65 A 0.10,0.10,0,0,0,183.99 9785.55 L 177.99 9785.55 A 0.10,0.10,0,0,0,177.89 9785.65 L 177.89 9788.55 C 177.89 9788.65 177.99 9788.55 177.89 9788.55 H 172.09 C 171.99 9788.55 172.09 9788.65 172.09 9788.55 V 9785.65 A 0.10,0.10,0,0,0,171.99 9785.55 L 165.99 9785.55 A 0.10,0.10,0,0,0,165.89 9785.65 L 165.89 9788.55 C 165.89 9788.65 165.99 9788.55 165.89 9788.55 H 160.09 C 159.99 9788.55 160.09 9788.65 160.09 9788.55 V 9785.65 A 0.10,0.10,0,0,0,159.99 9785.55 L 153.99 9785.55 A 0.10,0.10,0,0,0,153.89 9785.65 L 153.89 9788.55 C 153.89 9788.65 153.99 9788.55 153.89 9788.55 H 148.09 C 147.99 9788.55 148.09 9788.65 148.09 9788.55 V 9785.65 A 0.10,0.10,0,0,0,147.99 9785.55 L 141.99 9785.55 H 111.92 L 105.92 9785.55 A 0.10,0.10,0,0,0,105.82 9785.65 L 105.82 9788.55 C 105.82 9788.65 105.92 9788.55 105.82 9788.55 H 100.02 C 99.92 9788.55 100.02 9788.65 100.02 9788.55 V 9785.65 A 0.10,0.10,0,0,0,99.92 9785.55 L 93.92 9785.55 A 0.10,0.10,0,0,0,93.82 9785.65 L 93.82 9788.55 C 93.82 9788.65 93.92 9788.55 93.82 9788.55 H 88.02 C 87.92 9788.55 88.02 9788.65 88.02 9788.55 V 9785.65 A 0.10,0.10,0,0,0,87.92 9785.55 L 81.92 9785.55 A 0.10,0.10,0,0,0,81.82 9785.65 L 81.82 9788.55 C 81.82 9788.65 81.92 9788.55 81.82 9788.55 H 76.02 C 75.92 9788.55 76.02 9788.65 76.02 9788.55 V 9785.65 A 0.10,0.10,0,0,0,75.92 9785.55 L 69.92 9785.55 A 0.10,0.10,0,0,0,69.82 9785.65 L 69.82 9788.55 C 69.82 9788.65 69.92 9788.55 69.82 9788.55 H 64.02 C 63.92 9788.55 64.02 9788.65 64.02 9788.55 V 9785.65 A 0.10,0.10,0,0,0,63.92 9785.55 L 57.92 9785.55 H 27.85 L 17.35 9785.55 A 0.10,0.10,0,0,0,17.25 9785.65 L 17.25 9788.55 C 17.25 9788.65 17.35 9788.55 17.25 9788.55 H 11.45 C 11.35 9788.55 11.45 9788.65 11.45 9788.55 V 9785.65 A 0.10,0.10,0,0,0,11.35 9785.55 L 0.85 9785.55 A 0.10,0.10,0,0,0,0.75 9785.65 L 0.75 9788.65 V 9796.18 A 0.70,0.70,0,0,0,1.99 9796.63 L 4.37 9793.80 A 0.50,0.50,0,0,1,5.25 9794.12 L 5.25 9805.18 A 0.50,0.50,0,0,1,4.37 9805.50 L 1.99 9802.67 A 0.70,0.70,0,0,0,0.75 9803.12 L 0.75 9814.18 A 0.70,0.70,0,0,0,1.99 9814.63 L 4.37 9811.80 A 0.50,0.50,0,0,1,5.25 9812.12 L 5.25 9823.18 A 0.50,0.50,0,0,1,4.37 9823.50 L 1.99 9820.67 A 0.70,0.70,0,0,0,0.75 9821.12 L 0.75 9832.18 A 0.70,0.70,0,0,0,1.99 9832.63 L 4.37 9829.80 A 0.50,0.50,0,0,1,5.25 9830.12 L 5.25 9841.18 A 0.50,0.50,0,0,1,4.37 9841.50 L 1.99 9838.67 A 0.70,0.70,0,0,0,0.75 9839.12 L 0.75 9850.18 A 0.70,0.70,0,0,0,1.99 9850.63 L 4.37 9847.80 A 0.50,0.50,0,0,1,5.25 9848.12 L 5.25 9859.18 A 0.50,0.50,0,0,1,4.37 9859.50 L 1.99 9856.67 A 0.70,0.70,0,0,0,0.75 9857.12 L 0.75 9868.18 A 0.70,0.70,0,0,0,1.99 9868.63 L 4.37 9865.80 A 0.50,0.50,0,0,1,5.25 9866.12 L 5.25 9877.18 A 0.50,0.50,0,0,1,4.37 9877.50 L 1.99 9874.67 A 0.70,0.70,0,0,0,0.75 9875.12 L 0.75 9882.65 V 9885.65 A 0.10,0.10,0,0,0,0.85 9885.75"))
        self.assertLess(time.time()-start, 0.1, "Parsing path string is too slow")
if __name__ == '__main__':
    unittest.main()
