use ui;
use layout;
use graphics;
use core:geometry;
use core:io;
use progvis:program;
use progvis:data;
use progvis:view;

class MainPainter extends Painter {
	// The workspace we want to draw.
	private view:Workspace workspace;

	// World of the program's state. Also has a reference to the actual program.
	private World? world;

	// Memory model to check for.
	private MemoryChecks memChecks;

	// Problem panel, if any.
	private ProblemPanel? panel;

	// Message to show, if any. Shown "on top" of the panel.
	private ProblemPanel? message;

	// Lock for traversals.
	private core:sync:Lock traverseLock;

	// Stop threads when an error occurred?
	private Bool stopOnError;

	// Left mouse button pressed?
	private Bool mouseDown;

	// Track memory leaks?
	private Bool checkLeaks;

	// Did we already find a memory leak?
	private Bool leakFound;

	// Current drawable object in focus. Only useful when 'mousedown' is pressed.
	private view:Drawable? mouseFocus;

	// Location mouse was pressed at.
	private Point mouseOffset;

	// Offset for the workspace.
	private Point offset;

	// Scale factor for the workspace. Applied last, meaning that "offset" and "mouseOffset" are also scaled.
	private Float scale;

	// Behavior to use in the painter.
	private Behavior? behavior;

	// Config from the behavior, if any.
	private BehaviorConfig? config;

	init() {
		init() { scale = 1.0; stopOnError = true; }
		bgColor = Color(0.8, 0.8, 0.8);
		workspace.enableStep = &this.allowStep;
	}

	// Check if we should allow stepping?
	private Bool allowStep() {
		if (config)
			config.allowStep;
		else
			true;
	}

	// Set the memory model.
	void checkMemory(MemoryChecks mode) {
		memChecks = mode;
		if (world) {
			world.trackMemory = mode;
			// Might deadlock otherwise (if we get an error message).
			spawn update(true);
		}
	}

	// Set leak checking.
	void checkLeaks(Bool check) {
		checkLeaks = check;
		leakFound = false;
	}

	// Set zoom. Also pans to the center to make sure nothing gets hidden too badly.
	void setZoom(Float zoom) {
		if (scale != zoom) {
			offset = Point();
			scale = zoom;
		}
	}

	// Stop program execution when an error occurs? This is good as it halts the program state a bit
	// earlier when an error occurrs. However, if the goal is to use the visualization alongside a
	// model-checker where determinism is important, this feature has to be turned off.
	void stopOnError(Bool value) {
		stopOnError = value;
	}

	// Override default size of threads.
	void defaultThreadSize(Size size) {
		workspace.threadSize = size;
	}

	// Override the output size.
	void defaultOutputWidth(Float width) {
		workspace.outputWidth = width;
	}

	// Set current offset. Useful for scripting the behavior of the visualization. The offset is the
	// screen-coordinates where coordinate 0, 0 appears.
	void setOffset(Point off) {
		offset = -off;
		repaint();
	}

	// Do we have an open "program" instance?
	Bool hasProgram() {
		world.any;
	}

	// Update the behavior without re-loading a program.
	void updateBehavior(Behavior b) {
		behavior = b;
		config = b.config;
		panel = b.panel;

		if (config) {
			workspace.clearNext();
			if (m = config.mark)
				markAction(m);
		}
	}

	// Show a message.
	void showMessage(Str message) {
		this.message = ProblemPanel(message);
		repaint();
	}

	// Hide the message.
	void hideMessage() {
		this.message = null;
		repaint();
	}

	// Open a program.
	void open(Url[] files, Behavior b) {
		var loaded = Program:load(files);

		// Terminate the previous program, if any. We do that here in case the load failed.
		cleanup();

		var w = World(loaded);
		world = w;
		updateBehavior(b);
		w.trackDeadlock = true;
		w.trackMemory = memChecks;
		loaded.onChange = &this.update(Bool);
		loaded.onError = &this.onProgramError;
		loaded.onAdvance = &this.onProgramAdvance;
		update(true);
		loaded.spawn(loaded.main);
	}

	// Try to find the loaded source code for the program.
	Str? sourceFor(Url url) {
		if (world)
			if (text = world.program.source(url))
				return text.text;
		null;
	}

	// Are all threads currently waiting?
	Bool threadsWaiting() {
		unless (world)
			return false;

		world.program.threadsWaiting();
	}

	// Apply an action. These are the same that we produce to the Behavior.
	// Will only apply the action if all threads are paused, returns false if this is not the case.
	Bool applyAction(Str action) {
		applyAction(action, true);
	}

	// Apply an action. These are the same that we produce to the Behavior.
	// Will only apply the action if all threads are paused, returns false if this is not the case.
	// "animate" indicates if barrier steps shall be animated.
	Bool applyAction(Str action, Bool animate) {
		unless (world)
			return false;

		unless (world.program.threadsWaiting)
			return false;

		Bool barrier = false;
		if (action.endsWith("*")) {
			barrier = true;
			action = action.cut(action.begin, action.findLast("*".begin.v));
		}

		if (action == "s") {
			// Spawn new thread.
			spawnThread();
		} else if (action.isNat) {
			// Step a thread.
			Nat thread = action.toNat;
			for (v in world.program.threads) {
				if (v.threadId == thread) {
					if (barrier & v.barriersAvailable) {
						if (animate)
							v.resumeBarrier(100 ms);
						else
							v.resumeBarrier();
					} else {
						v.resume();
					}
				}
			}
		}

		true;
	}

	private void onProgramError(ThreadError error) {
		unless (behavior)
			return;

		if (stopOnError)
			stopThreads();
		behavior.onProgramError(error);
	}

	private void onProgramAdvance(Nat thread, Bool barrier) {
		unless (behavior)
			return;

		var action = thread.toS;
		if (barrier)
			action += "*";

		behavior.onUserAction(action);
	}

	// Clean up the running program (i.e. make sure all threads are terminated).
	void cleanup() {
		// Don't call any behavior during termination.
		behavior = null;
		config = null;
		panel = null;

		if (world) {
			// This calls terminate if necessary.
			world.clear();
		}

		workspace.clear();
		offset = Point();
		repaint();
	}

	// Restart the program.
	void restart() {
		if (world) {
			// Don't call behavior during restart.
			var tmp = behavior;
			behavior = null;
			config = null;

			world.clear();
			workspace.clear();

			behavior = tmp;
			if (behavior) {
				config = behavior.config;
				behavior.onRestart();
			}
			world.program.spawn(world.program.main);
		}
	}

	// Spawn another thread.
	void spawnThread() {
		if (world) {
			world.program.spawn(world.program.main);
			if (behavior)
				behavior.onUserAction("s");
		}
	}

	// Stop all running threads.
	void stopThreads() {
		if (world) {
			for (k, v in world.program.threads) {
				v.pause();
			}
		}
	}

	// Start all threads with the specified speed
	void resumeThreads(Duration speed) {
		if (world) {
			for (k, v in world.program.threads) {
				v.resume(speed);
			}
		}
	}

	// Render everything.
	Bool render(Size me, Graphics g) : override {
		g.transform(translate(-offset) * scale(scale));

		Rect viewport(offset, me / scale);
		workspace.render(viewport, g);

		if (message)
			message.render(viewport, g);
		else if (panel)
			panel.render(viewport, g);

		false;
	}

	// Update the visualization of the data. The parameter indicates whether or not the thread hit a
	// "true" breakpoint or not, and thus whether or not we need to update the screen.
	void update(Bool breakpoint) {
		ProgramError? error;

		if (world) {
			Bool waitForThreads = false;
			{
				core:sync:Lock:Guard z(traverseLock);

				// If some thread is running code, we need to wait in this function until the thread
				// paused. If we update the world while another thread is running, we will get an
				// incomplete view of the world and possibly report sporadic errors. We still want
				// to wait here until the other thread eventually reaches a pause, otherwise we
				// might miss some errors. We do not, however, need to continue validating if we
				// detect that some other thread is currently running, as that thread will call
				// 'update' whenever it reaches a location where it pauses.
				if (!world.program.threadsPaused) {
					waitForThreads = true;
				} else {
					error = world.update();

					if (o = world.program.getOutput())
						workspace.outputText(o);

					workspace.update(world);

					if (checkLeaks & !leakFound) {
						if (workspace.anyUnreachable()) {
							error = MemoryLeakError();
							leakFound = true;
						}
					}
				}
			}

			// We need to stop threads before the repaint to catch them before an implicit yield in
			// repaint. We also do it before any other things that could yield.
			if (error.any & stopOnError)
				stopThreads();

			if (waitForThreads) {
				// We don't want to do this while holding the lock. Otherwise the "stop threads"
				// notification will be delayed for this thread.
				while (!world.program.threadsPaused)
					yield();
				return;
			}
		}

		if (config) {
			workspace.clearNext();
			if (m = config.mark)
				markAction(m);
		}

		if (breakpoint | error.any)
			repaint();

		// Show the error after the repaint, so that the error is visible in the background.
		if (error) {
			if (behavior) {
				behavior.onProgramError(error);
			}
		}
	}

	// Mark a particular action as being the next action.
	private void markAction(Str action) {
		Bool barrier = false;
		if (action.endsWith("*")) {
			barrier = true;
			action = action.cut(action.begin, action.findLast("*".begin.v));
		}

		if (action.isNat())
			workspace.markNext(action.toNat, barrier);
	}

	// Get a summary of the program's current state.
	Str programSummary(Bool findUnreachable) {
		if (world) {
			core:sync:Lock:Guard z(traverseLock);
			world.summary(findUnreachable);
		} else {
			"";
		}
	}

	// Get active threads in the program.
	Nat[] programActiveThreads() {
		Nat[] result;
		if (world) {
			core:sync:Lock:Guard z(traverseLock);
			for (k, v in world.program.threads) {
				if (!v.sleeping)
					result << v.threadId;
			}
		}
		result;
	}

	// Called when the mouse is moved.
	void mouseMoved(Point pt) {
		if (!mouseDown)
			return;

		pt /= scale;
		pt += offset;

		if (mouseFocus) {
			// Call the drawable.
			if (mouseFocus.mouseMoved(pt, mouseOffset))
				repaint();
		} else {
			// Update our transform.
			offset -= pt - mouseOffset;
			repaint();
		}
	}

	// Called when a mouse button is clicked.
	void mouseClicked(Point pt, Bool down, MouseButton button) {
		pt /= scale;
		pt += offset;

		if (button == MouseButton:left) {
			if (down) {
				if (message.any) {
				} else if (panel) {
					if (panel.mouseClicked(pt))
						return;
				}

				mouseDown = true;
				mouseOffset = pt;
				mouseFocus = workspace.findAtActivate(pt);

				if (mouseFocus) {
					mouseFocus.mouseClicked(pt, down);
				}

				repaint();
			} else {
				mouseOffset = pt;
				mouseLeave();
			}
		}
	}

	// Called when the mouse leaves the window.
	void mouseLeave() {
		if (mouseFocus) {
			// Tell it we released the cursor. We might want to differentiate between the two cases later on...
			if (mouseFocus.mouseClicked(mouseOffset, false))
				repaint();
		}

		mouseDown = false;
		mouseFocus = null;
	}
}
