/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2021      Kunwar Maheep Singh <kunwar.maheep@students.iiit.ac.in>
 */

#define SIMDE_TEST_X86_AVX512_INSN round

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/round.h>

static int
test_simde_x_mm512_round_ps (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float32 a[16];
    const simde_float32 nearest[16];
    const simde_float32 neginf[16];
    const simde_float32 posinf[16];
    const simde_float32 truncate[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   377.51), SIMDE_FLOAT32_C(   690.81), SIMDE_FLOAT32_C(  -550.14), SIMDE_FLOAT32_C(  -261.39),
        SIMDE_FLOAT32_C(   204.11), SIMDE_FLOAT32_C(  -834.33), SIMDE_FLOAT32_C(   407.87), SIMDE_FLOAT32_C(   788.71),
        SIMDE_FLOAT32_C(   723.60), SIMDE_FLOAT32_C(   -70.60), SIMDE_FLOAT32_C(  -397.88), SIMDE_FLOAT32_C(   152.13),
        SIMDE_FLOAT32_C(   335.14), SIMDE_FLOAT32_C(  -552.54), SIMDE_FLOAT32_C(   256.00), SIMDE_FLOAT32_C(  -297.89) },
      { SIMDE_FLOAT32_C(   378.00), SIMDE_FLOAT32_C(   691.00), SIMDE_FLOAT32_C(  -550.00), SIMDE_FLOAT32_C(  -261.00),
        SIMDE_FLOAT32_C(   204.00), SIMDE_FLOAT32_C(  -834.00), SIMDE_FLOAT32_C(   408.00), SIMDE_FLOAT32_C(   789.00),
        SIMDE_FLOAT32_C(   724.00), SIMDE_FLOAT32_C(   -71.00), SIMDE_FLOAT32_C(  -398.00), SIMDE_FLOAT32_C(   152.00),
        SIMDE_FLOAT32_C(   335.00), SIMDE_FLOAT32_C(  -553.00), SIMDE_FLOAT32_C(   256.00), SIMDE_FLOAT32_C(  -298.00) },
      { SIMDE_FLOAT32_C(   377.00), SIMDE_FLOAT32_C(   690.00), SIMDE_FLOAT32_C(  -551.00), SIMDE_FLOAT32_C(  -262.00),
        SIMDE_FLOAT32_C(   204.00), SIMDE_FLOAT32_C(  -835.00), SIMDE_FLOAT32_C(   407.00), SIMDE_FLOAT32_C(   788.00),
        SIMDE_FLOAT32_C(   723.00), SIMDE_FLOAT32_C(   -71.00), SIMDE_FLOAT32_C(  -398.00), SIMDE_FLOAT32_C(   152.00),
        SIMDE_FLOAT32_C(   335.00), SIMDE_FLOAT32_C(  -553.00), SIMDE_FLOAT32_C(   256.00), SIMDE_FLOAT32_C(  -298.00) },
      { SIMDE_FLOAT32_C(   378.00), SIMDE_FLOAT32_C(   691.00), SIMDE_FLOAT32_C(  -550.00), SIMDE_FLOAT32_C(  -261.00),
        SIMDE_FLOAT32_C(   205.00), SIMDE_FLOAT32_C(  -834.00), SIMDE_FLOAT32_C(   408.00), SIMDE_FLOAT32_C(   789.00),
        SIMDE_FLOAT32_C(   724.00), SIMDE_FLOAT32_C(   -70.00), SIMDE_FLOAT32_C(  -397.00), SIMDE_FLOAT32_C(   153.00),
        SIMDE_FLOAT32_C(   336.00), SIMDE_FLOAT32_C(  -552.00), SIMDE_FLOAT32_C(   256.00), SIMDE_FLOAT32_C(  -297.00) },
      { SIMDE_FLOAT32_C(   377.00), SIMDE_FLOAT32_C(   690.00), SIMDE_FLOAT32_C(  -550.00), SIMDE_FLOAT32_C(  -261.00),
        SIMDE_FLOAT32_C(   204.00), SIMDE_FLOAT32_C(  -834.00), SIMDE_FLOAT32_C(   407.00), SIMDE_FLOAT32_C(   788.00),
        SIMDE_FLOAT32_C(   723.00), SIMDE_FLOAT32_C(   -70.00), SIMDE_FLOAT32_C(  -397.00), SIMDE_FLOAT32_C(   152.00),
        SIMDE_FLOAT32_C(   335.00), SIMDE_FLOAT32_C(  -552.00), SIMDE_FLOAT32_C(   256.00), SIMDE_FLOAT32_C(  -297.00) } },
    { { SIMDE_FLOAT32_C(   290.21), SIMDE_FLOAT32_C(  -984.08), SIMDE_FLOAT32_C(  -458.53), SIMDE_FLOAT32_C(   425.84),
        SIMDE_FLOAT32_C(   -89.58), SIMDE_FLOAT32_C(  -131.32), SIMDE_FLOAT32_C(  -292.49), SIMDE_FLOAT32_C(  -134.59),
        SIMDE_FLOAT32_C(  -987.42), SIMDE_FLOAT32_C(   218.74), SIMDE_FLOAT32_C(   100.57), SIMDE_FLOAT32_C(    32.56),
        SIMDE_FLOAT32_C(   971.80), SIMDE_FLOAT32_C(  -180.15), SIMDE_FLOAT32_C(  -673.85), SIMDE_FLOAT32_C(   349.31) },
      { SIMDE_FLOAT32_C(   290.00), SIMDE_FLOAT32_C(  -984.00), SIMDE_FLOAT32_C(  -459.00), SIMDE_FLOAT32_C(   426.00),
        SIMDE_FLOAT32_C(   -90.00), SIMDE_FLOAT32_C(  -131.00), SIMDE_FLOAT32_C(  -292.00), SIMDE_FLOAT32_C(  -135.00),
        SIMDE_FLOAT32_C(  -987.00), SIMDE_FLOAT32_C(   219.00), SIMDE_FLOAT32_C(   101.00), SIMDE_FLOAT32_C(    33.00),
        SIMDE_FLOAT32_C(   972.00), SIMDE_FLOAT32_C(  -180.00), SIMDE_FLOAT32_C(  -674.00), SIMDE_FLOAT32_C(   349.00) },
      { SIMDE_FLOAT32_C(   290.00), SIMDE_FLOAT32_C(  -985.00), SIMDE_FLOAT32_C(  -459.00), SIMDE_FLOAT32_C(   425.00),
        SIMDE_FLOAT32_C(   -90.00), SIMDE_FLOAT32_C(  -132.00), SIMDE_FLOAT32_C(  -293.00), SIMDE_FLOAT32_C(  -135.00),
        SIMDE_FLOAT32_C(  -988.00), SIMDE_FLOAT32_C(   218.00), SIMDE_FLOAT32_C(   100.00), SIMDE_FLOAT32_C(    32.00),
        SIMDE_FLOAT32_C(   971.00), SIMDE_FLOAT32_C(  -181.00), SIMDE_FLOAT32_C(  -674.00), SIMDE_FLOAT32_C(   349.00) },
      { SIMDE_FLOAT32_C(   291.00), SIMDE_FLOAT32_C(  -984.00), SIMDE_FLOAT32_C(  -458.00), SIMDE_FLOAT32_C(   426.00),
        SIMDE_FLOAT32_C(   -89.00), SIMDE_FLOAT32_C(  -131.00), SIMDE_FLOAT32_C(  -292.00), SIMDE_FLOAT32_C(  -134.00),
        SIMDE_FLOAT32_C(  -987.00), SIMDE_FLOAT32_C(   219.00), SIMDE_FLOAT32_C(   101.00), SIMDE_FLOAT32_C(    33.00),
        SIMDE_FLOAT32_C(   972.00), SIMDE_FLOAT32_C(  -180.00), SIMDE_FLOAT32_C(  -673.00), SIMDE_FLOAT32_C(   350.00) },
      { SIMDE_FLOAT32_C(   290.00), SIMDE_FLOAT32_C(  -984.00), SIMDE_FLOAT32_C(  -458.00), SIMDE_FLOAT32_C(   425.00),
        SIMDE_FLOAT32_C(   -89.00), SIMDE_FLOAT32_C(  -131.00), SIMDE_FLOAT32_C(  -292.00), SIMDE_FLOAT32_C(  -134.00),
        SIMDE_FLOAT32_C(  -987.00), SIMDE_FLOAT32_C(   218.00), SIMDE_FLOAT32_C(   100.00), SIMDE_FLOAT32_C(    32.00),
        SIMDE_FLOAT32_C(   971.00), SIMDE_FLOAT32_C(  -180.00), SIMDE_FLOAT32_C(  -673.00), SIMDE_FLOAT32_C(   349.00) } },
    { { SIMDE_FLOAT32_C(  -489.34), SIMDE_FLOAT32_C(  -223.98), SIMDE_FLOAT32_C(  -912.08), SIMDE_FLOAT32_C(   714.77),
        SIMDE_FLOAT32_C(   -58.31), SIMDE_FLOAT32_C(   495.80), SIMDE_FLOAT32_C(   503.48), SIMDE_FLOAT32_C(  -334.71),
        SIMDE_FLOAT32_C(  -574.81), SIMDE_FLOAT32_C(  -894.40), SIMDE_FLOAT32_C(   817.43), SIMDE_FLOAT32_C(   760.34),
        SIMDE_FLOAT32_C(  -446.94), SIMDE_FLOAT32_C(    73.42), SIMDE_FLOAT32_C(  -537.55), SIMDE_FLOAT32_C(   843.27) },
      { SIMDE_FLOAT32_C(  -489.00), SIMDE_FLOAT32_C(  -224.00), SIMDE_FLOAT32_C(  -912.00), SIMDE_FLOAT32_C(   715.00),
        SIMDE_FLOAT32_C(   -58.00), SIMDE_FLOAT32_C(   496.00), SIMDE_FLOAT32_C(   503.00), SIMDE_FLOAT32_C(  -335.00),
        SIMDE_FLOAT32_C(  -575.00), SIMDE_FLOAT32_C(  -894.00), SIMDE_FLOAT32_C(   817.00), SIMDE_FLOAT32_C(   760.00),
        SIMDE_FLOAT32_C(  -447.00), SIMDE_FLOAT32_C(    73.00), SIMDE_FLOAT32_C(  -538.00), SIMDE_FLOAT32_C(   843.00) },
      { SIMDE_FLOAT32_C(  -490.00), SIMDE_FLOAT32_C(  -224.00), SIMDE_FLOAT32_C(  -913.00), SIMDE_FLOAT32_C(   714.00),
        SIMDE_FLOAT32_C(   -59.00), SIMDE_FLOAT32_C(   495.00), SIMDE_FLOAT32_C(   503.00), SIMDE_FLOAT32_C(  -335.00),
        SIMDE_FLOAT32_C(  -575.00), SIMDE_FLOAT32_C(  -895.00), SIMDE_FLOAT32_C(   817.00), SIMDE_FLOAT32_C(   760.00),
        SIMDE_FLOAT32_C(  -447.00), SIMDE_FLOAT32_C(    73.00), SIMDE_FLOAT32_C(  -538.00), SIMDE_FLOAT32_C(   843.00) },
      { SIMDE_FLOAT32_C(  -489.00), SIMDE_FLOAT32_C(  -223.00), SIMDE_FLOAT32_C(  -912.00), SIMDE_FLOAT32_C(   715.00),
        SIMDE_FLOAT32_C(   -58.00), SIMDE_FLOAT32_C(   496.00), SIMDE_FLOAT32_C(   504.00), SIMDE_FLOAT32_C(  -334.00),
        SIMDE_FLOAT32_C(  -574.00), SIMDE_FLOAT32_C(  -894.00), SIMDE_FLOAT32_C(   818.00), SIMDE_FLOAT32_C(   761.00),
        SIMDE_FLOAT32_C(  -446.00), SIMDE_FLOAT32_C(    74.00), SIMDE_FLOAT32_C(  -537.00), SIMDE_FLOAT32_C(   844.00) },
      { SIMDE_FLOAT32_C(  -489.00), SIMDE_FLOAT32_C(  -223.00), SIMDE_FLOAT32_C(  -912.00), SIMDE_FLOAT32_C(   714.00),
        SIMDE_FLOAT32_C(   -58.00), SIMDE_FLOAT32_C(   495.00), SIMDE_FLOAT32_C(   503.00), SIMDE_FLOAT32_C(  -334.00),
        SIMDE_FLOAT32_C(  -574.00), SIMDE_FLOAT32_C(  -894.00), SIMDE_FLOAT32_C(   817.00), SIMDE_FLOAT32_C(   760.00),
        SIMDE_FLOAT32_C(  -446.00), SIMDE_FLOAT32_C(    73.00), SIMDE_FLOAT32_C(  -537.00), SIMDE_FLOAT32_C(   843.00) } },
    { { SIMDE_FLOAT32_C(    89.34), SIMDE_FLOAT32_C(     3.91), SIMDE_FLOAT32_C(   269.11), SIMDE_FLOAT32_C(   999.76),
        SIMDE_FLOAT32_C(   872.60), SIMDE_FLOAT32_C(   976.62), SIMDE_FLOAT32_C(  -134.83), SIMDE_FLOAT32_C(   885.17),
        SIMDE_FLOAT32_C(   195.36), SIMDE_FLOAT32_C(   965.74), SIMDE_FLOAT32_C(   -82.27), SIMDE_FLOAT32_C(   167.16),
        SIMDE_FLOAT32_C(  -214.42), SIMDE_FLOAT32_C(   243.88), SIMDE_FLOAT32_C(  -483.54), SIMDE_FLOAT32_C(   296.24) },
      { SIMDE_FLOAT32_C(    89.00), SIMDE_FLOAT32_C(     4.00), SIMDE_FLOAT32_C(   269.00), SIMDE_FLOAT32_C(  1000.00),
        SIMDE_FLOAT32_C(   873.00), SIMDE_FLOAT32_C(   977.00), SIMDE_FLOAT32_C(  -135.00), SIMDE_FLOAT32_C(   885.00),
        SIMDE_FLOAT32_C(   195.00), SIMDE_FLOAT32_C(   966.00), SIMDE_FLOAT32_C(   -82.00), SIMDE_FLOAT32_C(   167.00),
        SIMDE_FLOAT32_C(  -214.00), SIMDE_FLOAT32_C(   244.00), SIMDE_FLOAT32_C(  -484.00), SIMDE_FLOAT32_C(   296.00) },
      { SIMDE_FLOAT32_C(    89.00), SIMDE_FLOAT32_C(     3.00), SIMDE_FLOAT32_C(   269.00), SIMDE_FLOAT32_C(   999.00),
        SIMDE_FLOAT32_C(   872.00), SIMDE_FLOAT32_C(   976.00), SIMDE_FLOAT32_C(  -135.00), SIMDE_FLOAT32_C(   885.00),
        SIMDE_FLOAT32_C(   195.00), SIMDE_FLOAT32_C(   965.00), SIMDE_FLOAT32_C(   -83.00), SIMDE_FLOAT32_C(   167.00),
        SIMDE_FLOAT32_C(  -215.00), SIMDE_FLOAT32_C(   243.00), SIMDE_FLOAT32_C(  -484.00), SIMDE_FLOAT32_C(   296.00) },
      { SIMDE_FLOAT32_C(    90.00), SIMDE_FLOAT32_C(     4.00), SIMDE_FLOAT32_C(   270.00), SIMDE_FLOAT32_C(  1000.00),
        SIMDE_FLOAT32_C(   873.00), SIMDE_FLOAT32_C(   977.00), SIMDE_FLOAT32_C(  -134.00), SIMDE_FLOAT32_C(   886.00),
        SIMDE_FLOAT32_C(   196.00), SIMDE_FLOAT32_C(   966.00), SIMDE_FLOAT32_C(   -82.00), SIMDE_FLOAT32_C(   168.00),
        SIMDE_FLOAT32_C(  -214.00), SIMDE_FLOAT32_C(   244.00), SIMDE_FLOAT32_C(  -483.00), SIMDE_FLOAT32_C(   297.00) },
      { SIMDE_FLOAT32_C(    89.00), SIMDE_FLOAT32_C(     3.00), SIMDE_FLOAT32_C(   269.00), SIMDE_FLOAT32_C(   999.00),
        SIMDE_FLOAT32_C(   872.00), SIMDE_FLOAT32_C(   976.00), SIMDE_FLOAT32_C(  -134.00), SIMDE_FLOAT32_C(   885.00),
        SIMDE_FLOAT32_C(   195.00), SIMDE_FLOAT32_C(   965.00), SIMDE_FLOAT32_C(   -82.00), SIMDE_FLOAT32_C(   167.00),
        SIMDE_FLOAT32_C(  -214.00), SIMDE_FLOAT32_C(   243.00), SIMDE_FLOAT32_C(  -483.00), SIMDE_FLOAT32_C(   296.00) } },
    { { SIMDE_FLOAT32_C(  -980.10), SIMDE_FLOAT32_C(  -395.61), SIMDE_FLOAT32_C(    11.02), SIMDE_FLOAT32_C(   -38.41),
        SIMDE_FLOAT32_C(  -899.82), SIMDE_FLOAT32_C(  -485.50), SIMDE_FLOAT32_C(   626.88), SIMDE_FLOAT32_C(  -474.62),
        SIMDE_FLOAT32_C(  -379.91), SIMDE_FLOAT32_C(   444.30), SIMDE_FLOAT32_C(  -714.28), SIMDE_FLOAT32_C(   173.15),
        SIMDE_FLOAT32_C(  -482.27), SIMDE_FLOAT32_C(  -251.84), SIMDE_FLOAT32_C(    16.42), SIMDE_FLOAT32_C(   607.07) },
      { SIMDE_FLOAT32_C(  -980.00), SIMDE_FLOAT32_C(  -396.00), SIMDE_FLOAT32_C(    11.00), SIMDE_FLOAT32_C(   -38.00),
        SIMDE_FLOAT32_C(  -900.00), SIMDE_FLOAT32_C(  -486.00), SIMDE_FLOAT32_C(   627.00), SIMDE_FLOAT32_C(  -475.00),
        SIMDE_FLOAT32_C(  -380.00), SIMDE_FLOAT32_C(   444.00), SIMDE_FLOAT32_C(  -714.00), SIMDE_FLOAT32_C(   173.00),
        SIMDE_FLOAT32_C(  -482.00), SIMDE_FLOAT32_C(  -252.00), SIMDE_FLOAT32_C(    16.00), SIMDE_FLOAT32_C(   607.00) },
      { SIMDE_FLOAT32_C(  -981.00), SIMDE_FLOAT32_C(  -396.00), SIMDE_FLOAT32_C(    11.00), SIMDE_FLOAT32_C(   -39.00),
        SIMDE_FLOAT32_C(  -900.00), SIMDE_FLOAT32_C(  -486.00), SIMDE_FLOAT32_C(   626.00), SIMDE_FLOAT32_C(  -475.00),
        SIMDE_FLOAT32_C(  -380.00), SIMDE_FLOAT32_C(   444.00), SIMDE_FLOAT32_C(  -715.00), SIMDE_FLOAT32_C(   173.00),
        SIMDE_FLOAT32_C(  -483.00), SIMDE_FLOAT32_C(  -252.00), SIMDE_FLOAT32_C(    16.00), SIMDE_FLOAT32_C(   607.00) },
      { SIMDE_FLOAT32_C(  -980.00), SIMDE_FLOAT32_C(  -395.00), SIMDE_FLOAT32_C(    12.00), SIMDE_FLOAT32_C(   -38.00),
        SIMDE_FLOAT32_C(  -899.00), SIMDE_FLOAT32_C(  -485.00), SIMDE_FLOAT32_C(   627.00), SIMDE_FLOAT32_C(  -474.00),
        SIMDE_FLOAT32_C(  -379.00), SIMDE_FLOAT32_C(   445.00), SIMDE_FLOAT32_C(  -714.00), SIMDE_FLOAT32_C(   174.00),
        SIMDE_FLOAT32_C(  -482.00), SIMDE_FLOAT32_C(  -251.00), SIMDE_FLOAT32_C(    17.00), SIMDE_FLOAT32_C(   608.00) },
      { SIMDE_FLOAT32_C(  -980.00), SIMDE_FLOAT32_C(  -395.00), SIMDE_FLOAT32_C(    11.00), SIMDE_FLOAT32_C(   -38.00),
        SIMDE_FLOAT32_C(  -899.00), SIMDE_FLOAT32_C(  -485.00), SIMDE_FLOAT32_C(   626.00), SIMDE_FLOAT32_C(  -474.00),
        SIMDE_FLOAT32_C(  -379.00), SIMDE_FLOAT32_C(   444.00), SIMDE_FLOAT32_C(  -714.00), SIMDE_FLOAT32_C(   173.00),
        SIMDE_FLOAT32_C(  -482.00), SIMDE_FLOAT32_C(  -251.00), SIMDE_FLOAT32_C(    16.00), SIMDE_FLOAT32_C(   607.00) } },
    { { SIMDE_FLOAT32_C(   752.07), SIMDE_FLOAT32_C(  -714.48), SIMDE_FLOAT32_C(   606.83), SIMDE_FLOAT32_C(   624.67),
        SIMDE_FLOAT32_C(  -737.86), SIMDE_FLOAT32_C(  -528.01), SIMDE_FLOAT32_C(   509.84), SIMDE_FLOAT32_C(   457.49),
        SIMDE_FLOAT32_C(  -562.27), SIMDE_FLOAT32_C(  -572.43), SIMDE_FLOAT32_C(  -375.35), SIMDE_FLOAT32_C(   223.31),
        SIMDE_FLOAT32_C(   671.45), SIMDE_FLOAT32_C(   141.12), SIMDE_FLOAT32_C(  -480.44), SIMDE_FLOAT32_C(   691.34) },
      { SIMDE_FLOAT32_C(   752.00), SIMDE_FLOAT32_C(  -714.00), SIMDE_FLOAT32_C(   607.00), SIMDE_FLOAT32_C(   625.00),
        SIMDE_FLOAT32_C(  -738.00), SIMDE_FLOAT32_C(  -528.00), SIMDE_FLOAT32_C(   510.00), SIMDE_FLOAT32_C(   457.00),
        SIMDE_FLOAT32_C(  -562.00), SIMDE_FLOAT32_C(  -572.00), SIMDE_FLOAT32_C(  -375.00), SIMDE_FLOAT32_C(   223.00),
        SIMDE_FLOAT32_C(   671.00), SIMDE_FLOAT32_C(   141.00), SIMDE_FLOAT32_C(  -480.00), SIMDE_FLOAT32_C(   691.00) },
      { SIMDE_FLOAT32_C(   752.00), SIMDE_FLOAT32_C(  -715.00), SIMDE_FLOAT32_C(   606.00), SIMDE_FLOAT32_C(   624.00),
        SIMDE_FLOAT32_C(  -738.00), SIMDE_FLOAT32_C(  -529.00), SIMDE_FLOAT32_C(   509.00), SIMDE_FLOAT32_C(   457.00),
        SIMDE_FLOAT32_C(  -563.00), SIMDE_FLOAT32_C(  -573.00), SIMDE_FLOAT32_C(  -376.00), SIMDE_FLOAT32_C(   223.00),
        SIMDE_FLOAT32_C(   671.00), SIMDE_FLOAT32_C(   141.00), SIMDE_FLOAT32_C(  -481.00), SIMDE_FLOAT32_C(   691.00) },
      { SIMDE_FLOAT32_C(   753.00), SIMDE_FLOAT32_C(  -714.00), SIMDE_FLOAT32_C(   607.00), SIMDE_FLOAT32_C(   625.00),
        SIMDE_FLOAT32_C(  -737.00), SIMDE_FLOAT32_C(  -528.00), SIMDE_FLOAT32_C(   510.00), SIMDE_FLOAT32_C(   458.00),
        SIMDE_FLOAT32_C(  -562.00), SIMDE_FLOAT32_C(  -572.00), SIMDE_FLOAT32_C(  -375.00), SIMDE_FLOAT32_C(   224.00),
        SIMDE_FLOAT32_C(   672.00), SIMDE_FLOAT32_C(   142.00), SIMDE_FLOAT32_C(  -480.00), SIMDE_FLOAT32_C(   692.00) },
      { SIMDE_FLOAT32_C(   752.00), SIMDE_FLOAT32_C(  -714.00), SIMDE_FLOAT32_C(   606.00), SIMDE_FLOAT32_C(   624.00),
        SIMDE_FLOAT32_C(  -737.00), SIMDE_FLOAT32_C(  -528.00), SIMDE_FLOAT32_C(   509.00), SIMDE_FLOAT32_C(   457.00),
        SIMDE_FLOAT32_C(  -562.00), SIMDE_FLOAT32_C(  -572.00), SIMDE_FLOAT32_C(  -375.00), SIMDE_FLOAT32_C(   223.00),
        SIMDE_FLOAT32_C(   671.00), SIMDE_FLOAT32_C(   141.00), SIMDE_FLOAT32_C(  -480.00), SIMDE_FLOAT32_C(   691.00) } },
    { { SIMDE_FLOAT32_C(   745.50), SIMDE_FLOAT32_C(   530.58), SIMDE_FLOAT32_C(  -347.07), SIMDE_FLOAT32_C(   845.69),
        SIMDE_FLOAT32_C(  -954.93), SIMDE_FLOAT32_C(  -720.19), SIMDE_FLOAT32_C(  -628.94), SIMDE_FLOAT32_C(  -334.83),
        SIMDE_FLOAT32_C(   724.11), SIMDE_FLOAT32_C(  -343.22), SIMDE_FLOAT32_C(   838.31), SIMDE_FLOAT32_C(  -758.16),
        SIMDE_FLOAT32_C(   404.94), SIMDE_FLOAT32_C(  -145.27), SIMDE_FLOAT32_C(   848.91), SIMDE_FLOAT32_C(   157.02) },
      { SIMDE_FLOAT32_C(   746.00), SIMDE_FLOAT32_C(   531.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(   846.00),
        SIMDE_FLOAT32_C(  -955.00), SIMDE_FLOAT32_C(  -720.00), SIMDE_FLOAT32_C(  -629.00), SIMDE_FLOAT32_C(  -335.00),
        SIMDE_FLOAT32_C(   724.00), SIMDE_FLOAT32_C(  -343.00), SIMDE_FLOAT32_C(   838.00), SIMDE_FLOAT32_C(  -758.00),
        SIMDE_FLOAT32_C(   405.00), SIMDE_FLOAT32_C(  -145.00), SIMDE_FLOAT32_C(   849.00), SIMDE_FLOAT32_C(   157.00) },
      { SIMDE_FLOAT32_C(   745.00), SIMDE_FLOAT32_C(   530.00), SIMDE_FLOAT32_C(  -348.00), SIMDE_FLOAT32_C(   845.00),
        SIMDE_FLOAT32_C(  -955.00), SIMDE_FLOAT32_C(  -721.00), SIMDE_FLOAT32_C(  -629.00), SIMDE_FLOAT32_C(  -335.00),
        SIMDE_FLOAT32_C(   724.00), SIMDE_FLOAT32_C(  -344.00), SIMDE_FLOAT32_C(   838.00), SIMDE_FLOAT32_C(  -759.00),
        SIMDE_FLOAT32_C(   404.00), SIMDE_FLOAT32_C(  -146.00), SIMDE_FLOAT32_C(   848.00), SIMDE_FLOAT32_C(   157.00) },
      { SIMDE_FLOAT32_C(   746.00), SIMDE_FLOAT32_C(   531.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(   846.00),
        SIMDE_FLOAT32_C(  -954.00), SIMDE_FLOAT32_C(  -720.00), SIMDE_FLOAT32_C(  -628.00), SIMDE_FLOAT32_C(  -334.00),
        SIMDE_FLOAT32_C(   725.00), SIMDE_FLOAT32_C(  -343.00), SIMDE_FLOAT32_C(   839.00), SIMDE_FLOAT32_C(  -758.00),
        SIMDE_FLOAT32_C(   405.00), SIMDE_FLOAT32_C(  -145.00), SIMDE_FLOAT32_C(   849.00), SIMDE_FLOAT32_C(   158.00) },
      { SIMDE_FLOAT32_C(   745.00), SIMDE_FLOAT32_C(   530.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(   845.00),
        SIMDE_FLOAT32_C(  -954.00), SIMDE_FLOAT32_C(  -720.00), SIMDE_FLOAT32_C(  -628.00), SIMDE_FLOAT32_C(  -334.00),
        SIMDE_FLOAT32_C(   724.00), SIMDE_FLOAT32_C(  -343.00), SIMDE_FLOAT32_C(   838.00), SIMDE_FLOAT32_C(  -758.00),
        SIMDE_FLOAT32_C(   404.00), SIMDE_FLOAT32_C(  -145.00), SIMDE_FLOAT32_C(   848.00), SIMDE_FLOAT32_C(   157.00) } },
    { { SIMDE_FLOAT32_C(   140.25), SIMDE_FLOAT32_C(   455.73), SIMDE_FLOAT32_C(  -218.31), SIMDE_FLOAT32_C(   402.39),
        SIMDE_FLOAT32_C(   927.73), SIMDE_FLOAT32_C(  -708.47), SIMDE_FLOAT32_C(  -140.12), SIMDE_FLOAT32_C(  -634.55),
        SIMDE_FLOAT32_C(  -280.90), SIMDE_FLOAT32_C(   484.53), SIMDE_FLOAT32_C(   588.77), SIMDE_FLOAT32_C(  -609.46),
        SIMDE_FLOAT32_C(  -374.35), SIMDE_FLOAT32_C(  -891.68), SIMDE_FLOAT32_C(  -918.11), SIMDE_FLOAT32_C(  -628.85) },
      { SIMDE_FLOAT32_C(   140.00), SIMDE_FLOAT32_C(   456.00), SIMDE_FLOAT32_C(  -218.00), SIMDE_FLOAT32_C(   402.00),
        SIMDE_FLOAT32_C(   928.00), SIMDE_FLOAT32_C(  -708.00), SIMDE_FLOAT32_C(  -140.00), SIMDE_FLOAT32_C(  -635.00),
        SIMDE_FLOAT32_C(  -281.00), SIMDE_FLOAT32_C(   485.00), SIMDE_FLOAT32_C(   589.00), SIMDE_FLOAT32_C(  -609.00),
        SIMDE_FLOAT32_C(  -374.00), SIMDE_FLOAT32_C(  -892.00), SIMDE_FLOAT32_C(  -918.00), SIMDE_FLOAT32_C(  -629.00) },
      { SIMDE_FLOAT32_C(   140.00), SIMDE_FLOAT32_C(   455.00), SIMDE_FLOAT32_C(  -219.00), SIMDE_FLOAT32_C(   402.00),
        SIMDE_FLOAT32_C(   927.00), SIMDE_FLOAT32_C(  -709.00), SIMDE_FLOAT32_C(  -141.00), SIMDE_FLOAT32_C(  -635.00),
        SIMDE_FLOAT32_C(  -281.00), SIMDE_FLOAT32_C(   484.00), SIMDE_FLOAT32_C(   588.00), SIMDE_FLOAT32_C(  -610.00),
        SIMDE_FLOAT32_C(  -375.00), SIMDE_FLOAT32_C(  -892.00), SIMDE_FLOAT32_C(  -919.00), SIMDE_FLOAT32_C(  -629.00) },
      { SIMDE_FLOAT32_C(   141.00), SIMDE_FLOAT32_C(   456.00), SIMDE_FLOAT32_C(  -218.00), SIMDE_FLOAT32_C(   403.00),
        SIMDE_FLOAT32_C(   928.00), SIMDE_FLOAT32_C(  -708.00), SIMDE_FLOAT32_C(  -140.00), SIMDE_FLOAT32_C(  -634.00),
        SIMDE_FLOAT32_C(  -280.00), SIMDE_FLOAT32_C(   485.00), SIMDE_FLOAT32_C(   589.00), SIMDE_FLOAT32_C(  -609.00),
        SIMDE_FLOAT32_C(  -374.00), SIMDE_FLOAT32_C(  -891.00), SIMDE_FLOAT32_C(  -918.00), SIMDE_FLOAT32_C(  -628.00) },
      { SIMDE_FLOAT32_C(   140.00), SIMDE_FLOAT32_C(   455.00), SIMDE_FLOAT32_C(  -218.00), SIMDE_FLOAT32_C(   402.00),
        SIMDE_FLOAT32_C(   927.00), SIMDE_FLOAT32_C(  -708.00), SIMDE_FLOAT32_C(  -140.00), SIMDE_FLOAT32_C(  -634.00),
        SIMDE_FLOAT32_C(  -280.00), SIMDE_FLOAT32_C(   484.00), SIMDE_FLOAT32_C(   588.00), SIMDE_FLOAT32_C(  -609.00),
        SIMDE_FLOAT32_C(  -374.00), SIMDE_FLOAT32_C(  -891.00), SIMDE_FLOAT32_C(  -918.00), SIMDE_FLOAT32_C(  -628.00) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 r, a, nearest, neginf, posinf, truncate;
    a = simde_mm512_loadu_ps(test_vec[i].a);

    nearest = simde_mm512_loadu_ps(test_vec[i].nearest);
    r = simde_x_mm512_round_ps(a, SIMDE_MM_FROUND_TO_NEAREST_INT);
    simde_assert_m512_close(r, nearest, 1);

    neginf = simde_mm512_loadu_ps(test_vec[i].neginf);
    r = simde_x_mm512_round_ps(a, SIMDE_MM_FROUND_TO_NEG_INF);
    simde_assert_m512_close(r, neginf, 1);

    posinf = simde_mm512_loadu_ps(test_vec[i].posinf);
    r = simde_x_mm512_round_ps(a, SIMDE_MM_FROUND_TO_POS_INF);
    simde_assert_m512_close(r, posinf, 1);

    truncate = simde_mm512_loadu_ps(test_vec[i].truncate);
    r = simde_x_mm512_round_ps(a, SIMDE_MM_FROUND_TO_ZERO);
    simde_assert_m512_close(r, truncate, 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512 a = simde_test_x86_random_f32x16(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__m512 nearest = simde_x_mm512_round_ps(a, SIMDE_MM_FROUND_TO_NEAREST_INT);
    simde__m512 neginf = simde_x_mm512_round_ps(a, SIMDE_MM_FROUND_TO_NEG_INF);
    simde__m512 posinf = simde_x_mm512_round_ps(a, SIMDE_MM_FROUND_TO_POS_INF);
    simde__m512 truncate = simde_x_mm512_round_ps(a, SIMDE_MM_FROUND_TO_ZERO);

    simde_test_x86_write_f32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f32x16(2, nearest, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, neginf, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, posinf, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f32x16(2, truncate, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_x_mm512_round_pd (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde_float64 a[8];
    const simde_float64 nearest[8];
    const simde_float64 neginf[8];
    const simde_float64 posinf[8];
    const simde_float64 truncate[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(    -2.43), SIMDE_FLOAT64_C(  -556.24), SIMDE_FLOAT64_C(   551.80), SIMDE_FLOAT64_C(  -947.27),
        SIMDE_FLOAT64_C(   254.58), SIMDE_FLOAT64_C(  -914.47), SIMDE_FLOAT64_C(   -51.96), SIMDE_FLOAT64_C(  -801.35) },
      { SIMDE_FLOAT64_C(    -2.00), SIMDE_FLOAT64_C(  -556.00), SIMDE_FLOAT64_C(   552.00), SIMDE_FLOAT64_C(  -947.00),
        SIMDE_FLOAT64_C(   255.00), SIMDE_FLOAT64_C(  -914.00), SIMDE_FLOAT64_C(   -52.00), SIMDE_FLOAT64_C(  -801.00) },
      { SIMDE_FLOAT64_C(    -3.00), SIMDE_FLOAT64_C(  -557.00), SIMDE_FLOAT64_C(   551.00), SIMDE_FLOAT64_C(  -948.00),
        SIMDE_FLOAT64_C(   254.00), SIMDE_FLOAT64_C(  -915.00), SIMDE_FLOAT64_C(   -52.00), SIMDE_FLOAT64_C(  -802.00) },
      { SIMDE_FLOAT64_C(    -2.00), SIMDE_FLOAT64_C(  -556.00), SIMDE_FLOAT64_C(   552.00), SIMDE_FLOAT64_C(  -947.00),
        SIMDE_FLOAT64_C(   255.00), SIMDE_FLOAT64_C(  -914.00), SIMDE_FLOAT64_C(   -51.00), SIMDE_FLOAT64_C(  -801.00) },
      { SIMDE_FLOAT64_C(    -2.00), SIMDE_FLOAT64_C(  -556.00), SIMDE_FLOAT64_C(   551.00), SIMDE_FLOAT64_C(  -947.00),
        SIMDE_FLOAT64_C(   254.00), SIMDE_FLOAT64_C(  -914.00), SIMDE_FLOAT64_C(   -51.00), SIMDE_FLOAT64_C(  -801.00) } },
    { { SIMDE_FLOAT64_C(   105.94), SIMDE_FLOAT64_C(  -656.40), SIMDE_FLOAT64_C(    78.91), SIMDE_FLOAT64_C(  -815.72),
        SIMDE_FLOAT64_C(  -551.15), SIMDE_FLOAT64_C(  -213.35), SIMDE_FLOAT64_C(  -715.49), SIMDE_FLOAT64_C(  -507.96) },
      { SIMDE_FLOAT64_C(   106.00), SIMDE_FLOAT64_C(  -656.00), SIMDE_FLOAT64_C(    79.00), SIMDE_FLOAT64_C(  -816.00),
        SIMDE_FLOAT64_C(  -551.00), SIMDE_FLOAT64_C(  -213.00), SIMDE_FLOAT64_C(  -715.00), SIMDE_FLOAT64_C(  -508.00) },
      { SIMDE_FLOAT64_C(   105.00), SIMDE_FLOAT64_C(  -657.00), SIMDE_FLOAT64_C(    78.00), SIMDE_FLOAT64_C(  -816.00),
        SIMDE_FLOAT64_C(  -552.00), SIMDE_FLOAT64_C(  -214.00), SIMDE_FLOAT64_C(  -716.00), SIMDE_FLOAT64_C(  -508.00) },
      { SIMDE_FLOAT64_C(   106.00), SIMDE_FLOAT64_C(  -656.00), SIMDE_FLOAT64_C(    79.00), SIMDE_FLOAT64_C(  -815.00),
        SIMDE_FLOAT64_C(  -551.00), SIMDE_FLOAT64_C(  -213.00), SIMDE_FLOAT64_C(  -715.00), SIMDE_FLOAT64_C(  -507.00) },
      { SIMDE_FLOAT64_C(   105.00), SIMDE_FLOAT64_C(  -656.00), SIMDE_FLOAT64_C(    78.00), SIMDE_FLOAT64_C(  -815.00),
        SIMDE_FLOAT64_C(  -551.00), SIMDE_FLOAT64_C(  -213.00), SIMDE_FLOAT64_C(  -715.00), SIMDE_FLOAT64_C(  -507.00) } },
    { { SIMDE_FLOAT64_C(  -677.00), SIMDE_FLOAT64_C(  -389.91), SIMDE_FLOAT64_C(   742.17), SIMDE_FLOAT64_C(   828.13),
        SIMDE_FLOAT64_C(   337.82), SIMDE_FLOAT64_C(  -849.51), SIMDE_FLOAT64_C(  -153.85), SIMDE_FLOAT64_C(   638.56) },
      { SIMDE_FLOAT64_C(  -677.00), SIMDE_FLOAT64_C(  -390.00), SIMDE_FLOAT64_C(   742.00), SIMDE_FLOAT64_C(   828.00),
        SIMDE_FLOAT64_C(   338.00), SIMDE_FLOAT64_C(  -850.00), SIMDE_FLOAT64_C(  -154.00), SIMDE_FLOAT64_C(   639.00) },
      { SIMDE_FLOAT64_C(  -677.00), SIMDE_FLOAT64_C(  -390.00), SIMDE_FLOAT64_C(   742.00), SIMDE_FLOAT64_C(   828.00),
        SIMDE_FLOAT64_C(   337.00), SIMDE_FLOAT64_C(  -850.00), SIMDE_FLOAT64_C(  -154.00), SIMDE_FLOAT64_C(   638.00) },
      { SIMDE_FLOAT64_C(  -677.00), SIMDE_FLOAT64_C(  -389.00), SIMDE_FLOAT64_C(   743.00), SIMDE_FLOAT64_C(   829.00),
        SIMDE_FLOAT64_C(   338.00), SIMDE_FLOAT64_C(  -849.00), SIMDE_FLOAT64_C(  -153.00), SIMDE_FLOAT64_C(   639.00) },
      { SIMDE_FLOAT64_C(  -677.00), SIMDE_FLOAT64_C(  -389.00), SIMDE_FLOAT64_C(   742.00), SIMDE_FLOAT64_C(   828.00),
        SIMDE_FLOAT64_C(   337.00), SIMDE_FLOAT64_C(  -849.00), SIMDE_FLOAT64_C(  -153.00), SIMDE_FLOAT64_C(   638.00) } },
    { { SIMDE_FLOAT64_C(   367.86), SIMDE_FLOAT64_C(  -863.27), SIMDE_FLOAT64_C(  -201.65), SIMDE_FLOAT64_C(  -402.79),
        SIMDE_FLOAT64_C(   333.89), SIMDE_FLOAT64_C(  -134.32), SIMDE_FLOAT64_C(  -861.81), SIMDE_FLOAT64_C(  -668.54) },
      { SIMDE_FLOAT64_C(   368.00), SIMDE_FLOAT64_C(  -863.00), SIMDE_FLOAT64_C(  -202.00), SIMDE_FLOAT64_C(  -403.00),
        SIMDE_FLOAT64_C(   334.00), SIMDE_FLOAT64_C(  -134.00), SIMDE_FLOAT64_C(  -862.00), SIMDE_FLOAT64_C(  -669.00) },
      { SIMDE_FLOAT64_C(   367.00), SIMDE_FLOAT64_C(  -864.00), SIMDE_FLOAT64_C(  -202.00), SIMDE_FLOAT64_C(  -403.00),
        SIMDE_FLOAT64_C(   333.00), SIMDE_FLOAT64_C(  -135.00), SIMDE_FLOAT64_C(  -862.00), SIMDE_FLOAT64_C(  -669.00) },
      { SIMDE_FLOAT64_C(   368.00), SIMDE_FLOAT64_C(  -863.00), SIMDE_FLOAT64_C(  -201.00), SIMDE_FLOAT64_C(  -402.00),
        SIMDE_FLOAT64_C(   334.00), SIMDE_FLOAT64_C(  -134.00), SIMDE_FLOAT64_C(  -861.00), SIMDE_FLOAT64_C(  -668.00) },
      { SIMDE_FLOAT64_C(   367.00), SIMDE_FLOAT64_C(  -863.00), SIMDE_FLOAT64_C(  -201.00), SIMDE_FLOAT64_C(  -402.00),
        SIMDE_FLOAT64_C(   333.00), SIMDE_FLOAT64_C(  -134.00), SIMDE_FLOAT64_C(  -861.00), SIMDE_FLOAT64_C(  -668.00) } },
    { { SIMDE_FLOAT64_C(   309.44), SIMDE_FLOAT64_C(   689.99), SIMDE_FLOAT64_C(  -615.81), SIMDE_FLOAT64_C(  -435.98),
        SIMDE_FLOAT64_C(   775.52), SIMDE_FLOAT64_C(   332.22), SIMDE_FLOAT64_C(  -237.33), SIMDE_FLOAT64_C(  -118.54) },
      { SIMDE_FLOAT64_C(   309.00), SIMDE_FLOAT64_C(   690.00), SIMDE_FLOAT64_C(  -616.00), SIMDE_FLOAT64_C(  -436.00),
        SIMDE_FLOAT64_C(   776.00), SIMDE_FLOAT64_C(   332.00), SIMDE_FLOAT64_C(  -237.00), SIMDE_FLOAT64_C(  -119.00) },
      { SIMDE_FLOAT64_C(   309.00), SIMDE_FLOAT64_C(   689.00), SIMDE_FLOAT64_C(  -616.00), SIMDE_FLOAT64_C(  -436.00),
        SIMDE_FLOAT64_C(   775.00), SIMDE_FLOAT64_C(   332.00), SIMDE_FLOAT64_C(  -238.00), SIMDE_FLOAT64_C(  -119.00) },
      { SIMDE_FLOAT64_C(   310.00), SIMDE_FLOAT64_C(   690.00), SIMDE_FLOAT64_C(  -615.00), SIMDE_FLOAT64_C(  -435.00),
        SIMDE_FLOAT64_C(   776.00), SIMDE_FLOAT64_C(   333.00), SIMDE_FLOAT64_C(  -237.00), SIMDE_FLOAT64_C(  -118.00) },
      { SIMDE_FLOAT64_C(   309.00), SIMDE_FLOAT64_C(   689.00), SIMDE_FLOAT64_C(  -615.00), SIMDE_FLOAT64_C(  -435.00),
        SIMDE_FLOAT64_C(   775.00), SIMDE_FLOAT64_C(   332.00), SIMDE_FLOAT64_C(  -237.00), SIMDE_FLOAT64_C(  -118.00) } },
    { { SIMDE_FLOAT64_C(   675.82), SIMDE_FLOAT64_C(   841.58), SIMDE_FLOAT64_C(    65.74), SIMDE_FLOAT64_C(  -875.33),
        SIMDE_FLOAT64_C(  -371.77), SIMDE_FLOAT64_C(   350.25), SIMDE_FLOAT64_C(  -383.29), SIMDE_FLOAT64_C(   -48.77) },
      { SIMDE_FLOAT64_C(   676.00), SIMDE_FLOAT64_C(   842.00), SIMDE_FLOAT64_C(    66.00), SIMDE_FLOAT64_C(  -875.00),
        SIMDE_FLOAT64_C(  -372.00), SIMDE_FLOAT64_C(   350.00), SIMDE_FLOAT64_C(  -383.00), SIMDE_FLOAT64_C(   -49.00) },
      { SIMDE_FLOAT64_C(   675.00), SIMDE_FLOAT64_C(   841.00), SIMDE_FLOAT64_C(    65.00), SIMDE_FLOAT64_C(  -876.00),
        SIMDE_FLOAT64_C(  -372.00), SIMDE_FLOAT64_C(   350.00), SIMDE_FLOAT64_C(  -384.00), SIMDE_FLOAT64_C(   -49.00) },
      { SIMDE_FLOAT64_C(   676.00), SIMDE_FLOAT64_C(   842.00), SIMDE_FLOAT64_C(    66.00), SIMDE_FLOAT64_C(  -875.00),
        SIMDE_FLOAT64_C(  -371.00), SIMDE_FLOAT64_C(   351.00), SIMDE_FLOAT64_C(  -383.00), SIMDE_FLOAT64_C(   -48.00) },
      { SIMDE_FLOAT64_C(   675.00), SIMDE_FLOAT64_C(   841.00), SIMDE_FLOAT64_C(    65.00), SIMDE_FLOAT64_C(  -875.00),
        SIMDE_FLOAT64_C(  -371.00), SIMDE_FLOAT64_C(   350.00), SIMDE_FLOAT64_C(  -383.00), SIMDE_FLOAT64_C(   -48.00) } },
    { { SIMDE_FLOAT64_C(   960.34), SIMDE_FLOAT64_C(  -641.12), SIMDE_FLOAT64_C(  -220.64), SIMDE_FLOAT64_C(   298.16),
        SIMDE_FLOAT64_C(  -490.63), SIMDE_FLOAT64_C(   625.51), SIMDE_FLOAT64_C(   -63.28), SIMDE_FLOAT64_C(   877.23) },
      { SIMDE_FLOAT64_C(   960.00), SIMDE_FLOAT64_C(  -641.00), SIMDE_FLOAT64_C(  -221.00), SIMDE_FLOAT64_C(   298.00),
        SIMDE_FLOAT64_C(  -491.00), SIMDE_FLOAT64_C(   626.00), SIMDE_FLOAT64_C(   -63.00), SIMDE_FLOAT64_C(   877.00) },
      { SIMDE_FLOAT64_C(   960.00), SIMDE_FLOAT64_C(  -642.00), SIMDE_FLOAT64_C(  -221.00), SIMDE_FLOAT64_C(   298.00),
        SIMDE_FLOAT64_C(  -491.00), SIMDE_FLOAT64_C(   625.00), SIMDE_FLOAT64_C(   -64.00), SIMDE_FLOAT64_C(   877.00) },
      { SIMDE_FLOAT64_C(   961.00), SIMDE_FLOAT64_C(  -641.00), SIMDE_FLOAT64_C(  -220.00), SIMDE_FLOAT64_C(   299.00),
        SIMDE_FLOAT64_C(  -490.00), SIMDE_FLOAT64_C(   626.00), SIMDE_FLOAT64_C(   -63.00), SIMDE_FLOAT64_C(   878.00) },
      { SIMDE_FLOAT64_C(   960.00), SIMDE_FLOAT64_C(  -641.00), SIMDE_FLOAT64_C(  -220.00), SIMDE_FLOAT64_C(   298.00),
        SIMDE_FLOAT64_C(  -490.00), SIMDE_FLOAT64_C(   625.00), SIMDE_FLOAT64_C(   -63.00), SIMDE_FLOAT64_C(   877.00) } },
    { { SIMDE_FLOAT64_C(   762.24), SIMDE_FLOAT64_C(   735.07), SIMDE_FLOAT64_C(  -525.56), SIMDE_FLOAT64_C(    96.14),
        SIMDE_FLOAT64_C(  -399.24), SIMDE_FLOAT64_C(  -387.37), SIMDE_FLOAT64_C(   427.60), SIMDE_FLOAT64_C(   910.20) },
      { SIMDE_FLOAT64_C(   762.00), SIMDE_FLOAT64_C(   735.00), SIMDE_FLOAT64_C(  -526.00), SIMDE_FLOAT64_C(    96.00),
        SIMDE_FLOAT64_C(  -399.00), SIMDE_FLOAT64_C(  -387.00), SIMDE_FLOAT64_C(   428.00), SIMDE_FLOAT64_C(   910.00) },
      { SIMDE_FLOAT64_C(   762.00), SIMDE_FLOAT64_C(   735.00), SIMDE_FLOAT64_C(  -526.00), SIMDE_FLOAT64_C(    96.00),
        SIMDE_FLOAT64_C(  -400.00), SIMDE_FLOAT64_C(  -388.00), SIMDE_FLOAT64_C(   427.00), SIMDE_FLOAT64_C(   910.00) },
      { SIMDE_FLOAT64_C(   763.00), SIMDE_FLOAT64_C(   736.00), SIMDE_FLOAT64_C(  -525.00), SIMDE_FLOAT64_C(    97.00),
        SIMDE_FLOAT64_C(  -399.00), SIMDE_FLOAT64_C(  -387.00), SIMDE_FLOAT64_C(   428.00), SIMDE_FLOAT64_C(   911.00) },
      { SIMDE_FLOAT64_C(   762.00), SIMDE_FLOAT64_C(   735.00), SIMDE_FLOAT64_C(  -525.00), SIMDE_FLOAT64_C(    96.00),
        SIMDE_FLOAT64_C(  -399.00), SIMDE_FLOAT64_C(  -387.00), SIMDE_FLOAT64_C(   427.00), SIMDE_FLOAT64_C(   910.00) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d r, a, nearest, neginf, posinf, truncate;
    a = simde_mm512_loadu_pd(test_vec[i].a);

    nearest = simde_mm512_loadu_pd(test_vec[i].nearest);
    r = simde_x_mm512_round_pd(a, SIMDE_MM_FROUND_TO_NEAREST_INT);
    simde_assert_m512d_close(r, nearest, 1);

    neginf = simde_mm512_loadu_pd(test_vec[i].neginf);
    r = simde_x_mm512_round_pd(a, SIMDE_MM_FROUND_TO_NEG_INF);
    simde_assert_m512d_close(r, neginf, 1);

    posinf = simde_mm512_loadu_pd(test_vec[i].posinf);
    r = simde_x_mm512_round_pd(a, SIMDE_MM_FROUND_TO_POS_INF);
    simde_assert_m512d_close(r, posinf, 1);

    truncate = simde_mm512_loadu_pd(test_vec[i].truncate);
    r = simde_x_mm512_round_pd(a, SIMDE_MM_FROUND_TO_ZERO);
    simde_assert_m512d_close(r, truncate, 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512d a = simde_test_x86_random_f64x8(SIMDE_FLOAT32_C(-1000.0), SIMDE_FLOAT32_C(1000.0));
    simde__m512d nearest = simde_x_mm512_round_pd(a, SIMDE_MM_FROUND_TO_NEAREST_INT);
    simde__m512d neginf = simde_x_mm512_round_pd(a, SIMDE_MM_FROUND_TO_NEG_INF);
    simde__m512d posinf = simde_x_mm512_round_pd(a, SIMDE_MM_FROUND_TO_POS_INF);
    simde__m512d truncate = simde_x_mm512_round_pd(a, SIMDE_MM_FROUND_TO_ZERO);

    simde_test_x86_write_f64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_f64x8(2, nearest, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, neginf, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, posinf, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_f64x8(2, truncate, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(x_mm512_round_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(x_mm512_round_pd)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
