/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2023      Chi-Wei Chu <wewe5215@gapp.nthu.edu.tw> (Copyright owned by NTHU pllab)
 */

#define SIMDE_TEST_ARM_NEON_INSN cadd_rot270

#include "test-neon.h"
#include <simde/arm/neon/cadd_rot270.h>

static int
test_simde_vcadd_rot270_f16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    simde_float16_t a[4];
    simde_float16_t b[4];
    simde_float16_t r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT16_VALUE(   816.50), SIMDE_FLOAT16_VALUE(   773.50), SIMDE_FLOAT16_VALUE(  -377.75), SIMDE_FLOAT16_VALUE(   544.50) },
      { SIMDE_FLOAT16_VALUE(  -188.38), SIMDE_FLOAT16_VALUE(   269.75), SIMDE_FLOAT16_VALUE(   115.62), SIMDE_FLOAT16_VALUE(  -544.00) },
      { SIMDE_FLOAT16_VALUE(  1086.00), SIMDE_FLOAT16_VALUE(   962.00), SIMDE_FLOAT16_VALUE(  -922.00), SIMDE_FLOAT16_VALUE(   429.00) } },
    { { SIMDE_FLOAT16_VALUE(  -659.50), SIMDE_FLOAT16_VALUE(   924.50), SIMDE_FLOAT16_VALUE(   484.00), SIMDE_FLOAT16_VALUE(  -757.00) },
      { SIMDE_FLOAT16_VALUE(   730.00), SIMDE_FLOAT16_VALUE(   103.12), SIMDE_FLOAT16_VALUE(  -381.25), SIMDE_FLOAT16_VALUE(   898.50) },
      { SIMDE_FLOAT16_VALUE(  -556.50), SIMDE_FLOAT16_VALUE(   194.50), SIMDE_FLOAT16_VALUE(  1382.00), SIMDE_FLOAT16_VALUE(  -375.75) } },
    { { SIMDE_FLOAT16_VALUE(   293.00), SIMDE_FLOAT16_VALUE(  -114.06), SIMDE_FLOAT16_VALUE(  -137.62), SIMDE_FLOAT16_VALUE(   722.50) },
      { SIMDE_FLOAT16_VALUE(    -6.33), SIMDE_FLOAT16_VALUE(   161.38), SIMDE_FLOAT16_VALUE(   115.50), SIMDE_FLOAT16_VALUE(    70.38) },
      { SIMDE_FLOAT16_VALUE(   454.50), SIMDE_FLOAT16_VALUE(  -107.75), SIMDE_FLOAT16_VALUE(   -67.25), SIMDE_FLOAT16_VALUE(   607.00) } },
    { { SIMDE_FLOAT16_VALUE(  -899.00), SIMDE_FLOAT16_VALUE(  -582.50), SIMDE_FLOAT16_VALUE(   359.00), SIMDE_FLOAT16_VALUE(   342.25) },
      { SIMDE_FLOAT16_VALUE(   913.00), SIMDE_FLOAT16_VALUE(   740.50), SIMDE_FLOAT16_VALUE(  -436.25), SIMDE_FLOAT16_VALUE(  -904.00) },
      { SIMDE_FLOAT16_VALUE(  -158.50), SIMDE_FLOAT16_VALUE( -1496.00), SIMDE_FLOAT16_VALUE(  -545.00), SIMDE_FLOAT16_VALUE(   778.50) } },
    { { SIMDE_FLOAT16_VALUE(  -335.25), SIMDE_FLOAT16_VALUE(  -252.38), SIMDE_FLOAT16_VALUE(   227.25), SIMDE_FLOAT16_VALUE(    -8.67) },
      { SIMDE_FLOAT16_VALUE(   490.50), SIMDE_FLOAT16_VALUE(   507.50), SIMDE_FLOAT16_VALUE(  -979.50), SIMDE_FLOAT16_VALUE(  -718.00) },
      { SIMDE_FLOAT16_VALUE(   172.25), SIMDE_FLOAT16_VALUE(  -743.00), SIMDE_FLOAT16_VALUE(  -490.75), SIMDE_FLOAT16_VALUE(   971.00) } },
    { { SIMDE_FLOAT16_VALUE(   498.50), SIMDE_FLOAT16_VALUE(   205.75), SIMDE_FLOAT16_VALUE(  -797.50), SIMDE_FLOAT16_VALUE(   741.50) },
      { SIMDE_FLOAT16_VALUE(  -835.00), SIMDE_FLOAT16_VALUE(  -308.00), SIMDE_FLOAT16_VALUE(  -674.50), SIMDE_FLOAT16_VALUE(  -247.00) },
      { SIMDE_FLOAT16_VALUE(   190.50), SIMDE_FLOAT16_VALUE(  1041.00), SIMDE_FLOAT16_VALUE( -1044.00), SIMDE_FLOAT16_VALUE(  1416.00) } },
    { { SIMDE_FLOAT16_VALUE(  -210.88), SIMDE_FLOAT16_VALUE(  -430.75), SIMDE_FLOAT16_VALUE(   725.50), SIMDE_FLOAT16_VALUE(  -944.50) },
      { SIMDE_FLOAT16_VALUE(   142.00), SIMDE_FLOAT16_VALUE(  -966.00), SIMDE_FLOAT16_VALUE(  -131.50), SIMDE_FLOAT16_VALUE(   462.50) },
      { SIMDE_FLOAT16_VALUE( -1177.00), SIMDE_FLOAT16_VALUE(  -573.00), SIMDE_FLOAT16_VALUE(  1188.00), SIMDE_FLOAT16_VALUE(  -813.00) } },
    { { SIMDE_FLOAT16_VALUE(  -361.00), SIMDE_FLOAT16_VALUE(   185.62), SIMDE_FLOAT16_VALUE(    60.47), SIMDE_FLOAT16_VALUE(   749.50) },
      { SIMDE_FLOAT16_VALUE(  -975.50), SIMDE_FLOAT16_VALUE(   383.50), SIMDE_FLOAT16_VALUE(   500.25), SIMDE_FLOAT16_VALUE(   517.00) },
      { SIMDE_FLOAT16_VALUE(    22.50), SIMDE_FLOAT16_VALUE(  1161.00), SIMDE_FLOAT16_VALUE(   577.50), SIMDE_FLOAT16_VALUE(   249.25) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_float16x4_t a = simde_vld1_f16(test_vec[i].a);
    simde_float16x4_t b = simde_vld1_f16(test_vec[i].b);
    simde_float16x4_t r = simde_vcadd_rot270_f16(a, b);

    simde_test_arm_neon_assert_equal_f16x4(r, simde_vld1_f16(test_vec[i].r), 1);
  }
  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_float16x4_t a = simde_test_arm_neon_random_f16x4(-1000.0f, 1000.0f);
    simde_float16x4_t b = simde_test_arm_neon_random_f16x4(-1000.0f, 1000.0f);
    simde_float16x4_t r = simde_vcadd_rot270_f16(a, b);

    simde_test_arm_neon_write_f16x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_f16x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_f16x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vcaddq_rot270_f16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  struct {
    simde_float16_t a[8];
    simde_float16_t b[8];
    simde_float16_t r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT16_VALUE(   540.00), SIMDE_FLOAT16_VALUE(  -927.50), SIMDE_FLOAT16_VALUE(    17.94), SIMDE_FLOAT16_VALUE(   340.75),
        SIMDE_FLOAT16_VALUE(  -397.75), SIMDE_FLOAT16_VALUE(   166.12), SIMDE_FLOAT16_VALUE(   854.00), SIMDE_FLOAT16_VALUE(  -110.75) },
      { SIMDE_FLOAT16_VALUE(  -936.50), SIMDE_FLOAT16_VALUE(  -465.00), SIMDE_FLOAT16_VALUE(  -104.00), SIMDE_FLOAT16_VALUE(  -348.00),
        SIMDE_FLOAT16_VALUE(  -370.75), SIMDE_FLOAT16_VALUE(    18.20), SIMDE_FLOAT16_VALUE(  -209.25), SIMDE_FLOAT16_VALUE(  -856.00) },
      { SIMDE_FLOAT16_VALUE(  75.00  ), SIMDE_FLOAT16_VALUE( 9.00), SIMDE_FLOAT16_VALUE(-330.00), SIMDE_FLOAT16_VALUE(444.75), 
        SIMDE_FLOAT16_VALUE(  -379.50), SIMDE_FLOAT16_VALUE(  537.00 ), SIMDE_FLOAT16_VALUE(-2.00), SIMDE_FLOAT16_VALUE(98.50) } },
    { { SIMDE_FLOAT16_VALUE(   251.25), SIMDE_FLOAT16_VALUE(   523.50), SIMDE_FLOAT16_VALUE(   -62.34), SIMDE_FLOAT16_VALUE(  -412.00),
        SIMDE_FLOAT16_VALUE(   818.50), SIMDE_FLOAT16_VALUE(  -266.25), SIMDE_FLOAT16_VALUE(   558.50), SIMDE_FLOAT16_VALUE(  -474.75) },
      { SIMDE_FLOAT16_VALUE(   583.50), SIMDE_FLOAT16_VALUE(  -666.00), SIMDE_FLOAT16_VALUE(   875.50), SIMDE_FLOAT16_VALUE(   382.50),
        SIMDE_FLOAT16_VALUE(  -535.00), SIMDE_FLOAT16_VALUE(   374.50), SIMDE_FLOAT16_VALUE(    89.81), SIMDE_FLOAT16_VALUE(   432.50) },
      { SIMDE_FLOAT16_VALUE(-414.75), SIMDE_FLOAT16_VALUE(-60.00), SIMDE_FLOAT16_VALUE(320.25), SIMDE_FLOAT16_VALUE(-1288.00), 
        SIMDE_FLOAT16_VALUE(1193.00), SIMDE_FLOAT16_VALUE(268.75), SIMDE_FLOAT16_VALUE(991.00), SIMDE_FLOAT16_VALUE(-564.50) } },
    { { SIMDE_FLOAT16_VALUE(    83.19), SIMDE_FLOAT16_VALUE(   418.00), SIMDE_FLOAT16_VALUE(  -875.50), SIMDE_FLOAT16_VALUE(   830.50),
        SIMDE_FLOAT16_VALUE(  -633.50), SIMDE_FLOAT16_VALUE(   832.00), SIMDE_FLOAT16_VALUE(  -890.00), SIMDE_FLOAT16_VALUE(   650.00) },
      { SIMDE_FLOAT16_VALUE(  -111.25), SIMDE_FLOAT16_VALUE(  -830.50), SIMDE_FLOAT16_VALUE(    59.75), SIMDE_FLOAT16_VALUE(   970.50),
        SIMDE_FLOAT16_VALUE(   522.50), SIMDE_FLOAT16_VALUE(  -822.50), SIMDE_FLOAT16_VALUE(   411.25), SIMDE_FLOAT16_VALUE(  -692.50) },
      { SIMDE_FLOAT16_VALUE(-747.50), SIMDE_FLOAT16_VALUE(529.00), SIMDE_FLOAT16_VALUE(95.00), SIMDE_FLOAT16_VALUE(771.000000), 
        SIMDE_FLOAT16_VALUE(-1456.00), SIMDE_FLOAT16_VALUE(309.50), SIMDE_FLOAT16_VALUE(-1582.00), SIMDE_FLOAT16_VALUE(238.750000) } },
    { { SIMDE_FLOAT16_VALUE(   648.50), SIMDE_FLOAT16_VALUE(   385.25), SIMDE_FLOAT16_VALUE(   479.25), SIMDE_FLOAT16_VALUE(  -793.50),
        SIMDE_FLOAT16_VALUE(  -740.50), SIMDE_FLOAT16_VALUE(   245.00), SIMDE_FLOAT16_VALUE(   229.25), SIMDE_FLOAT16_VALUE(  -113.25) },
      { SIMDE_FLOAT16_VALUE(   331.50), SIMDE_FLOAT16_VALUE(  -677.50), SIMDE_FLOAT16_VALUE(    97.31), SIMDE_FLOAT16_VALUE(   -52.09),
        SIMDE_FLOAT16_VALUE(   239.12), SIMDE_FLOAT16_VALUE(   469.75), SIMDE_FLOAT16_VALUE(  -543.50), SIMDE_FLOAT16_VALUE(   827.00) },
      { SIMDE_FLOAT16_VALUE(-29.00), SIMDE_FLOAT16_VALUE(53.75), SIMDE_FLOAT16_VALUE(427.25), SIMDE_FLOAT16_VALUE(-891.000000), 
        SIMDE_FLOAT16_VALUE(-270.75), SIMDE_FLOAT16_VALUE( 5.88), SIMDE_FLOAT16_VALUE(1056.00), SIMDE_FLOAT16_VALUE(430.250000) } },
    { { SIMDE_FLOAT16_VALUE(   226.38), SIMDE_FLOAT16_VALUE(   178.88), SIMDE_FLOAT16_VALUE(   181.75), SIMDE_FLOAT16_VALUE(   420.50),
        SIMDE_FLOAT16_VALUE(  -699.00), SIMDE_FLOAT16_VALUE(  -151.12), SIMDE_FLOAT16_VALUE(  -388.25), SIMDE_FLOAT16_VALUE(   350.75) },
      { SIMDE_FLOAT16_VALUE(   890.50), SIMDE_FLOAT16_VALUE(  -665.00), SIMDE_FLOAT16_VALUE(  -618.00), SIMDE_FLOAT16_VALUE(   190.88),
        SIMDE_FLOAT16_VALUE(   218.12), SIMDE_FLOAT16_VALUE(  -329.00), SIMDE_FLOAT16_VALUE(  -549.50), SIMDE_FLOAT16_VALUE(  -460.00) },
      { SIMDE_FLOAT16_VALUE(-438.50), SIMDE_FLOAT16_VALUE(-711.500000), SIMDE_FLOAT16_VALUE(372.500000), SIMDE_FLOAT16_VALUE(1038.000000), 
        SIMDE_FLOAT16_VALUE(-1028.00), SIMDE_FLOAT16_VALUE(-369.250000), SIMDE_FLOAT16_VALUE(-848.000000), SIMDE_FLOAT16_VALUE(900.000000) } },
    { { SIMDE_FLOAT16_VALUE(  -378.00), SIMDE_FLOAT16_VALUE(  -695.50), SIMDE_FLOAT16_VALUE(   121.50), SIMDE_FLOAT16_VALUE(   -78.44),
        SIMDE_FLOAT16_VALUE(  -869.50), SIMDE_FLOAT16_VALUE(  -710.00), SIMDE_FLOAT16_VALUE(   457.25), SIMDE_FLOAT16_VALUE(   579.00) },
      { SIMDE_FLOAT16_VALUE(  -786.50), SIMDE_FLOAT16_VALUE(  -974.00), SIMDE_FLOAT16_VALUE(  -570.50), SIMDE_FLOAT16_VALUE(   369.25),
        SIMDE_FLOAT16_VALUE(    48.06), SIMDE_FLOAT16_VALUE(  -640.00), SIMDE_FLOAT16_VALUE(  -552.00), SIMDE_FLOAT16_VALUE(    75.88) },
      { SIMDE_FLOAT16_VALUE(-1352.00), SIMDE_FLOAT16_VALUE(91.00), SIMDE_FLOAT16_VALUE(490.750000), SIMDE_FLOAT16_VALUE(492.000000), 
        SIMDE_FLOAT16_VALUE(-1510.00), SIMDE_FLOAT16_VALUE(-758.00), SIMDE_FLOAT16_VALUE(533.000000), SIMDE_FLOAT16_VALUE(1131.000000) } },
    { { SIMDE_FLOAT16_VALUE(   352.25), SIMDE_FLOAT16_VALUE(   324.25), SIMDE_FLOAT16_VALUE(   485.50), SIMDE_FLOAT16_VALUE(   943.50),
        SIMDE_FLOAT16_VALUE(  -976.00), SIMDE_FLOAT16_VALUE(   395.50), SIMDE_FLOAT16_VALUE(  -100.62), SIMDE_FLOAT16_VALUE(  -463.75) },
      { SIMDE_FLOAT16_VALUE(  -973.00), SIMDE_FLOAT16_VALUE(  -159.62), SIMDE_FLOAT16_VALUE(  -377.00), SIMDE_FLOAT16_VALUE(  -853.00),
        SIMDE_FLOAT16_VALUE(   831.00), SIMDE_FLOAT16_VALUE(   661.00), SIMDE_FLOAT16_VALUE(  -717.00), SIMDE_FLOAT16_VALUE(   891.50) },
      { SIMDE_FLOAT16_VALUE(192.63), SIMDE_FLOAT16_VALUE(1297.00), SIMDE_FLOAT16_VALUE(-367.500000), SIMDE_FLOAT16_VALUE(1320.000000), 
        SIMDE_FLOAT16_VALUE(-315.00), SIMDE_FLOAT16_VALUE(-435.50), SIMDE_FLOAT16_VALUE(791.000000), SIMDE_FLOAT16_VALUE(253.250000) } },
    { { SIMDE_FLOAT16_VALUE(   535.50), SIMDE_FLOAT16_VALUE(   621.50), SIMDE_FLOAT16_VALUE(   578.50), SIMDE_FLOAT16_VALUE(   430.50),
        SIMDE_FLOAT16_VALUE(  -574.50), SIMDE_FLOAT16_VALUE(  -319.75), SIMDE_FLOAT16_VALUE(  -735.00), SIMDE_FLOAT16_VALUE(   248.50) },
      { SIMDE_FLOAT16_VALUE(   274.50), SIMDE_FLOAT16_VALUE(   192.38), SIMDE_FLOAT16_VALUE(   679.00), SIMDE_FLOAT16_VALUE(   502.25),
        SIMDE_FLOAT16_VALUE(  -618.50), SIMDE_FLOAT16_VALUE(   -70.81), SIMDE_FLOAT16_VALUE(  -596.50), SIMDE_FLOAT16_VALUE(   810.50) },
      { SIMDE_FLOAT16_VALUE(728.00), SIMDE_FLOAT16_VALUE(347.00), SIMDE_FLOAT16_VALUE(1081.000000), SIMDE_FLOAT16_VALUE(-248.500000), 
        SIMDE_FLOAT16_VALUE(-645.50), SIMDE_FLOAT16_VALUE(298.75), SIMDE_FLOAT16_VALUE(75.500000), SIMDE_FLOAT16_VALUE(845.000000) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_float16x8_t a = simde_vld1q_f16(test_vec[i].a);
    simde_float16x8_t b = simde_vld1q_f16(test_vec[i].b);
    simde_float16x8_t r = simde_vcaddq_rot270_f16(a, b);
    simde_test_arm_neon_assert_equal_f16x8(r, simde_vld1q_f16(test_vec[i].r), 1);
  }
  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_float16x8_t a = simde_test_arm_neon_random_f16x8(-1000.0f, 1000.0f);
    simde_float16x8_t b = simde_test_arm_neon_random_f16x8(-1000.0f, 1000.0f);
    simde_float16x8_t r = simde_vcaddq_rot270_f16(a, b);

    simde_test_arm_neon_write_f16x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_f16x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_f16x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vcadd_rot270_f32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    simde_float32 a[2];
    simde_float32 b[2];
    simde_float32 r[2];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   863.66), SIMDE_FLOAT32_C(   828.31) },
      { SIMDE_FLOAT32_C(  -563.51), SIMDE_FLOAT32_C(  -576.51) },
      { SIMDE_FLOAT32_C(287.149963), SIMDE_FLOAT32_C(1391.820068) } },
    { { SIMDE_FLOAT32_C(  -703.45), SIMDE_FLOAT32_C(   383.90) },
      { SIMDE_FLOAT32_C(  -772.46), SIMDE_FLOAT32_C(   457.40) },
      { SIMDE_FLOAT32_C(-246.050018), SIMDE_FLOAT32_C(1156.359985) } },
    { { SIMDE_FLOAT32_C(   295.99), SIMDE_FLOAT32_C(   653.10) },
      { SIMDE_FLOAT32_C(  -120.98), SIMDE_FLOAT32_C(   945.50) },
      { SIMDE_FLOAT32_C(1241.489990), SIMDE_FLOAT32_C(774.079956) } },
    { { SIMDE_FLOAT32_C(  -280.81), SIMDE_FLOAT32_C(   631.32) },
      { SIMDE_FLOAT32_C(   688.34), SIMDE_FLOAT32_C(   191.95) },
      { SIMDE_FLOAT32_C(-88.860001), SIMDE_FLOAT32_C(-57.020020) } },
    { { SIMDE_FLOAT32_C(  -522.88), SIMDE_FLOAT32_C(  -323.79) },
      { SIMDE_FLOAT32_C(  -887.99), SIMDE_FLOAT32_C(  -283.70) },
      { SIMDE_FLOAT32_C(-806.580017), SIMDE_FLOAT32_C(564.199951) } },
    { { SIMDE_FLOAT32_C(  -117.76), SIMDE_FLOAT32_C(  -841.45) },
      { SIMDE_FLOAT32_C(   664.94), SIMDE_FLOAT32_C(  -987.19) },
      { SIMDE_FLOAT32_C(-1104.949951), SIMDE_FLOAT32_C(-1506.390015) } },
    { { SIMDE_FLOAT32_C(  -642.89), SIMDE_FLOAT32_C(  -152.10) },
      { SIMDE_FLOAT32_C(   963.83), SIMDE_FLOAT32_C(   919.89) },
      { SIMDE_FLOAT32_C(277.000000), SIMDE_FLOAT32_C(-1115.930054) } },
    { { SIMDE_FLOAT32_C(   630.40), SIMDE_FLOAT32_C(  -669.33) },
      { SIMDE_FLOAT32_C(   671.13), SIMDE_FLOAT32_C(   256.93) },
      { SIMDE_FLOAT32_C(887.330017), SIMDE_FLOAT32_C(-1340.459961) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_float32x2_t a = simde_vld1_f32(test_vec[i].a);
    simde_float32x2_t b = simde_vld1_f32(test_vec[i].b);
    simde_float32x2_t r = simde_vcadd_rot270_f32(a, b);

    simde_test_arm_neon_assert_equal_f32x2(r, simde_vld1_f32(test_vec[i].r), 1);
  }
  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_float32x2_t a = simde_test_arm_neon_random_f32x2(-1000.0f, 1000.0f);
    simde_float32x2_t b = simde_test_arm_neon_random_f32x2(-1000.0f, 1000.0f);
    simde_float32x2_t r = simde_vcadd_rot270_f32(a, b);

    simde_test_arm_neon_write_f32x2(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_f32x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_f32x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vcaddq_rot270_f32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    simde_float32 a[4];
    simde_float32 b[4];
    simde_float32 r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -337.31), SIMDE_FLOAT32_C(  -857.36), SIMDE_FLOAT32_C(   334.71), SIMDE_FLOAT32_C(  -617.33) },
      { SIMDE_FLOAT32_C(  -439.38), SIMDE_FLOAT32_C(   245.13), SIMDE_FLOAT32_C(   111.06), SIMDE_FLOAT32_C(   520.69) },
      { SIMDE_FLOAT32_C(-92.179993), SIMDE_FLOAT32_C(-417.979980), SIMDE_FLOAT32_C(855.400024), SIMDE_FLOAT32_C(-728.390015) } },
    { { SIMDE_FLOAT32_C(    85.49), SIMDE_FLOAT32_C(   250.19), SIMDE_FLOAT32_C(  -679.96), SIMDE_FLOAT32_C(  -750.25) },
      { SIMDE_FLOAT32_C(  -138.26), SIMDE_FLOAT32_C(   -14.62), SIMDE_FLOAT32_C(  -921.52), SIMDE_FLOAT32_C(   225.91) },
      { SIMDE_FLOAT32_C(70.869995), SIMDE_FLOAT32_C(388.450012), SIMDE_FLOAT32_C(-454.050018), SIMDE_FLOAT32_C(171.270020) } },
    { { SIMDE_FLOAT32_C(   242.83), SIMDE_FLOAT32_C(   869.28), SIMDE_FLOAT32_C(   297.95), SIMDE_FLOAT32_C(   105.66) },
      { SIMDE_FLOAT32_C(  -722.51), SIMDE_FLOAT32_C(  -802.37), SIMDE_FLOAT32_C(  -245.78), SIMDE_FLOAT32_C(   915.39) },
      { SIMDE_FLOAT32_C(-559.539978), SIMDE_FLOAT32_C(1591.790039), SIMDE_FLOAT32_C(1213.340088), SIMDE_FLOAT32_C(351.440002) } },
    { { SIMDE_FLOAT32_C(    54.20), SIMDE_FLOAT32_C(  -928.06), SIMDE_FLOAT32_C(   362.39), SIMDE_FLOAT32_C(  -936.63) },
      { SIMDE_FLOAT32_C(   185.82), SIMDE_FLOAT32_C(  -244.43), SIMDE_FLOAT32_C(   924.66), SIMDE_FLOAT32_C(  -643.82) },
      { SIMDE_FLOAT32_C(-190.229996), SIMDE_FLOAT32_C(-1113.880005), SIMDE_FLOAT32_C(-281.429993), SIMDE_FLOAT32_C(-1861.290039) } },
    { { SIMDE_FLOAT32_C(  -516.92), SIMDE_FLOAT32_C(  -615.16), SIMDE_FLOAT32_C(  -751.52), SIMDE_FLOAT32_C(  -974.04) },
      { SIMDE_FLOAT32_C(  -144.42), SIMDE_FLOAT32_C(   338.27), SIMDE_FLOAT32_C(   704.92), SIMDE_FLOAT32_C(   116.90) },
      { SIMDE_FLOAT32_C(-178.649994), SIMDE_FLOAT32_C(-470.739990), SIMDE_FLOAT32_C(-634.619995), SIMDE_FLOAT32_C(-1678.959961) } },
    { { SIMDE_FLOAT32_C(    49.39), SIMDE_FLOAT32_C(  -363.00), SIMDE_FLOAT32_C(  -476.30), SIMDE_FLOAT32_C(   106.71) },
      { SIMDE_FLOAT32_C(  -725.84), SIMDE_FLOAT32_C(  -353.71), SIMDE_FLOAT32_C(   268.41), SIMDE_FLOAT32_C(   728.83) },
      { SIMDE_FLOAT32_C(-304.320007), SIMDE_FLOAT32_C(362.840027), SIMDE_FLOAT32_C(252.530029), SIMDE_FLOAT32_C(-161.700012) } },
    { { SIMDE_FLOAT32_C(  -470.26), SIMDE_FLOAT32_C(    81.87), SIMDE_FLOAT32_C(    72.27), SIMDE_FLOAT32_C(  -992.59) },
      { SIMDE_FLOAT32_C(  -615.67), SIMDE_FLOAT32_C(   620.28), SIMDE_FLOAT32_C(  -606.51), SIMDE_FLOAT32_C(  -327.77) },
      { SIMDE_FLOAT32_C(150.020020), SIMDE_FLOAT32_C(697.539978), SIMDE_FLOAT32_C(-255.500000), SIMDE_FLOAT32_C(-386.080017) } },
    { { SIMDE_FLOAT32_C(  -331.80), SIMDE_FLOAT32_C(  -605.94), SIMDE_FLOAT32_C(  -295.86), SIMDE_FLOAT32_C(  -275.42) },
      { SIMDE_FLOAT32_C(  -752.64), SIMDE_FLOAT32_C(    21.91), SIMDE_FLOAT32_C(   826.83), SIMDE_FLOAT32_C(   599.82) },
      { SIMDE_FLOAT32_C(-309.889984), SIMDE_FLOAT32_C(146.700012), SIMDE_FLOAT32_C(303.960022), SIMDE_FLOAT32_C(-1102.250000) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_float32x4_t a = simde_vld1q_f32(test_vec[i].a);
    simde_float32x4_t b = simde_vld1q_f32(test_vec[i].b);
    simde_float32x4_t r = simde_vcaddq_rot270_f32(a, b);
    simde_test_arm_neon_assert_equal_f32x4(r, simde_vld1q_f32(test_vec[i].r), 1);
  }
  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_float32x4_t a = simde_test_arm_neon_random_f32x4(-1000.0f, 1000.0f);
    simde_float32x4_t b = simde_test_arm_neon_random_f32x4(-1000.0f, 1000.0f);
    simde_float32x4_t r = simde_vcaddq_rot270_f32(a, b);

    simde_test_arm_neon_write_f32x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_f32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_f32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_vcaddq_rot270_f64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    simde_float64 a[2];
    simde_float64 b[2];
    simde_float64 r[2];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   -30.36), SIMDE_FLOAT64_C(   631.53) },
      { SIMDE_FLOAT64_C(   850.75), SIMDE_FLOAT64_C(  -263.55) },
      { SIMDE_FLOAT64_C(-293.910000), SIMDE_FLOAT64_C(-219.220000) } },
    { { SIMDE_FLOAT64_C(   139.96), SIMDE_FLOAT64_C(   859.14) },
      { SIMDE_FLOAT64_C(  -834.47), SIMDE_FLOAT64_C(   216.10) },
      { SIMDE_FLOAT64_C(356.060000), SIMDE_FLOAT64_C(1693.610000) } },
    { { SIMDE_FLOAT64_C(   995.86), SIMDE_FLOAT64_C(   529.74) },
      { SIMDE_FLOAT64_C(    79.08), SIMDE_FLOAT64_C(   947.13) },
      { SIMDE_FLOAT64_C(1942.990000), SIMDE_FLOAT64_C(450.660000) } },
    { { SIMDE_FLOAT64_C(   122.02), SIMDE_FLOAT64_C(  -250.00) },
      { SIMDE_FLOAT64_C(  -361.82), SIMDE_FLOAT64_C(   265.24) },
      { SIMDE_FLOAT64_C(387.260000), SIMDE_FLOAT64_C(111.820000) } },
    { { SIMDE_FLOAT64_C(   275.71), SIMDE_FLOAT64_C(     2.71) },
      { SIMDE_FLOAT64_C(    99.79), SIMDE_FLOAT64_C(  -137.67) },
      { SIMDE_FLOAT64_C(138.040000), SIMDE_FLOAT64_C(-97.080000) } },
    { { SIMDE_FLOAT64_C(  -761.19), SIMDE_FLOAT64_C(   813.19) },
      { SIMDE_FLOAT64_C(  -897.68), SIMDE_FLOAT64_C(   653.58) },
      { SIMDE_FLOAT64_C(-107.610000), SIMDE_FLOAT64_C(1710.870000) } },
    { { SIMDE_FLOAT64_C(   396.02), SIMDE_FLOAT64_C(   413.06) },
      { SIMDE_FLOAT64_C(   514.09), SIMDE_FLOAT64_C(  -977.67) },
      { SIMDE_FLOAT64_C(-581.650000), SIMDE_FLOAT64_C(-101.030000) } },
    { { SIMDE_FLOAT64_C(  -671.79), SIMDE_FLOAT64_C(   -92.13) },
      { SIMDE_FLOAT64_C(  -441.32), SIMDE_FLOAT64_C(  -374.27) },
      { SIMDE_FLOAT64_C(-1046.060000), SIMDE_FLOAT64_C(349.190000) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde_float64x2_t a = simde_vld1q_f64(test_vec[i].a);
    simde_float64x2_t b = simde_vld1q_f64(test_vec[i].b);
    simde_float64x2_t r = simde_vcaddq_rot270_f64(a, b);
    simde_test_arm_neon_assert_equal_f64x2(r, simde_vld1q_f64(test_vec[i].r), 1);
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde_float64x2_t a = simde_test_arm_neon_random_f64x2(-1000.0f, 1000.0f);
    simde_float64x2_t b = simde_test_arm_neon_random_f64x2(-1000.0f, 1000.0f);
    simde_float64x2_t r = simde_vcaddq_rot270_f64(a, b);

    simde_test_arm_neon_write_f64x2(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_arm_neon_write_f64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_arm_neon_write_f64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
SIMDE_TEST_FUNC_LIST_ENTRY(vcadd_rot270_f16)
SIMDE_TEST_FUNC_LIST_ENTRY(vcaddq_rot270_f16)
SIMDE_TEST_FUNC_LIST_ENTRY(vcadd_rot270_f32)
SIMDE_TEST_FUNC_LIST_ENTRY(vcaddq_rot270_f32)
SIMDE_TEST_FUNC_LIST_ENTRY(vcaddq_rot270_f64)
SIMDE_TEST_FUNC_LIST_END

#include "test-neon-footer.h"
