/* Copyright (C) 1993,1994 by the author(s).
 
 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with ShapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 * AtFS -- Attribute Filesystem
 *
 * config.c -- system dependent definitions
 *
 * Author: Andreas Lampen (Andreas.Lampen@cs.tu-berlin.de)
 *
 * $Header: config.c[7.0] Fri Jun 25 14:32:35 1993 andy@cs.tu-berlin.de frozen $
 */

#include "config.h"

/*=========================
 * symlink
 *=========================*/

#if defined(NEED_SYMLINK)
int symlink (origpath, name) char *origpath, *name; {
  /*
   * This is a faked BSD-type system-call to create a symbolic link.
   * What actually happens is that a file "name" will be created that
   * contains "origpath". This type of pseudo symbolic link is supported
   * by AtFS.
   */
  FILE *slink;

  if ((slink = fopen (name, "w")) != NULL) {
    fprintf (slink, "%s", origpath);
    fclose (slink);
    return 0;
  }
  return -1;
}
#endif

/*=========================
 * regex
 *=========================*/

#if defined(NEED_REGEX)
extern char *regcmp(), *regex();

static char *re_pat;            /* previously compiled pattern */

char *re_comp (s)
     char *s;
{
  if (re_pat && *re_pat)
    free(re_pat);

  if ((re_pat = regcmp(s, (char *) NULL)) == (char *) NULL)
    return "Invalid argument";

  return (char *) NULL;
}

int re_exec(s)
     char *s;
{
  if (re_pat && *re_pat)
    return regex(re_pat, s) == (char *) NULL ? 0 : 1;
  else
    return 0;
}
#endif

/*=========================
 * disable/enable signals
 *=========================*/

#if !defined(NO_SIGSET) /* Posix */
static sigset_t signalSet;
static int      firstTime = 1;
#endif

void disableSig (signo)
     int signo;
{
#if !defined(NO_SIGSET)
  if (firstTime) {
    sigemptyset (&signalSet);
    firstTime = 0;
  }
  sigaddset (&signalSet, signo);
  sigprocmask (SIG_BLOCK, &signalSet, NULL);
#else
  sigblock (sigmask(signo));
#endif
}

void enableSig () {
#if !defined(NO_SIGSET)
  sigprocmask (SIG_UNBLOCK, &signalSet, NULL);
  sigemptyset (&signalSet);
#else
  sigsetmask (0);
#endif
}

/*==================
 *  strdup
 *==================*/

#if defined(NEED_STRDUP)
#ifdef __STDC__
char *strdup (const char *str)
#else
char *strdup (str)
     char *str;
#endif
{
  char *newStr = malloc (strlen(str)+1);
  if (newStr)
    strcpy (newStr, str);
  return (newStr);
}
#endif

/*==================
 *  getcwd
 *==================*/

#if defined(NEED_GETCWD)
char *getcwd (buf, size)
     char   *buf;
     size_t size;
{
  char *getwd();
  if (!buf)
    if ((buf = malloc (size)) == NULL)
      return (NULL);

  return (getwd (buf));
}
#endif

/*===============
 *  mntent
 *===============*/

#if defined(NEED_MNTENT)

#include <fstab.h>

#ifdef __STDC__
FILE *atSetmntent (const char *path, const char *type)
#else
FILE *atSetmntent (path, type)
     char *path, *type;
#endif
{
  setfsent ();
  return ((FILE *)1);
}

struct atMntent *atGetmntent (file)
     FILE *file;
{
  static struct atMntent mountEntry;
  struct fstab *fstabEntry;

  if (fstabEntry = getfsent ()) {
    mountEntry.at_mnt_fsname = fstabEntry->fs_spec;
    mountEntry.at_mnt_dir = fstabEntry->fs_file;
    mountEntry.at_mnt_type = fstabEntry->fs_type;
    mountEntry.at_mnt_opts = "";
    mountEntry.at_mnt_freq = fstabEntry->fs_freq;
    mountEntry.at_mnt_passno = fstabEntry->fs_passno;
    return (&mountEntry);
  }
  return (NULL);
}

int atEndmntent (file)
     FILE *file;
{
  endfsent ();
}
#elif defined(HAS_MNTTAB)

struct atMntent *atGetmntent (mountTable)
     FILE *mountTable;
{
  static struct atMntent mountEntry;
  if (getmntent (mountTable, &mountEntry))
    return (&mountEntry);
  return (NULL);
}
#endif
