// ScintillaEdit.cpp
// Extended version of ScintillaEditBase with a method for each API
// Copyright (c) 2011 Archaeopteryx Software, Inc. d/b/a Wingware

#include "ScintillaEdit.h"

#ifdef SCI_NAMESPACE
using namespace Scintilla;
#endif

ScintillaEdit::ScintillaEdit(QWidget *parent) : ScintillaEditBase(parent) {
}

ScintillaEdit::~ScintillaEdit() {
}

QByteArray ScintillaEdit::TextReturner(int message, uptr_t wParam) const {
    int length = send(message, wParam, 0);
    QByteArray ba(length, '\0');
    send(message, wParam, (sptr_t)ba.data());
    // Remove extra NULs
    if (ba.size() > 0 && ba.at(ba.size()-1) == 0)
        ba.chop(1);
    return ba;
}

QPair<int, int>ScintillaEdit::find_text(int flags, const char *text, int cpMin, int cpMax) {
    struct Sci_TextToFind ft = {{0, 0}, 0, {0, 0}};
    ft.chrg.cpMin = cpMin;
    ft.chrg.cpMax = cpMax;
    ft.chrgText.cpMin = cpMin;
    ft.chrgText.cpMax = cpMax;
    ft.lpstrText = const_cast<char *>(text);

    int start = send(SCI_FINDTEXT, flags, (uptr_t) (&ft));

    return QPair<int,int>(start, ft.chrgText.cpMax);
}

QByteArray ScintillaEdit::get_text_range(int start, int end) {
    if (start > end)
        start = end;

    int length = end-start;
    QByteArray ba(length+1, '\0');
    struct Sci_TextRange tr = {{start, end}, ba.data()};

    send(SCI_GETTEXTRANGE, 0, (sptr_t)&tr);
    ba.chop(1); // Remove extra NUL

    return ba;
}

ScintillaDocument *ScintillaEdit::get_doc() {
    return new ScintillaDocument(0, (void *)send(SCI_GETDOCPOINTER, 0, 0));
}

void ScintillaEdit::set_doc(ScintillaDocument *pdoc_) {
    send(SCI_SETDOCPOINTER, 0, (sptr_t)(pdoc_->pointer()));
}

long ScintillaEdit::format_range(bool draw, QPaintDevice* target, QPaintDevice* measure,
                                 const QRect& print_rect, const QRect& page_rect,
                                 long range_start, long range_end)
{
    Sci_RangeToFormat to_format;

    to_format.hdc = target;
    to_format.hdcTarget = measure;

    to_format.rc.left = print_rect.left();
    to_format.rc.top = print_rect.top();
    to_format.rc.right = print_rect.right();
    to_format.rc.bottom = print_rect.bottom();

    to_format.rcPage.left = page_rect.left();
    to_format.rcPage.top = page_rect.top();
    to_format.rcPage.right = page_rect.right();
    to_format.rcPage.bottom = page_rect.bottom();

    to_format.chrg.cpMin = range_start;
    to_format.chrg.cpMax = range_end;

    return send(SCI_FORMATRANGE, draw, reinterpret_cast<sptr_t>(&to_format));
}

/* ++Autogenerated -- start of section automatically generated from Scintilla.iface */
void ScintillaEdit::addText(sptr_t length, const char * text) {
    send(SCI_ADDTEXT, length, (sptr_t)text);
}

void ScintillaEdit::addStyledText(sptr_t length, const char * c) {
    send(SCI_ADDSTYLEDTEXT, length, (sptr_t)c);
}

void ScintillaEdit::insertText(sptr_t pos, const char * text) {
    send(SCI_INSERTTEXT, pos, (sptr_t)text);
}

void ScintillaEdit::changeInsertion(sptr_t length, const char * text) {
    send(SCI_CHANGEINSERTION, length, (sptr_t)text);
}

void ScintillaEdit::clearAll() {
    send(SCI_CLEARALL, 0, 0);
}

void ScintillaEdit::deleteRange(sptr_t start, sptr_t lengthDelete) {
    send(SCI_DELETERANGE, start, lengthDelete);
}

void ScintillaEdit::clearDocumentStyle() {
    send(SCI_CLEARDOCUMENTSTYLE, 0, 0);
}

sptr_t ScintillaEdit::length() const {
    return send(SCI_GETLENGTH, 0, 0);
}

sptr_t ScintillaEdit::charAt(sptr_t pos) const {
    return send(SCI_GETCHARAT, pos, 0);
}

sptr_t ScintillaEdit::currentPos() const {
    return send(SCI_GETCURRENTPOS, 0, 0);
}

sptr_t ScintillaEdit::anchor() const {
    return send(SCI_GETANCHOR, 0, 0);
}

sptr_t ScintillaEdit::styleAt(sptr_t pos) const {
    return send(SCI_GETSTYLEAT, pos, 0);
}

void ScintillaEdit::redo() {
    send(SCI_REDO, 0, 0);
}

void ScintillaEdit::setUndoCollection(bool collectUndo) {
    send(SCI_SETUNDOCOLLECTION, collectUndo, 0);
}

void ScintillaEdit::selectAll() {
    send(SCI_SELECTALL, 0, 0);
}

void ScintillaEdit::setSavePoint() {
    send(SCI_SETSAVEPOINT, 0, 0);
}

bool ScintillaEdit::canRedo() {
    return send(SCI_CANREDO, 0, 0);
}

sptr_t ScintillaEdit::markerLineFromHandle(sptr_t markerHandle) {
    return send(SCI_MARKERLINEFROMHANDLE, markerHandle, 0);
}

void ScintillaEdit::markerDeleteHandle(sptr_t markerHandle) {
    send(SCI_MARKERDELETEHANDLE, markerHandle, 0);
}

bool ScintillaEdit::undoCollection() const {
    return send(SCI_GETUNDOCOLLECTION, 0, 0);
}

sptr_t ScintillaEdit::viewWS() const {
    return send(SCI_GETVIEWWS, 0, 0);
}

void ScintillaEdit::setViewWS(sptr_t viewWS) {
    send(SCI_SETVIEWWS, viewWS, 0);
}

sptr_t ScintillaEdit::tabDrawMode() const {
    return send(SCI_GETTABDRAWMODE, 0, 0);
}

void ScintillaEdit::setTabDrawMode(sptr_t tabDrawMode) {
    send(SCI_SETTABDRAWMODE, tabDrawMode, 0);
}

sptr_t ScintillaEdit::positionFromPoint(sptr_t x, sptr_t y) {
    return send(SCI_POSITIONFROMPOINT, x, y);
}

sptr_t ScintillaEdit::positionFromPointClose(sptr_t x, sptr_t y) {
    return send(SCI_POSITIONFROMPOINTCLOSE, x, y);
}

void ScintillaEdit::gotoLine(sptr_t line) {
    send(SCI_GOTOLINE, line, 0);
}

void ScintillaEdit::gotoPos(sptr_t caret) {
    send(SCI_GOTOPOS, caret, 0);
}

void ScintillaEdit::setAnchor(sptr_t anchor) {
    send(SCI_SETANCHOR, anchor, 0);
}

QByteArray ScintillaEdit::getCurLine(sptr_t length) {
    return TextReturner(SCI_GETCURLINE, length);
}

sptr_t ScintillaEdit::endStyled() const {
    return send(SCI_GETENDSTYLED, 0, 0);
}

void ScintillaEdit::convertEOLs(sptr_t eolMode) {
    send(SCI_CONVERTEOLS, eolMode, 0);
}

sptr_t ScintillaEdit::eOLMode() const {
    return send(SCI_GETEOLMODE, 0, 0);
}

void ScintillaEdit::setEOLMode(sptr_t eolMode) {
    send(SCI_SETEOLMODE, eolMode, 0);
}

void ScintillaEdit::startStyling(sptr_t start, sptr_t unused) {
    send(SCI_STARTSTYLING, start, unused);
}

void ScintillaEdit::setStyling(sptr_t length, sptr_t style) {
    send(SCI_SETSTYLING, length, style);
}

bool ScintillaEdit::bufferedDraw() const {
    return send(SCI_GETBUFFEREDDRAW, 0, 0);
}

void ScintillaEdit::setBufferedDraw(bool buffered) {
    send(SCI_SETBUFFEREDDRAW, buffered, 0);
}

void ScintillaEdit::setTabWidth(sptr_t tabWidth) {
    send(SCI_SETTABWIDTH, tabWidth, 0);
}

sptr_t ScintillaEdit::tabWidth() const {
    return send(SCI_GETTABWIDTH, 0, 0);
}

void ScintillaEdit::clearTabStops(sptr_t line) {
    send(SCI_CLEARTABSTOPS, line, 0);
}

void ScintillaEdit::addTabStop(sptr_t line, sptr_t x) {
    send(SCI_ADDTABSTOP, line, x);
}

sptr_t ScintillaEdit::getNextTabStop(sptr_t line, sptr_t x) {
    return send(SCI_GETNEXTTABSTOP, line, x);
}

void ScintillaEdit::setCodePage(sptr_t codePage) {
    send(SCI_SETCODEPAGE, codePage, 0);
}

sptr_t ScintillaEdit::iMEInteraction() const {
    return send(SCI_GETIMEINTERACTION, 0, 0);
}

void ScintillaEdit::setIMEInteraction(sptr_t imeInteraction) {
    send(SCI_SETIMEINTERACTION, imeInteraction, 0);
}

void ScintillaEdit::markerDefine(sptr_t markerNumber, sptr_t markerSymbol) {
    send(SCI_MARKERDEFINE, markerNumber, markerSymbol);
}

void ScintillaEdit::markerSetFore(sptr_t markerNumber, sptr_t fore) {
    send(SCI_MARKERSETFORE, markerNumber, fore);
}

void ScintillaEdit::markerSetBack(sptr_t markerNumber, sptr_t back) {
    send(SCI_MARKERSETBACK, markerNumber, back);
}

void ScintillaEdit::markerSetBackSelected(sptr_t markerNumber, sptr_t back) {
    send(SCI_MARKERSETBACKSELECTED, markerNumber, back);
}

void ScintillaEdit::markerEnableHighlight(bool enabled) {
    send(SCI_MARKERENABLEHIGHLIGHT, enabled, 0);
}

sptr_t ScintillaEdit::markerAdd(sptr_t line, sptr_t markerNumber) {
    return send(SCI_MARKERADD, line, markerNumber);
}

void ScintillaEdit::markerDelete(sptr_t line, sptr_t markerNumber) {
    send(SCI_MARKERDELETE, line, markerNumber);
}

void ScintillaEdit::markerDeleteAll(sptr_t markerNumber) {
    send(SCI_MARKERDELETEALL, markerNumber, 0);
}

sptr_t ScintillaEdit::markerGet(sptr_t line) {
    return send(SCI_MARKERGET, line, 0);
}

sptr_t ScintillaEdit::markerNext(sptr_t lineStart, sptr_t markerMask) {
    return send(SCI_MARKERNEXT, lineStart, markerMask);
}

sptr_t ScintillaEdit::markerPrevious(sptr_t lineStart, sptr_t markerMask) {
    return send(SCI_MARKERPREVIOUS, lineStart, markerMask);
}

void ScintillaEdit::markerDefinePixmap(sptr_t markerNumber, const char * pixmap) {
    send(SCI_MARKERDEFINEPIXMAP, markerNumber, (sptr_t)pixmap);
}

void ScintillaEdit::markerAddSet(sptr_t line, sptr_t markerSet) {
    send(SCI_MARKERADDSET, line, markerSet);
}

void ScintillaEdit::markerSetAlpha(sptr_t markerNumber, sptr_t alpha) {
    send(SCI_MARKERSETALPHA, markerNumber, alpha);
}

void ScintillaEdit::setMarginTypeN(sptr_t margin, sptr_t marginType) {
    send(SCI_SETMARGINTYPEN, margin, marginType);
}

sptr_t ScintillaEdit::marginTypeN(sptr_t margin) const {
    return send(SCI_GETMARGINTYPEN, margin, 0);
}

void ScintillaEdit::setMarginWidthN(sptr_t margin, sptr_t pixelWidth) {
    send(SCI_SETMARGINWIDTHN, margin, pixelWidth);
}

sptr_t ScintillaEdit::marginWidthN(sptr_t margin) const {
    return send(SCI_GETMARGINWIDTHN, margin, 0);
}

void ScintillaEdit::setMarginMaskN(sptr_t margin, sptr_t mask) {
    send(SCI_SETMARGINMASKN, margin, mask);
}

sptr_t ScintillaEdit::marginMaskN(sptr_t margin) const {
    return send(SCI_GETMARGINMASKN, margin, 0);
}

void ScintillaEdit::setMarginSensitiveN(sptr_t margin, bool sensitive) {
    send(SCI_SETMARGINSENSITIVEN, margin, sensitive);
}

bool ScintillaEdit::marginSensitiveN(sptr_t margin) const {
    return send(SCI_GETMARGINSENSITIVEN, margin, 0);
}

void ScintillaEdit::setMarginCursorN(sptr_t margin, sptr_t cursor) {
    send(SCI_SETMARGINCURSORN, margin, cursor);
}

sptr_t ScintillaEdit::marginCursorN(sptr_t margin) const {
    return send(SCI_GETMARGINCURSORN, margin, 0);
}

void ScintillaEdit::setMarginBackN(sptr_t margin, sptr_t back) {
    send(SCI_SETMARGINBACKN, margin, back);
}

sptr_t ScintillaEdit::marginBackN(sptr_t margin) const {
    return send(SCI_GETMARGINBACKN, margin, 0);
}

void ScintillaEdit::setMargins(sptr_t margins) {
    send(SCI_SETMARGINS, margins, 0);
}

sptr_t ScintillaEdit::margins() const {
    return send(SCI_GETMARGINS, 0, 0);
}

void ScintillaEdit::styleClearAll() {
    send(SCI_STYLECLEARALL, 0, 0);
}

void ScintillaEdit::styleSetFore(sptr_t style, sptr_t fore) {
    send(SCI_STYLESETFORE, style, fore);
}

void ScintillaEdit::styleSetBack(sptr_t style, sptr_t back) {
    send(SCI_STYLESETBACK, style, back);
}

void ScintillaEdit::styleSetBold(sptr_t style, bool bold) {
    send(SCI_STYLESETBOLD, style, bold);
}

void ScintillaEdit::styleSetItalic(sptr_t style, bool italic) {
    send(SCI_STYLESETITALIC, style, italic);
}

void ScintillaEdit::styleSetSize(sptr_t style, sptr_t sizePoints) {
    send(SCI_STYLESETSIZE, style, sizePoints);
}

void ScintillaEdit::styleSetFont(sptr_t style, const char * fontName) {
    send(SCI_STYLESETFONT, style, (sptr_t)fontName);
}

void ScintillaEdit::styleSetEOLFilled(sptr_t style, bool eolFilled) {
    send(SCI_STYLESETEOLFILLED, style, eolFilled);
}

void ScintillaEdit::styleResetDefault() {
    send(SCI_STYLERESETDEFAULT, 0, 0);
}

void ScintillaEdit::styleSetUnderline(sptr_t style, bool underline) {
    send(SCI_STYLESETUNDERLINE, style, underline);
}

sptr_t ScintillaEdit::styleFore(sptr_t style) const {
    return send(SCI_STYLEGETFORE, style, 0);
}

sptr_t ScintillaEdit::styleBack(sptr_t style) const {
    return send(SCI_STYLEGETBACK, style, 0);
}

bool ScintillaEdit::styleBold(sptr_t style) const {
    return send(SCI_STYLEGETBOLD, style, 0);
}

bool ScintillaEdit::styleItalic(sptr_t style) const {
    return send(SCI_STYLEGETITALIC, style, 0);
}

sptr_t ScintillaEdit::styleSize(sptr_t style) const {
    return send(SCI_STYLEGETSIZE, style, 0);
}

QByteArray ScintillaEdit::styleFont(sptr_t style) const {
    return TextReturner(SCI_STYLEGETFONT, style);
}

bool ScintillaEdit::styleEOLFilled(sptr_t style) const {
    return send(SCI_STYLEGETEOLFILLED, style, 0);
}

bool ScintillaEdit::styleUnderline(sptr_t style) const {
    return send(SCI_STYLEGETUNDERLINE, style, 0);
}

sptr_t ScintillaEdit::styleCase(sptr_t style) const {
    return send(SCI_STYLEGETCASE, style, 0);
}

sptr_t ScintillaEdit::styleCharacterSet(sptr_t style) const {
    return send(SCI_STYLEGETCHARACTERSET, style, 0);
}

bool ScintillaEdit::styleVisible(sptr_t style) const {
    return send(SCI_STYLEGETVISIBLE, style, 0);
}

bool ScintillaEdit::styleChangeable(sptr_t style) const {
    return send(SCI_STYLEGETCHANGEABLE, style, 0);
}

bool ScintillaEdit::styleHotSpot(sptr_t style) const {
    return send(SCI_STYLEGETHOTSPOT, style, 0);
}

void ScintillaEdit::styleSetCase(sptr_t style, sptr_t caseVisible) {
    send(SCI_STYLESETCASE, style, caseVisible);
}

void ScintillaEdit::styleSetSizeFractional(sptr_t style, sptr_t sizeHundredthPoints) {
    send(SCI_STYLESETSIZEFRACTIONAL, style, sizeHundredthPoints);
}

sptr_t ScintillaEdit::styleSizeFractional(sptr_t style) const {
    return send(SCI_STYLEGETSIZEFRACTIONAL, style, 0);
}

void ScintillaEdit::styleSetWeight(sptr_t style, sptr_t weight) {
    send(SCI_STYLESETWEIGHT, style, weight);
}

sptr_t ScintillaEdit::styleWeight(sptr_t style) const {
    return send(SCI_STYLEGETWEIGHT, style, 0);
}

void ScintillaEdit::styleSetCharacterSet(sptr_t style, sptr_t characterSet) {
    send(SCI_STYLESETCHARACTERSET, style, characterSet);
}

void ScintillaEdit::styleSetHotSpot(sptr_t style, bool hotspot) {
    send(SCI_STYLESETHOTSPOT, style, hotspot);
}

void ScintillaEdit::setSelFore(bool useSetting, sptr_t fore) {
    send(SCI_SETSELFORE, useSetting, fore);
}

void ScintillaEdit::setSelBack(bool useSetting, sptr_t back) {
    send(SCI_SETSELBACK, useSetting, back);
}

sptr_t ScintillaEdit::selAlpha() const {
    return send(SCI_GETSELALPHA, 0, 0);
}

void ScintillaEdit::setSelAlpha(sptr_t alpha) {
    send(SCI_SETSELALPHA, alpha, 0);
}

bool ScintillaEdit::selEOLFilled() const {
    return send(SCI_GETSELEOLFILLED, 0, 0);
}

void ScintillaEdit::setSelEOLFilled(bool filled) {
    send(SCI_SETSELEOLFILLED, filled, 0);
}

void ScintillaEdit::setCaretFore(sptr_t fore) {
    send(SCI_SETCARETFORE, fore, 0);
}

void ScintillaEdit::assignCmdKey(sptr_t keyDefinition, sptr_t sciCommand) {
    send(SCI_ASSIGNCMDKEY, keyDefinition, sciCommand);
}

void ScintillaEdit::clearCmdKey(sptr_t keyDefinition) {
    send(SCI_CLEARCMDKEY, keyDefinition, 0);
}

void ScintillaEdit::clearAllCmdKeys() {
    send(SCI_CLEARALLCMDKEYS, 0, 0);
}

void ScintillaEdit::setStylingEx(sptr_t length, const char * styles) {
    send(SCI_SETSTYLINGEX, length, (sptr_t)styles);
}

void ScintillaEdit::styleSetVisible(sptr_t style, bool visible) {
    send(SCI_STYLESETVISIBLE, style, visible);
}

sptr_t ScintillaEdit::caretPeriod() const {
    return send(SCI_GETCARETPERIOD, 0, 0);
}

void ScintillaEdit::setCaretPeriod(sptr_t periodMilliseconds) {
    send(SCI_SETCARETPERIOD, periodMilliseconds, 0);
}

void ScintillaEdit::setWordChars(const char * characters) {
    send(SCI_SETWORDCHARS, 0, (sptr_t)characters);
}

QByteArray ScintillaEdit::wordChars() const {
    return TextReturner(SCI_GETWORDCHARS, 0);
}

void ScintillaEdit::beginUndoAction() {
    send(SCI_BEGINUNDOACTION, 0, 0);
}

void ScintillaEdit::endUndoAction() {
    send(SCI_ENDUNDOACTION, 0, 0);
}

void ScintillaEdit::indicSetStyle(sptr_t indicator, sptr_t indicatorStyle) {
    send(SCI_INDICSETSTYLE, indicator, indicatorStyle);
}

sptr_t ScintillaEdit::indicStyle(sptr_t indicator) const {
    return send(SCI_INDICGETSTYLE, indicator, 0);
}

void ScintillaEdit::indicSetFore(sptr_t indicator, sptr_t fore) {
    send(SCI_INDICSETFORE, indicator, fore);
}

sptr_t ScintillaEdit::indicFore(sptr_t indicator) const {
    return send(SCI_INDICGETFORE, indicator, 0);
}

void ScintillaEdit::indicSetUnder(sptr_t indicator, bool under) {
    send(SCI_INDICSETUNDER, indicator, under);
}

bool ScintillaEdit::indicUnder(sptr_t indicator) const {
    return send(SCI_INDICGETUNDER, indicator, 0);
}

void ScintillaEdit::indicSetHoverStyle(sptr_t indicator, sptr_t indicatorStyle) {
    send(SCI_INDICSETHOVERSTYLE, indicator, indicatorStyle);
}

sptr_t ScintillaEdit::indicHoverStyle(sptr_t indicator) const {
    return send(SCI_INDICGETHOVERSTYLE, indicator, 0);
}

void ScintillaEdit::indicSetHoverFore(sptr_t indicator, sptr_t fore) {
    send(SCI_INDICSETHOVERFORE, indicator, fore);
}

sptr_t ScintillaEdit::indicHoverFore(sptr_t indicator) const {
    return send(SCI_INDICGETHOVERFORE, indicator, 0);
}

void ScintillaEdit::indicSetFlags(sptr_t indicator, sptr_t flags) {
    send(SCI_INDICSETFLAGS, indicator, flags);
}

sptr_t ScintillaEdit::indicFlags(sptr_t indicator) const {
    return send(SCI_INDICGETFLAGS, indicator, 0);
}

void ScintillaEdit::setWhitespaceFore(bool useSetting, sptr_t fore) {
    send(SCI_SETWHITESPACEFORE, useSetting, fore);
}

void ScintillaEdit::setWhitespaceBack(bool useSetting, sptr_t back) {
    send(SCI_SETWHITESPACEBACK, useSetting, back);
}

void ScintillaEdit::setWhitespaceSize(sptr_t size) {
    send(SCI_SETWHITESPACESIZE, size, 0);
}

sptr_t ScintillaEdit::whitespaceSize() const {
    return send(SCI_GETWHITESPACESIZE, 0, 0);
}

void ScintillaEdit::setStyleBits(sptr_t bits) {
    send(SCI_SETSTYLEBITS, bits, 0);
}

sptr_t ScintillaEdit::styleBits() const {
    return send(SCI_GETSTYLEBITS, 0, 0);
}

void ScintillaEdit::setLineState(sptr_t line, sptr_t state) {
    send(SCI_SETLINESTATE, line, state);
}

sptr_t ScintillaEdit::lineState(sptr_t line) const {
    return send(SCI_GETLINESTATE, line, 0);
}

sptr_t ScintillaEdit::maxLineState() const {
    return send(SCI_GETMAXLINESTATE, 0, 0);
}

bool ScintillaEdit::caretLineVisible() const {
    return send(SCI_GETCARETLINEVISIBLE, 0, 0);
}

void ScintillaEdit::setCaretLineVisible(bool show) {
    send(SCI_SETCARETLINEVISIBLE, show, 0);
}

sptr_t ScintillaEdit::caretLineBack() const {
    return send(SCI_GETCARETLINEBACK, 0, 0);
}

void ScintillaEdit::setCaretLineBack(sptr_t back) {
    send(SCI_SETCARETLINEBACK, back, 0);
}

void ScintillaEdit::styleSetChangeable(sptr_t style, bool changeable) {
    send(SCI_STYLESETCHANGEABLE, style, changeable);
}

void ScintillaEdit::autoCShow(sptr_t lengthEntered, const char * itemList) {
    send(SCI_AUTOCSHOW, lengthEntered, (sptr_t)itemList);
}

void ScintillaEdit::autoCCancel() {
    send(SCI_AUTOCCANCEL, 0, 0);
}

bool ScintillaEdit::autoCActive() {
    return send(SCI_AUTOCACTIVE, 0, 0);
}

sptr_t ScintillaEdit::autoCPosStart() {
    return send(SCI_AUTOCPOSSTART, 0, 0);
}

void ScintillaEdit::autoCComplete() {
    send(SCI_AUTOCCOMPLETE, 0, 0);
}

void ScintillaEdit::autoCStops(const char * characterSet) {
    send(SCI_AUTOCSTOPS, 0, (sptr_t)characterSet);
}

void ScintillaEdit::autoCSetSeparator(sptr_t separatorCharacter) {
    send(SCI_AUTOCSETSEPARATOR, separatorCharacter, 0);
}

sptr_t ScintillaEdit::autoCSeparator() const {
    return send(SCI_AUTOCGETSEPARATOR, 0, 0);
}

void ScintillaEdit::autoCSelect(const char * select) {
    send(SCI_AUTOCSELECT, 0, (sptr_t)select);
}

void ScintillaEdit::autoCSetCancelAtStart(bool cancel) {
    send(SCI_AUTOCSETCANCELATSTART, cancel, 0);
}

bool ScintillaEdit::autoCCancelAtStart() const {
    return send(SCI_AUTOCGETCANCELATSTART, 0, 0);
}

void ScintillaEdit::autoCSetFillUps(const char * characterSet) {
    send(SCI_AUTOCSETFILLUPS, 0, (sptr_t)characterSet);
}

void ScintillaEdit::autoCSetChooseSingle(bool chooseSingle) {
    send(SCI_AUTOCSETCHOOSESINGLE, chooseSingle, 0);
}

bool ScintillaEdit::autoCChooseSingle() const {
    return send(SCI_AUTOCGETCHOOSESINGLE, 0, 0);
}

void ScintillaEdit::autoCSetIgnoreCase(bool ignoreCase) {
    send(SCI_AUTOCSETIGNORECASE, ignoreCase, 0);
}

bool ScintillaEdit::autoCIgnoreCase() const {
    return send(SCI_AUTOCGETIGNORECASE, 0, 0);
}

void ScintillaEdit::userListShow(sptr_t listType, const char * itemList) {
    send(SCI_USERLISTSHOW, listType, (sptr_t)itemList);
}

void ScintillaEdit::autoCSetAutoHide(bool autoHide) {
    send(SCI_AUTOCSETAUTOHIDE, autoHide, 0);
}

bool ScintillaEdit::autoCAutoHide() const {
    return send(SCI_AUTOCGETAUTOHIDE, 0, 0);
}

void ScintillaEdit::autoCSetDropRestOfWord(bool dropRestOfWord) {
    send(SCI_AUTOCSETDROPRESTOFWORD, dropRestOfWord, 0);
}

bool ScintillaEdit::autoCDropRestOfWord() const {
    return send(SCI_AUTOCGETDROPRESTOFWORD, 0, 0);
}

void ScintillaEdit::registerImage(sptr_t type, const char * xpmData) {
    send(SCI_REGISTERIMAGE, type, (sptr_t)xpmData);
}

void ScintillaEdit::clearRegisteredImages() {
    send(SCI_CLEARREGISTEREDIMAGES, 0, 0);
}

sptr_t ScintillaEdit::autoCTypeSeparator() const {
    return send(SCI_AUTOCGETTYPESEPARATOR, 0, 0);
}

void ScintillaEdit::autoCSetTypeSeparator(sptr_t separatorCharacter) {
    send(SCI_AUTOCSETTYPESEPARATOR, separatorCharacter, 0);
}

void ScintillaEdit::autoCSetMaxWidth(sptr_t characterCount) {
    send(SCI_AUTOCSETMAXWIDTH, characterCount, 0);
}

sptr_t ScintillaEdit::autoCMaxWidth() const {
    return send(SCI_AUTOCGETMAXWIDTH, 0, 0);
}

void ScintillaEdit::autoCSetMaxHeight(sptr_t rowCount) {
    send(SCI_AUTOCSETMAXHEIGHT, rowCount, 0);
}

sptr_t ScintillaEdit::autoCMaxHeight() const {
    return send(SCI_AUTOCGETMAXHEIGHT, 0, 0);
}

void ScintillaEdit::setIndent(sptr_t indentSize) {
    send(SCI_SETINDENT, indentSize, 0);
}

sptr_t ScintillaEdit::indent() const {
    return send(SCI_GETINDENT, 0, 0);
}

void ScintillaEdit::setUseTabs(bool useTabs) {
    send(SCI_SETUSETABS, useTabs, 0);
}

bool ScintillaEdit::useTabs() const {
    return send(SCI_GETUSETABS, 0, 0);
}

void ScintillaEdit::setLineIndentation(sptr_t line, sptr_t indentation) {
    send(SCI_SETLINEINDENTATION, line, indentation);
}

sptr_t ScintillaEdit::lineIndentation(sptr_t line) const {
    return send(SCI_GETLINEINDENTATION, line, 0);
}

sptr_t ScintillaEdit::lineIndentPosition(sptr_t line) const {
    return send(SCI_GETLINEINDENTPOSITION, line, 0);
}

sptr_t ScintillaEdit::column(sptr_t pos) const {
    return send(SCI_GETCOLUMN, pos, 0);
}

sptr_t ScintillaEdit::countCharacters(sptr_t start, sptr_t end) {
    return send(SCI_COUNTCHARACTERS, start, end);
}

void ScintillaEdit::setHScrollBar(bool visible) {
    send(SCI_SETHSCROLLBAR, visible, 0);
}

bool ScintillaEdit::hScrollBar() const {
    return send(SCI_GETHSCROLLBAR, 0, 0);
}

void ScintillaEdit::setIndentationGuides(sptr_t indentView) {
    send(SCI_SETINDENTATIONGUIDES, indentView, 0);
}

sptr_t ScintillaEdit::indentationGuides() const {
    return send(SCI_GETINDENTATIONGUIDES, 0, 0);
}

void ScintillaEdit::setHighlightGuide(sptr_t column) {
    send(SCI_SETHIGHLIGHTGUIDE, column, 0);
}

sptr_t ScintillaEdit::highlightGuide() const {
    return send(SCI_GETHIGHLIGHTGUIDE, 0, 0);
}

sptr_t ScintillaEdit::lineEndPosition(sptr_t line) const {
    return send(SCI_GETLINEENDPOSITION, line, 0);
}

sptr_t ScintillaEdit::codePage() const {
    return send(SCI_GETCODEPAGE, 0, 0);
}

sptr_t ScintillaEdit::caretFore() const {
    return send(SCI_GETCARETFORE, 0, 0);
}

bool ScintillaEdit::readOnly() const {
    return send(SCI_GETREADONLY, 0, 0);
}

void ScintillaEdit::setCurrentPos(sptr_t caret) {
    send(SCI_SETCURRENTPOS, caret, 0);
}

void ScintillaEdit::setSelectionStart(sptr_t anchor) {
    send(SCI_SETSELECTIONSTART, anchor, 0);
}

sptr_t ScintillaEdit::selectionStart() const {
    return send(SCI_GETSELECTIONSTART, 0, 0);
}

void ScintillaEdit::setSelectionEnd(sptr_t caret) {
    send(SCI_SETSELECTIONEND, caret, 0);
}

sptr_t ScintillaEdit::selectionEnd() const {
    return send(SCI_GETSELECTIONEND, 0, 0);
}

void ScintillaEdit::setEmptySelection(sptr_t caret) {
    send(SCI_SETEMPTYSELECTION, caret, 0);
}

void ScintillaEdit::setPrintMagnification(sptr_t magnification) {
    send(SCI_SETPRINTMAGNIFICATION, magnification, 0);
}

sptr_t ScintillaEdit::printMagnification() const {
    return send(SCI_GETPRINTMAGNIFICATION, 0, 0);
}

void ScintillaEdit::setPrintColourMode(sptr_t mode) {
    send(SCI_SETPRINTCOLOURMODE, mode, 0);
}

sptr_t ScintillaEdit::printColourMode() const {
    return send(SCI_GETPRINTCOLOURMODE, 0, 0);
}

sptr_t ScintillaEdit::firstVisibleLine() const {
    return send(SCI_GETFIRSTVISIBLELINE, 0, 0);
}

QByteArray ScintillaEdit::getLine(sptr_t line) {
    return TextReturner(SCI_GETLINE, line);
}

sptr_t ScintillaEdit::lineCount() const {
    return send(SCI_GETLINECOUNT, 0, 0);
}

void ScintillaEdit::setMarginLeft(sptr_t pixelWidth) {
    send(SCI_SETMARGINLEFT, 0, pixelWidth);
}

sptr_t ScintillaEdit::marginLeft() const {
    return send(SCI_GETMARGINLEFT, 0, 0);
}

void ScintillaEdit::setMarginRight(sptr_t pixelWidth) {
    send(SCI_SETMARGINRIGHT, 0, pixelWidth);
}

sptr_t ScintillaEdit::marginRight() const {
    return send(SCI_GETMARGINRIGHT, 0, 0);
}

bool ScintillaEdit::modify() const {
    return send(SCI_GETMODIFY, 0, 0);
}

void ScintillaEdit::setSel(sptr_t anchor, sptr_t caret) {
    send(SCI_SETSEL, anchor, caret);
}

QByteArray ScintillaEdit::getSelText() {
    return TextReturner(SCI_GETSELTEXT, 0);
}

void ScintillaEdit::hideSelection(bool hide) {
    send(SCI_HIDESELECTION, hide, 0);
}

sptr_t ScintillaEdit::pointXFromPosition(sptr_t pos) {
    return send(SCI_POINTXFROMPOSITION, 0, pos);
}

sptr_t ScintillaEdit::pointYFromPosition(sptr_t pos) {
    return send(SCI_POINTYFROMPOSITION, 0, pos);
}

sptr_t ScintillaEdit::lineFromPosition(sptr_t pos) {
    return send(SCI_LINEFROMPOSITION, pos, 0);
}

sptr_t ScintillaEdit::positionFromLine(sptr_t line) {
    return send(SCI_POSITIONFROMLINE, line, 0);
}

void ScintillaEdit::lineScroll(sptr_t columns, sptr_t lines) {
    send(SCI_LINESCROLL, columns, lines);
}

void ScintillaEdit::scrollCaret() {
    send(SCI_SCROLLCARET, 0, 0);
}

void ScintillaEdit::scrollRange(sptr_t secondary, sptr_t primary) {
    send(SCI_SCROLLRANGE, secondary, primary);
}

void ScintillaEdit::replaceSel(const char * text) {
    send(SCI_REPLACESEL, 0, (sptr_t)text);
}

void ScintillaEdit::setReadOnly(bool readOnly) {
    send(SCI_SETREADONLY, readOnly, 0);
}

void ScintillaEdit::null() {
    send(SCI_NULL, 0, 0);
}

bool ScintillaEdit::canPaste() {
    return send(SCI_CANPASTE, 0, 0);
}

bool ScintillaEdit::canUndo() {
    return send(SCI_CANUNDO, 0, 0);
}

void ScintillaEdit::emptyUndoBuffer() {
    send(SCI_EMPTYUNDOBUFFER, 0, 0);
}

void ScintillaEdit::undo() {
    send(SCI_UNDO, 0, 0);
}

void ScintillaEdit::cut() {
    send(SCI_CUT, 0, 0);
}

void ScintillaEdit::copy() {
    send(SCI_COPY, 0, 0);
}

void ScintillaEdit::paste() {
    send(SCI_PASTE, 0, 0);
}

void ScintillaEdit::clear() {
    send(SCI_CLEAR, 0, 0);
}

void ScintillaEdit::setText(const char * text) {
    send(SCI_SETTEXT, 0, (sptr_t)text);
}

QByteArray ScintillaEdit::getText(sptr_t length) {
    return TextReturner(SCI_GETTEXT, length);
}

sptr_t ScintillaEdit::textLength() const {
    return send(SCI_GETTEXTLENGTH, 0, 0);
}

sptr_t ScintillaEdit::directFunction() const {
    return send(SCI_GETDIRECTFUNCTION, 0, 0);
}

sptr_t ScintillaEdit::directPointer() const {
    return send(SCI_GETDIRECTPOINTER, 0, 0);
}

void ScintillaEdit::setOvertype(bool overType) {
    send(SCI_SETOVERTYPE, overType, 0);
}

bool ScintillaEdit::overtype() const {
    return send(SCI_GETOVERTYPE, 0, 0);
}

void ScintillaEdit::setCaretWidth(sptr_t pixelWidth) {
    send(SCI_SETCARETWIDTH, pixelWidth, 0);
}

sptr_t ScintillaEdit::caretWidth() const {
    return send(SCI_GETCARETWIDTH, 0, 0);
}

void ScintillaEdit::setTargetStart(sptr_t start) {
    send(SCI_SETTARGETSTART, start, 0);
}

sptr_t ScintillaEdit::targetStart() const {
    return send(SCI_GETTARGETSTART, 0, 0);
}

void ScintillaEdit::setTargetEnd(sptr_t end) {
    send(SCI_SETTARGETEND, end, 0);
}

sptr_t ScintillaEdit::targetEnd() const {
    return send(SCI_GETTARGETEND, 0, 0);
}

void ScintillaEdit::setTargetRange(sptr_t start, sptr_t end) {
    send(SCI_SETTARGETRANGE, start, end);
}

QByteArray ScintillaEdit::targetText() const {
    return TextReturner(SCI_GETTARGETTEXT, 0);
}

void ScintillaEdit::targetFromSelection() {
    send(SCI_TARGETFROMSELECTION, 0, 0);
}

void ScintillaEdit::targetWholeDocument() {
    send(SCI_TARGETWHOLEDOCUMENT, 0, 0);
}

sptr_t ScintillaEdit::replaceTarget(sptr_t length, const char * text) {
    return send(SCI_REPLACETARGET, length, (sptr_t)text);
}

sptr_t ScintillaEdit::replaceTargetRE(sptr_t length, const char * text) {
    return send(SCI_REPLACETARGETRE, length, (sptr_t)text);
}

sptr_t ScintillaEdit::searchInTarget(sptr_t length, const char * text) {
    return send(SCI_SEARCHINTARGET, length, (sptr_t)text);
}

void ScintillaEdit::setSearchFlags(sptr_t searchFlags) {
    send(SCI_SETSEARCHFLAGS, searchFlags, 0);
}

sptr_t ScintillaEdit::searchFlags() const {
    return send(SCI_GETSEARCHFLAGS, 0, 0);
}

void ScintillaEdit::callTipShow(sptr_t pos, const char * definition) {
    send(SCI_CALLTIPSHOW, pos, (sptr_t)definition);
}

void ScintillaEdit::callTipCancel() {
    send(SCI_CALLTIPCANCEL, 0, 0);
}

bool ScintillaEdit::callTipActive() {
    return send(SCI_CALLTIPACTIVE, 0, 0);
}

sptr_t ScintillaEdit::callTipPosStart() {
    return send(SCI_CALLTIPPOSSTART, 0, 0);
}

void ScintillaEdit::callTipSetPosStart(sptr_t posStart) {
    send(SCI_CALLTIPSETPOSSTART, posStart, 0);
}

void ScintillaEdit::callTipSetHlt(sptr_t highlightStart, sptr_t highlightEnd) {
    send(SCI_CALLTIPSETHLT, highlightStart, highlightEnd);
}

void ScintillaEdit::callTipSetBack(sptr_t back) {
    send(SCI_CALLTIPSETBACK, back, 0);
}

void ScintillaEdit::callTipSetFore(sptr_t fore) {
    send(SCI_CALLTIPSETFORE, fore, 0);
}

void ScintillaEdit::callTipSetForeHlt(sptr_t fore) {
    send(SCI_CALLTIPSETFOREHLT, fore, 0);
}

void ScintillaEdit::callTipUseStyle(sptr_t tabSize) {
    send(SCI_CALLTIPUSESTYLE, tabSize, 0);
}

void ScintillaEdit::callTipSetPosition(bool above) {
    send(SCI_CALLTIPSETPOSITION, above, 0);
}

sptr_t ScintillaEdit::visibleFromDocLine(sptr_t docLine) {
    return send(SCI_VISIBLEFROMDOCLINE, docLine, 0);
}

sptr_t ScintillaEdit::docLineFromVisible(sptr_t displayLine) {
    return send(SCI_DOCLINEFROMVISIBLE, displayLine, 0);
}

sptr_t ScintillaEdit::wrapCount(sptr_t docLine) {
    return send(SCI_WRAPCOUNT, docLine, 0);
}

void ScintillaEdit::setFoldLevel(sptr_t line, sptr_t level) {
    send(SCI_SETFOLDLEVEL, line, level);
}

sptr_t ScintillaEdit::foldLevel(sptr_t line) const {
    return send(SCI_GETFOLDLEVEL, line, 0);
}

sptr_t ScintillaEdit::lastChild(sptr_t line, sptr_t level) const {
    return send(SCI_GETLASTCHILD, line, level);
}

sptr_t ScintillaEdit::foldParent(sptr_t line) const {
    return send(SCI_GETFOLDPARENT, line, 0);
}

void ScintillaEdit::showLines(sptr_t lineStart, sptr_t lineEnd) {
    send(SCI_SHOWLINES, lineStart, lineEnd);
}

void ScintillaEdit::hideLines(sptr_t lineStart, sptr_t lineEnd) {
    send(SCI_HIDELINES, lineStart, lineEnd);
}

bool ScintillaEdit::lineVisible(sptr_t line) const {
    return send(SCI_GETLINEVISIBLE, line, 0);
}

bool ScintillaEdit::allLinesVisible() const {
    return send(SCI_GETALLLINESVISIBLE, 0, 0);
}

void ScintillaEdit::setFoldExpanded(sptr_t line, bool expanded) {
    send(SCI_SETFOLDEXPANDED, line, expanded);
}

bool ScintillaEdit::foldExpanded(sptr_t line) const {
    return send(SCI_GETFOLDEXPANDED, line, 0);
}

void ScintillaEdit::toggleFold(sptr_t line) {
    send(SCI_TOGGLEFOLD, line, 0);
}

void ScintillaEdit::toggleFoldShowText(sptr_t line, const char * text) {
    send(SCI_TOGGLEFOLDSHOWTEXT, line, (sptr_t)text);
}

void ScintillaEdit::foldDisplayTextSetStyle(sptr_t style) {
    send(SCI_FOLDDISPLAYTEXTSETSTYLE, style, 0);
}

void ScintillaEdit::foldLine(sptr_t line, sptr_t action) {
    send(SCI_FOLDLINE, line, action);
}

void ScintillaEdit::foldChildren(sptr_t line, sptr_t action) {
    send(SCI_FOLDCHILDREN, line, action);
}

void ScintillaEdit::expandChildren(sptr_t line, sptr_t level) {
    send(SCI_EXPANDCHILDREN, line, level);
}

void ScintillaEdit::foldAll(sptr_t action) {
    send(SCI_FOLDALL, action, 0);
}

void ScintillaEdit::ensureVisible(sptr_t line) {
    send(SCI_ENSUREVISIBLE, line, 0);
}

void ScintillaEdit::setAutomaticFold(sptr_t automaticFold) {
    send(SCI_SETAUTOMATICFOLD, automaticFold, 0);
}

sptr_t ScintillaEdit::automaticFold() const {
    return send(SCI_GETAUTOMATICFOLD, 0, 0);
}

void ScintillaEdit::setFoldFlags(sptr_t flags) {
    send(SCI_SETFOLDFLAGS, flags, 0);
}

void ScintillaEdit::ensureVisibleEnforcePolicy(sptr_t line) {
    send(SCI_ENSUREVISIBLEENFORCEPOLICY, line, 0);
}

void ScintillaEdit::setTabIndents(bool tabIndents) {
    send(SCI_SETTABINDENTS, tabIndents, 0);
}

bool ScintillaEdit::tabIndents() const {
    return send(SCI_GETTABINDENTS, 0, 0);
}

void ScintillaEdit::setBackSpaceUnIndents(bool bsUnIndents) {
    send(SCI_SETBACKSPACEUNINDENTS, bsUnIndents, 0);
}

bool ScintillaEdit::backSpaceUnIndents() const {
    return send(SCI_GETBACKSPACEUNINDENTS, 0, 0);
}

void ScintillaEdit::setMouseDwellTime(sptr_t periodMilliseconds) {
    send(SCI_SETMOUSEDWELLTIME, periodMilliseconds, 0);
}

sptr_t ScintillaEdit::mouseDwellTime() const {
    return send(SCI_GETMOUSEDWELLTIME, 0, 0);
}

sptr_t ScintillaEdit::wordStartPosition(sptr_t pos, bool onlyWordCharacters) {
    return send(SCI_WORDSTARTPOSITION, pos, onlyWordCharacters);
}

sptr_t ScintillaEdit::wordEndPosition(sptr_t pos, bool onlyWordCharacters) {
    return send(SCI_WORDENDPOSITION, pos, onlyWordCharacters);
}

bool ScintillaEdit::isRangeWord(sptr_t start, sptr_t end) {
    return send(SCI_ISRANGEWORD, start, end);
}

void ScintillaEdit::setIdleStyling(sptr_t idleStyling) {
    send(SCI_SETIDLESTYLING, idleStyling, 0);
}

sptr_t ScintillaEdit::idleStyling() const {
    return send(SCI_GETIDLESTYLING, 0, 0);
}

void ScintillaEdit::setWrapMode(sptr_t wrapMode) {
    send(SCI_SETWRAPMODE, wrapMode, 0);
}

sptr_t ScintillaEdit::wrapMode() const {
    return send(SCI_GETWRAPMODE, 0, 0);
}

void ScintillaEdit::setWrapVisualFlags(sptr_t wrapVisualFlags) {
    send(SCI_SETWRAPVISUALFLAGS, wrapVisualFlags, 0);
}

sptr_t ScintillaEdit::wrapVisualFlags() const {
    return send(SCI_GETWRAPVISUALFLAGS, 0, 0);
}

void ScintillaEdit::setWrapVisualFlagsLocation(sptr_t wrapVisualFlagsLocation) {
    send(SCI_SETWRAPVISUALFLAGSLOCATION, wrapVisualFlagsLocation, 0);
}

sptr_t ScintillaEdit::wrapVisualFlagsLocation() const {
    return send(SCI_GETWRAPVISUALFLAGSLOCATION, 0, 0);
}

void ScintillaEdit::setWrapStartIndent(sptr_t indent) {
    send(SCI_SETWRAPSTARTINDENT, indent, 0);
}

sptr_t ScintillaEdit::wrapStartIndent() const {
    return send(SCI_GETWRAPSTARTINDENT, 0, 0);
}

void ScintillaEdit::setWrapIndentMode(sptr_t wrapIndentMode) {
    send(SCI_SETWRAPINDENTMODE, wrapIndentMode, 0);
}

sptr_t ScintillaEdit::wrapIndentMode() const {
    return send(SCI_GETWRAPINDENTMODE, 0, 0);
}

void ScintillaEdit::setLayoutCache(sptr_t cacheMode) {
    send(SCI_SETLAYOUTCACHE, cacheMode, 0);
}

sptr_t ScintillaEdit::layoutCache() const {
    return send(SCI_GETLAYOUTCACHE, 0, 0);
}

void ScintillaEdit::setScrollWidth(sptr_t pixelWidth) {
    send(SCI_SETSCROLLWIDTH, pixelWidth, 0);
}

sptr_t ScintillaEdit::scrollWidth() const {
    return send(SCI_GETSCROLLWIDTH, 0, 0);
}

void ScintillaEdit::setScrollWidthTracking(bool tracking) {
    send(SCI_SETSCROLLWIDTHTRACKING, tracking, 0);
}

bool ScintillaEdit::scrollWidthTracking() const {
    return send(SCI_GETSCROLLWIDTHTRACKING, 0, 0);
}

sptr_t ScintillaEdit::textWidth(sptr_t style, const char * text) {
    return send(SCI_TEXTWIDTH, style, (sptr_t)text);
}

void ScintillaEdit::setEndAtLastLine(bool endAtLastLine) {
    send(SCI_SETENDATLASTLINE, endAtLastLine, 0);
}

bool ScintillaEdit::endAtLastLine() const {
    return send(SCI_GETENDATLASTLINE, 0, 0);
}

sptr_t ScintillaEdit::textHeight(sptr_t line) {
    return send(SCI_TEXTHEIGHT, line, 0);
}

void ScintillaEdit::setVScrollBar(bool visible) {
    send(SCI_SETVSCROLLBAR, visible, 0);
}

bool ScintillaEdit::vScrollBar() const {
    return send(SCI_GETVSCROLLBAR, 0, 0);
}

void ScintillaEdit::appendText(sptr_t length, const char * text) {
    send(SCI_APPENDTEXT, length, (sptr_t)text);
}

bool ScintillaEdit::twoPhaseDraw() const {
    return send(SCI_GETTWOPHASEDRAW, 0, 0);
}

void ScintillaEdit::setTwoPhaseDraw(bool twoPhase) {
    send(SCI_SETTWOPHASEDRAW, twoPhase, 0);
}

sptr_t ScintillaEdit::phasesDraw() const {
    return send(SCI_GETPHASESDRAW, 0, 0);
}

void ScintillaEdit::setPhasesDraw(sptr_t phases) {
    send(SCI_SETPHASESDRAW, phases, 0);
}

void ScintillaEdit::setFontQuality(sptr_t fontQuality) {
    send(SCI_SETFONTQUALITY, fontQuality, 0);
}

sptr_t ScintillaEdit::fontQuality() const {
    return send(SCI_GETFONTQUALITY, 0, 0);
}

void ScintillaEdit::setFirstVisibleLine(sptr_t displayLine) {
    send(SCI_SETFIRSTVISIBLELINE, displayLine, 0);
}

void ScintillaEdit::setMultiPaste(sptr_t multiPaste) {
    send(SCI_SETMULTIPASTE, multiPaste, 0);
}

sptr_t ScintillaEdit::multiPaste() const {
    return send(SCI_GETMULTIPASTE, 0, 0);
}

QByteArray ScintillaEdit::tag(sptr_t tagNumber) const {
    return TextReturner(SCI_GETTAG, tagNumber);
}

void ScintillaEdit::linesJoin() {
    send(SCI_LINESJOIN, 0, 0);
}

void ScintillaEdit::linesSplit(sptr_t pixelWidth) {
    send(SCI_LINESSPLIT, pixelWidth, 0);
}

void ScintillaEdit::setFoldMarginColour(bool useSetting, sptr_t back) {
    send(SCI_SETFOLDMARGINCOLOUR, useSetting, back);
}

void ScintillaEdit::setFoldMarginHiColour(bool useSetting, sptr_t fore) {
    send(SCI_SETFOLDMARGINHICOLOUR, useSetting, fore);
}

void ScintillaEdit::lineDown() {
    send(SCI_LINEDOWN, 0, 0);
}

void ScintillaEdit::lineDownExtend() {
    send(SCI_LINEDOWNEXTEND, 0, 0);
}

void ScintillaEdit::lineUp() {
    send(SCI_LINEUP, 0, 0);
}

void ScintillaEdit::lineUpExtend() {
    send(SCI_LINEUPEXTEND, 0, 0);
}

void ScintillaEdit::charLeft() {
    send(SCI_CHARLEFT, 0, 0);
}

void ScintillaEdit::charLeftExtend() {
    send(SCI_CHARLEFTEXTEND, 0, 0);
}

void ScintillaEdit::charRight() {
    send(SCI_CHARRIGHT, 0, 0);
}

void ScintillaEdit::charRightExtend() {
    send(SCI_CHARRIGHTEXTEND, 0, 0);
}

void ScintillaEdit::wordLeft() {
    send(SCI_WORDLEFT, 0, 0);
}

void ScintillaEdit::wordLeftExtend() {
    send(SCI_WORDLEFTEXTEND, 0, 0);
}

void ScintillaEdit::wordRight() {
    send(SCI_WORDRIGHT, 0, 0);
}

void ScintillaEdit::wordRightExtend() {
    send(SCI_WORDRIGHTEXTEND, 0, 0);
}

void ScintillaEdit::home() {
    send(SCI_HOME, 0, 0);
}

void ScintillaEdit::homeExtend() {
    send(SCI_HOMEEXTEND, 0, 0);
}

void ScintillaEdit::lineEnd() {
    send(SCI_LINEEND, 0, 0);
}

void ScintillaEdit::lineEndExtend() {
    send(SCI_LINEENDEXTEND, 0, 0);
}

void ScintillaEdit::documentStart() {
    send(SCI_DOCUMENTSTART, 0, 0);
}

void ScintillaEdit::documentStartExtend() {
    send(SCI_DOCUMENTSTARTEXTEND, 0, 0);
}

void ScintillaEdit::documentEnd() {
    send(SCI_DOCUMENTEND, 0, 0);
}

void ScintillaEdit::documentEndExtend() {
    send(SCI_DOCUMENTENDEXTEND, 0, 0);
}

void ScintillaEdit::pageUp() {
    send(SCI_PAGEUP, 0, 0);
}

void ScintillaEdit::pageUpExtend() {
    send(SCI_PAGEUPEXTEND, 0, 0);
}

void ScintillaEdit::pageDown() {
    send(SCI_PAGEDOWN, 0, 0);
}

void ScintillaEdit::pageDownExtend() {
    send(SCI_PAGEDOWNEXTEND, 0, 0);
}

void ScintillaEdit::editToggleOvertype() {
    send(SCI_EDITTOGGLEOVERTYPE, 0, 0);
}

void ScintillaEdit::cancel() {
    send(SCI_CANCEL, 0, 0);
}

void ScintillaEdit::deleteBack() {
    send(SCI_DELETEBACK, 0, 0);
}

void ScintillaEdit::tab() {
    send(SCI_TAB, 0, 0);
}

void ScintillaEdit::backTab() {
    send(SCI_BACKTAB, 0, 0);
}

void ScintillaEdit::newLine() {
    send(SCI_NEWLINE, 0, 0);
}

void ScintillaEdit::formFeed() {
    send(SCI_FORMFEED, 0, 0);
}

void ScintillaEdit::vCHome() {
    send(SCI_VCHOME, 0, 0);
}

void ScintillaEdit::vCHomeExtend() {
    send(SCI_VCHOMEEXTEND, 0, 0);
}

void ScintillaEdit::zoomIn() {
    send(SCI_ZOOMIN, 0, 0);
}

void ScintillaEdit::zoomOut() {
    send(SCI_ZOOMOUT, 0, 0);
}

void ScintillaEdit::delWordLeft() {
    send(SCI_DELWORDLEFT, 0, 0);
}

void ScintillaEdit::delWordRight() {
    send(SCI_DELWORDRIGHT, 0, 0);
}

void ScintillaEdit::delWordRightEnd() {
    send(SCI_DELWORDRIGHTEND, 0, 0);
}

void ScintillaEdit::lineCut() {
    send(SCI_LINECUT, 0, 0);
}

void ScintillaEdit::lineDelete() {
    send(SCI_LINEDELETE, 0, 0);
}

void ScintillaEdit::lineTranspose() {
    send(SCI_LINETRANSPOSE, 0, 0);
}

void ScintillaEdit::lineDuplicate() {
    send(SCI_LINEDUPLICATE, 0, 0);
}

void ScintillaEdit::lowerCase() {
    send(SCI_LOWERCASE, 0, 0);
}

void ScintillaEdit::upperCase() {
    send(SCI_UPPERCASE, 0, 0);
}

void ScintillaEdit::lineScrollDown() {
    send(SCI_LINESCROLLDOWN, 0, 0);
}

void ScintillaEdit::lineScrollUp() {
    send(SCI_LINESCROLLUP, 0, 0);
}

void ScintillaEdit::deleteBackNotLine() {
    send(SCI_DELETEBACKNOTLINE, 0, 0);
}

void ScintillaEdit::homeDisplay() {
    send(SCI_HOMEDISPLAY, 0, 0);
}

void ScintillaEdit::homeDisplayExtend() {
    send(SCI_HOMEDISPLAYEXTEND, 0, 0);
}

void ScintillaEdit::lineEndDisplay() {
    send(SCI_LINEENDDISPLAY, 0, 0);
}

void ScintillaEdit::lineEndDisplayExtend() {
    send(SCI_LINEENDDISPLAYEXTEND, 0, 0);
}

void ScintillaEdit::homeWrap() {
    send(SCI_HOMEWRAP, 0, 0);
}

void ScintillaEdit::homeWrapExtend() {
    send(SCI_HOMEWRAPEXTEND, 0, 0);
}

void ScintillaEdit::lineEndWrap() {
    send(SCI_LINEENDWRAP, 0, 0);
}

void ScintillaEdit::lineEndWrapExtend() {
    send(SCI_LINEENDWRAPEXTEND, 0, 0);
}

void ScintillaEdit::vCHomeWrap() {
    send(SCI_VCHOMEWRAP, 0, 0);
}

void ScintillaEdit::vCHomeWrapExtend() {
    send(SCI_VCHOMEWRAPEXTEND, 0, 0);
}

void ScintillaEdit::lineCopy() {
    send(SCI_LINECOPY, 0, 0);
}

void ScintillaEdit::moveCaretInsideView() {
    send(SCI_MOVECARETINSIDEVIEW, 0, 0);
}

sptr_t ScintillaEdit::lineLength(sptr_t line) {
    return send(SCI_LINELENGTH, line, 0);
}

void ScintillaEdit::braceHighlight(sptr_t posA, sptr_t posB) {
    send(SCI_BRACEHIGHLIGHT, posA, posB);
}

void ScintillaEdit::braceHighlightIndicator(bool useSetting, sptr_t indicator) {
    send(SCI_BRACEHIGHLIGHTINDICATOR, useSetting, indicator);
}

void ScintillaEdit::braceBadLight(sptr_t pos) {
    send(SCI_BRACEBADLIGHT, pos, 0);
}

void ScintillaEdit::braceBadLightIndicator(bool useSetting, sptr_t indicator) {
    send(SCI_BRACEBADLIGHTINDICATOR, useSetting, indicator);
}

sptr_t ScintillaEdit::braceMatch(sptr_t pos, sptr_t maxReStyle) {
    return send(SCI_BRACEMATCH, pos, maxReStyle);
}

bool ScintillaEdit::viewEOL() const {
    return send(SCI_GETVIEWEOL, 0, 0);
}

void ScintillaEdit::setViewEOL(bool visible) {
    send(SCI_SETVIEWEOL, visible, 0);
}

sptr_t ScintillaEdit::docPointer() const {
    return send(SCI_GETDOCPOINTER, 0, 0);
}

void ScintillaEdit::setDocPointer(sptr_t doc) {
    send(SCI_SETDOCPOINTER, 0, doc);
}

void ScintillaEdit::setModEventMask(sptr_t eventMask) {
    send(SCI_SETMODEVENTMASK, eventMask, 0);
}

sptr_t ScintillaEdit::edgeColumn() const {
    return send(SCI_GETEDGECOLUMN, 0, 0);
}

void ScintillaEdit::setEdgeColumn(sptr_t column) {
    send(SCI_SETEDGECOLUMN, column, 0);
}

sptr_t ScintillaEdit::edgeMode() const {
    return send(SCI_GETEDGEMODE, 0, 0);
}

void ScintillaEdit::setEdgeMode(sptr_t edgeMode) {
    send(SCI_SETEDGEMODE, edgeMode, 0);
}

sptr_t ScintillaEdit::edgeColour() const {
    return send(SCI_GETEDGECOLOUR, 0, 0);
}

void ScintillaEdit::setEdgeColour(sptr_t edgeColour) {
    send(SCI_SETEDGECOLOUR, edgeColour, 0);
}

void ScintillaEdit::multiEdgeAddLine(sptr_t column, sptr_t edgeColour) {
    send(SCI_MULTIEDGEADDLINE, column, edgeColour);
}

void ScintillaEdit::multiEdgeClearAll() {
    send(SCI_MULTIEDGECLEARALL, 0, 0);
}

void ScintillaEdit::searchAnchor() {
    send(SCI_SEARCHANCHOR, 0, 0);
}

sptr_t ScintillaEdit::searchNext(sptr_t searchFlags, const char * text) {
    return send(SCI_SEARCHNEXT, searchFlags, (sptr_t)text);
}

sptr_t ScintillaEdit::searchPrev(sptr_t searchFlags, const char * text) {
    return send(SCI_SEARCHPREV, searchFlags, (sptr_t)text);
}

sptr_t ScintillaEdit::linesOnScreen() const {
    return send(SCI_LINESONSCREEN, 0, 0);
}

void ScintillaEdit::usePopUp(sptr_t popUpMode) {
    send(SCI_USEPOPUP, popUpMode, 0);
}

bool ScintillaEdit::selectionIsRectangle() const {
    return send(SCI_SELECTIONISRECTANGLE, 0, 0);
}

void ScintillaEdit::setZoom(sptr_t zoomInPoints) {
    send(SCI_SETZOOM, zoomInPoints, 0);
}

sptr_t ScintillaEdit::zoom() const {
    return send(SCI_GETZOOM, 0, 0);
}

sptr_t ScintillaEdit::createDocument() {
    return send(SCI_CREATEDOCUMENT, 0, 0);
}

void ScintillaEdit::addRefDocument(sptr_t doc) {
    send(SCI_ADDREFDOCUMENT, 0, doc);
}

void ScintillaEdit::releaseDocument(sptr_t doc) {
    send(SCI_RELEASEDOCUMENT, 0, doc);
}

sptr_t ScintillaEdit::modEventMask() const {
    return send(SCI_GETMODEVENTMASK, 0, 0);
}

void ScintillaEdit::setFocus(bool focus) {
    send(SCI_SETFOCUS, focus, 0);
}

bool ScintillaEdit::focus() const {
    return send(SCI_GETFOCUS, 0, 0);
}

void ScintillaEdit::setStatus(sptr_t status) {
    send(SCI_SETSTATUS, status, 0);
}

sptr_t ScintillaEdit::status() const {
    return send(SCI_GETSTATUS, 0, 0);
}

void ScintillaEdit::setMouseDownCaptures(bool captures) {
    send(SCI_SETMOUSEDOWNCAPTURES, captures, 0);
}

bool ScintillaEdit::mouseDownCaptures() const {
    return send(SCI_GETMOUSEDOWNCAPTURES, 0, 0);
}

void ScintillaEdit::setMouseWheelCaptures(bool captures) {
    send(SCI_SETMOUSEWHEELCAPTURES, captures, 0);
}

bool ScintillaEdit::mouseWheelCaptures() const {
    return send(SCI_GETMOUSEWHEELCAPTURES, 0, 0);
}

void ScintillaEdit::setCursor(sptr_t cursorType) {
    send(SCI_SETCURSOR, cursorType, 0);
}

sptr_t ScintillaEdit::cursor() const {
    return send(SCI_GETCURSOR, 0, 0);
}

void ScintillaEdit::setControlCharSymbol(sptr_t symbol) {
    send(SCI_SETCONTROLCHARSYMBOL, symbol, 0);
}

sptr_t ScintillaEdit::controlCharSymbol() const {
    return send(SCI_GETCONTROLCHARSYMBOL, 0, 0);
}

void ScintillaEdit::wordPartLeft() {
    send(SCI_WORDPARTLEFT, 0, 0);
}

void ScintillaEdit::wordPartLeftExtend() {
    send(SCI_WORDPARTLEFTEXTEND, 0, 0);
}

void ScintillaEdit::wordPartRight() {
    send(SCI_WORDPARTRIGHT, 0, 0);
}

void ScintillaEdit::wordPartRightExtend() {
    send(SCI_WORDPARTRIGHTEXTEND, 0, 0);
}

void ScintillaEdit::setVisiblePolicy(sptr_t visiblePolicy, sptr_t visibleSlop) {
    send(SCI_SETVISIBLEPOLICY, visiblePolicy, visibleSlop);
}

void ScintillaEdit::delLineLeft() {
    send(SCI_DELLINELEFT, 0, 0);
}

void ScintillaEdit::delLineRight() {
    send(SCI_DELLINERIGHT, 0, 0);
}

void ScintillaEdit::setXOffset(sptr_t xOffset) {
    send(SCI_SETXOFFSET, xOffset, 0);
}

sptr_t ScintillaEdit::xOffset() const {
    return send(SCI_GETXOFFSET, 0, 0);
}

void ScintillaEdit::chooseCaretX() {
    send(SCI_CHOOSECARETX, 0, 0);
}

void ScintillaEdit::grabFocus() {
    send(SCI_GRABFOCUS, 0, 0);
}

void ScintillaEdit::setXCaretPolicy(sptr_t caretPolicy, sptr_t caretSlop) {
    send(SCI_SETXCARETPOLICY, caretPolicy, caretSlop);
}

void ScintillaEdit::setYCaretPolicy(sptr_t caretPolicy, sptr_t caretSlop) {
    send(SCI_SETYCARETPOLICY, caretPolicy, caretSlop);
}

void ScintillaEdit::setPrintWrapMode(sptr_t wrapMode) {
    send(SCI_SETPRINTWRAPMODE, wrapMode, 0);
}

sptr_t ScintillaEdit::printWrapMode() const {
    return send(SCI_GETPRINTWRAPMODE, 0, 0);
}

void ScintillaEdit::setHotspotActiveFore(bool useSetting, sptr_t fore) {
    send(SCI_SETHOTSPOTACTIVEFORE, useSetting, fore);
}

sptr_t ScintillaEdit::hotspotActiveFore() const {
    return send(SCI_GETHOTSPOTACTIVEFORE, 0, 0);
}

void ScintillaEdit::setHotspotActiveBack(bool useSetting, sptr_t back) {
    send(SCI_SETHOTSPOTACTIVEBACK, useSetting, back);
}

sptr_t ScintillaEdit::hotspotActiveBack() const {
    return send(SCI_GETHOTSPOTACTIVEBACK, 0, 0);
}

void ScintillaEdit::setHotspotActiveUnderline(bool underline) {
    send(SCI_SETHOTSPOTACTIVEUNDERLINE, underline, 0);
}

bool ScintillaEdit::hotspotActiveUnderline() const {
    return send(SCI_GETHOTSPOTACTIVEUNDERLINE, 0, 0);
}

void ScintillaEdit::setHotspotSingleLine(bool singleLine) {
    send(SCI_SETHOTSPOTSINGLELINE, singleLine, 0);
}

bool ScintillaEdit::hotspotSingleLine() const {
    return send(SCI_GETHOTSPOTSINGLELINE, 0, 0);
}

void ScintillaEdit::paraDown() {
    send(SCI_PARADOWN, 0, 0);
}

void ScintillaEdit::paraDownExtend() {
    send(SCI_PARADOWNEXTEND, 0, 0);
}

void ScintillaEdit::paraUp() {
    send(SCI_PARAUP, 0, 0);
}

void ScintillaEdit::paraUpExtend() {
    send(SCI_PARAUPEXTEND, 0, 0);
}

sptr_t ScintillaEdit::positionBefore(sptr_t pos) {
    return send(SCI_POSITIONBEFORE, pos, 0);
}

sptr_t ScintillaEdit::positionAfter(sptr_t pos) {
    return send(SCI_POSITIONAFTER, pos, 0);
}

sptr_t ScintillaEdit::positionRelative(sptr_t pos, sptr_t relative) {
    return send(SCI_POSITIONRELATIVE, pos, relative);
}

void ScintillaEdit::copyRange(sptr_t start, sptr_t end) {
    send(SCI_COPYRANGE, start, end);
}

void ScintillaEdit::copyText(sptr_t length, const char * text) {
    send(SCI_COPYTEXT, length, (sptr_t)text);
}

void ScintillaEdit::setSelectionMode(sptr_t selectionMode) {
    send(SCI_SETSELECTIONMODE, selectionMode, 0);
}

sptr_t ScintillaEdit::selectionMode() const {
    return send(SCI_GETSELECTIONMODE, 0, 0);
}

sptr_t ScintillaEdit::getLineSelStartPosition(sptr_t line) {
    return send(SCI_GETLINESELSTARTPOSITION, line, 0);
}

sptr_t ScintillaEdit::getLineSelEndPosition(sptr_t line) {
    return send(SCI_GETLINESELENDPOSITION, line, 0);
}

void ScintillaEdit::lineDownRectExtend() {
    send(SCI_LINEDOWNRECTEXTEND, 0, 0);
}

void ScintillaEdit::lineUpRectExtend() {
    send(SCI_LINEUPRECTEXTEND, 0, 0);
}

void ScintillaEdit::charLeftRectExtend() {
    send(SCI_CHARLEFTRECTEXTEND, 0, 0);
}

void ScintillaEdit::charRightRectExtend() {
    send(SCI_CHARRIGHTRECTEXTEND, 0, 0);
}

void ScintillaEdit::homeRectExtend() {
    send(SCI_HOMERECTEXTEND, 0, 0);
}

void ScintillaEdit::vCHomeRectExtend() {
    send(SCI_VCHOMERECTEXTEND, 0, 0);
}

void ScintillaEdit::lineEndRectExtend() {
    send(SCI_LINEENDRECTEXTEND, 0, 0);
}

void ScintillaEdit::pageUpRectExtend() {
    send(SCI_PAGEUPRECTEXTEND, 0, 0);
}

void ScintillaEdit::pageDownRectExtend() {
    send(SCI_PAGEDOWNRECTEXTEND, 0, 0);
}

void ScintillaEdit::stutteredPageUp() {
    send(SCI_STUTTEREDPAGEUP, 0, 0);
}

void ScintillaEdit::stutteredPageUpExtend() {
    send(SCI_STUTTEREDPAGEUPEXTEND, 0, 0);
}

void ScintillaEdit::stutteredPageDown() {
    send(SCI_STUTTEREDPAGEDOWN, 0, 0);
}

void ScintillaEdit::stutteredPageDownExtend() {
    send(SCI_STUTTEREDPAGEDOWNEXTEND, 0, 0);
}

void ScintillaEdit::wordLeftEnd() {
    send(SCI_WORDLEFTEND, 0, 0);
}

void ScintillaEdit::wordLeftEndExtend() {
    send(SCI_WORDLEFTENDEXTEND, 0, 0);
}

void ScintillaEdit::wordRightEnd() {
    send(SCI_WORDRIGHTEND, 0, 0);
}

void ScintillaEdit::wordRightEndExtend() {
    send(SCI_WORDRIGHTENDEXTEND, 0, 0);
}

void ScintillaEdit::setWhitespaceChars(const char * characters) {
    send(SCI_SETWHITESPACECHARS, 0, (sptr_t)characters);
}

QByteArray ScintillaEdit::whitespaceChars() const {
    return TextReturner(SCI_GETWHITESPACECHARS, 0);
}

void ScintillaEdit::setPunctuationChars(const char * characters) {
    send(SCI_SETPUNCTUATIONCHARS, 0, (sptr_t)characters);
}

QByteArray ScintillaEdit::punctuationChars() const {
    return TextReturner(SCI_GETPUNCTUATIONCHARS, 0);
}

void ScintillaEdit::setCharsDefault() {
    send(SCI_SETCHARSDEFAULT, 0, 0);
}

sptr_t ScintillaEdit::autoCCurrent() const {
    return send(SCI_AUTOCGETCURRENT, 0, 0);
}

QByteArray ScintillaEdit::autoCCurrentText() const {
    return TextReturner(SCI_AUTOCGETCURRENTTEXT, 0);
}

void ScintillaEdit::autoCSetCaseInsensitiveBehaviour(sptr_t behaviour) {
    send(SCI_AUTOCSETCASEINSENSITIVEBEHAVIOUR, behaviour, 0);
}

sptr_t ScintillaEdit::autoCCaseInsensitiveBehaviour() const {
    return send(SCI_AUTOCGETCASEINSENSITIVEBEHAVIOUR, 0, 0);
}

void ScintillaEdit::autoCSetMulti(sptr_t multi) {
    send(SCI_AUTOCSETMULTI, multi, 0);
}

sptr_t ScintillaEdit::autoCMulti() const {
    return send(SCI_AUTOCGETMULTI, 0, 0);
}

void ScintillaEdit::autoCSetOrder(sptr_t order) {
    send(SCI_AUTOCSETORDER, order, 0);
}

sptr_t ScintillaEdit::autoCOrder() const {
    return send(SCI_AUTOCGETORDER, 0, 0);
}

void ScintillaEdit::allocate(sptr_t bytes) {
    send(SCI_ALLOCATE, bytes, 0);
}

QByteArray ScintillaEdit::targetAsUTF8() {
    return TextReturner(SCI_TARGETASUTF8, 0);
}

void ScintillaEdit::setLengthForEncode(sptr_t bytes) {
    send(SCI_SETLENGTHFORENCODE, bytes, 0);
}

QByteArray ScintillaEdit::encodedFromUTF8(const char * utf8) {
    return TextReturner(SCI_ENCODEDFROMUTF8, (sptr_t)utf8);
}

sptr_t ScintillaEdit::findColumn(sptr_t line, sptr_t column) {
    return send(SCI_FINDCOLUMN, line, column);
}

sptr_t ScintillaEdit::caretSticky() const {
    return send(SCI_GETCARETSTICKY, 0, 0);
}

void ScintillaEdit::setCaretSticky(sptr_t useCaretStickyBehaviour) {
    send(SCI_SETCARETSTICKY, useCaretStickyBehaviour, 0);
}

void ScintillaEdit::toggleCaretSticky() {
    send(SCI_TOGGLECARETSTICKY, 0, 0);
}

void ScintillaEdit::setPasteConvertEndings(bool convert) {
    send(SCI_SETPASTECONVERTENDINGS, convert, 0);
}

bool ScintillaEdit::pasteConvertEndings() const {
    return send(SCI_GETPASTECONVERTENDINGS, 0, 0);
}

void ScintillaEdit::selectionDuplicate() {
    send(SCI_SELECTIONDUPLICATE, 0, 0);
}

void ScintillaEdit::setCaretLineBackAlpha(sptr_t alpha) {
    send(SCI_SETCARETLINEBACKALPHA, alpha, 0);
}

sptr_t ScintillaEdit::caretLineBackAlpha() const {
    return send(SCI_GETCARETLINEBACKALPHA, 0, 0);
}

void ScintillaEdit::setCaretStyle(sptr_t caretStyle) {
    send(SCI_SETCARETSTYLE, caretStyle, 0);
}

sptr_t ScintillaEdit::caretStyle() const {
    return send(SCI_GETCARETSTYLE, 0, 0);
}

void ScintillaEdit::setIndicatorCurrent(sptr_t indicator) {
    send(SCI_SETINDICATORCURRENT, indicator, 0);
}

sptr_t ScintillaEdit::indicatorCurrent() const {
    return send(SCI_GETINDICATORCURRENT, 0, 0);
}

void ScintillaEdit::setIndicatorValue(sptr_t value) {
    send(SCI_SETINDICATORVALUE, value, 0);
}

sptr_t ScintillaEdit::indicatorValue() const {
    return send(SCI_GETINDICATORVALUE, 0, 0);
}

void ScintillaEdit::indicatorFillRange(sptr_t start, sptr_t lengthFill) {
    send(SCI_INDICATORFILLRANGE, start, lengthFill);
}

void ScintillaEdit::indicatorClearRange(sptr_t start, sptr_t lengthClear) {
    send(SCI_INDICATORCLEARRANGE, start, lengthClear);
}

sptr_t ScintillaEdit::indicatorAllOnFor(sptr_t pos) {
    return send(SCI_INDICATORALLONFOR, pos, 0);
}

sptr_t ScintillaEdit::indicatorValueAt(sptr_t indicator, sptr_t pos) {
    return send(SCI_INDICATORVALUEAT, indicator, pos);
}

sptr_t ScintillaEdit::indicatorStart(sptr_t indicator, sptr_t pos) {
    return send(SCI_INDICATORSTART, indicator, pos);
}

sptr_t ScintillaEdit::indicatorEnd(sptr_t indicator, sptr_t pos) {
    return send(SCI_INDICATOREND, indicator, pos);
}

void ScintillaEdit::setPositionCache(sptr_t size) {
    send(SCI_SETPOSITIONCACHE, size, 0);
}

sptr_t ScintillaEdit::positionCache() const {
    return send(SCI_GETPOSITIONCACHE, 0, 0);
}

void ScintillaEdit::copyAllowLine() {
    send(SCI_COPYALLOWLINE, 0, 0);
}

sptr_t ScintillaEdit::characterPointer() const {
    return send(SCI_GETCHARACTERPOINTER, 0, 0);
}

sptr_t ScintillaEdit::rangePointer(sptr_t start, sptr_t lengthRange) const {
    return send(SCI_GETRANGEPOINTER, start, lengthRange);
}

sptr_t ScintillaEdit::gapPosition() const {
    return send(SCI_GETGAPPOSITION, 0, 0);
}

void ScintillaEdit::indicSetAlpha(sptr_t indicator, sptr_t alpha) {
    send(SCI_INDICSETALPHA, indicator, alpha);
}

sptr_t ScintillaEdit::indicAlpha(sptr_t indicator) const {
    return send(SCI_INDICGETALPHA, indicator, 0);
}

void ScintillaEdit::indicSetOutlineAlpha(sptr_t indicator, sptr_t alpha) {
    send(SCI_INDICSETOUTLINEALPHA, indicator, alpha);
}

sptr_t ScintillaEdit::indicOutlineAlpha(sptr_t indicator) const {
    return send(SCI_INDICGETOUTLINEALPHA, indicator, 0);
}

void ScintillaEdit::setExtraAscent(sptr_t extraAscent) {
    send(SCI_SETEXTRAASCENT, extraAscent, 0);
}

sptr_t ScintillaEdit::extraAscent() const {
    return send(SCI_GETEXTRAASCENT, 0, 0);
}

void ScintillaEdit::setExtraDescent(sptr_t extraDescent) {
    send(SCI_SETEXTRADESCENT, extraDescent, 0);
}

sptr_t ScintillaEdit::extraDescent() const {
    return send(SCI_GETEXTRADESCENT, 0, 0);
}

sptr_t ScintillaEdit::markerSymbolDefined(sptr_t markerNumber) {
    return send(SCI_MARKERSYMBOLDEFINED, markerNumber, 0);
}

void ScintillaEdit::marginSetText(sptr_t line, const char * text) {
    send(SCI_MARGINSETTEXT, line, (sptr_t)text);
}

QByteArray ScintillaEdit::marginText(sptr_t line) const {
    return TextReturner(SCI_MARGINGETTEXT, line);
}

void ScintillaEdit::marginSetStyle(sptr_t line, sptr_t style) {
    send(SCI_MARGINSETSTYLE, line, style);
}

sptr_t ScintillaEdit::marginStyle(sptr_t line) const {
    return send(SCI_MARGINGETSTYLE, line, 0);
}

void ScintillaEdit::marginSetStyles(sptr_t line, const char * styles) {
    send(SCI_MARGINSETSTYLES, line, (sptr_t)styles);
}

QByteArray ScintillaEdit::marginStyles(sptr_t line) const {
    return TextReturner(SCI_MARGINGETSTYLES, line);
}

void ScintillaEdit::marginTextClearAll() {
    send(SCI_MARGINTEXTCLEARALL, 0, 0);
}

void ScintillaEdit::marginSetStyleOffset(sptr_t style) {
    send(SCI_MARGINSETSTYLEOFFSET, style, 0);
}

sptr_t ScintillaEdit::marginStyleOffset() const {
    return send(SCI_MARGINGETSTYLEOFFSET, 0, 0);
}

void ScintillaEdit::setMarginOptions(sptr_t marginOptions) {
    send(SCI_SETMARGINOPTIONS, marginOptions, 0);
}

sptr_t ScintillaEdit::marginOptions() const {
    return send(SCI_GETMARGINOPTIONS, 0, 0);
}

void ScintillaEdit::annotationSetText(sptr_t line, const char * text) {
    send(SCI_ANNOTATIONSETTEXT, line, (sptr_t)text);
}

QByteArray ScintillaEdit::annotationText(sptr_t line) const {
    return TextReturner(SCI_ANNOTATIONGETTEXT, line);
}

void ScintillaEdit::annotationSetStyle(sptr_t line, sptr_t style) {
    send(SCI_ANNOTATIONSETSTYLE, line, style);
}

sptr_t ScintillaEdit::annotationStyle(sptr_t line) const {
    return send(SCI_ANNOTATIONGETSTYLE, line, 0);
}

void ScintillaEdit::annotationSetStyles(sptr_t line, const char * styles) {
    send(SCI_ANNOTATIONSETSTYLES, line, (sptr_t)styles);
}

QByteArray ScintillaEdit::annotationStyles(sptr_t line) const {
    return TextReturner(SCI_ANNOTATIONGETSTYLES, line);
}

sptr_t ScintillaEdit::annotationLines(sptr_t line) const {
    return send(SCI_ANNOTATIONGETLINES, line, 0);
}

void ScintillaEdit::annotationClearAll() {
    send(SCI_ANNOTATIONCLEARALL, 0, 0);
}

void ScintillaEdit::annotationSetVisible(sptr_t visible) {
    send(SCI_ANNOTATIONSETVISIBLE, visible, 0);
}

sptr_t ScintillaEdit::annotationVisible() const {
    return send(SCI_ANNOTATIONGETVISIBLE, 0, 0);
}

void ScintillaEdit::annotationSetStyleOffset(sptr_t style) {
    send(SCI_ANNOTATIONSETSTYLEOFFSET, style, 0);
}

sptr_t ScintillaEdit::annotationStyleOffset() const {
    return send(SCI_ANNOTATIONGETSTYLEOFFSET, 0, 0);
}

void ScintillaEdit::releaseAllExtendedStyles() {
    send(SCI_RELEASEALLEXTENDEDSTYLES, 0, 0);
}

sptr_t ScintillaEdit::allocateExtendedStyles(sptr_t numberStyles) {
    return send(SCI_ALLOCATEEXTENDEDSTYLES, numberStyles, 0);
}

void ScintillaEdit::addUndoAction(sptr_t token, sptr_t flags) {
    send(SCI_ADDUNDOACTION, token, flags);
}

sptr_t ScintillaEdit::charPositionFromPoint(sptr_t x, sptr_t y) {
    return send(SCI_CHARPOSITIONFROMPOINT, x, y);
}

sptr_t ScintillaEdit::charPositionFromPointClose(sptr_t x, sptr_t y) {
    return send(SCI_CHARPOSITIONFROMPOINTCLOSE, x, y);
}

void ScintillaEdit::setMouseSelectionRectangularSwitch(bool mouseSelectionRectangularSwitch) {
    send(SCI_SETMOUSESELECTIONRECTANGULARSWITCH, mouseSelectionRectangularSwitch, 0);
}

bool ScintillaEdit::mouseSelectionRectangularSwitch() const {
    return send(SCI_GETMOUSESELECTIONRECTANGULARSWITCH, 0, 0);
}

void ScintillaEdit::setMultipleSelection(bool multipleSelection) {
    send(SCI_SETMULTIPLESELECTION, multipleSelection, 0);
}

bool ScintillaEdit::multipleSelection() const {
    return send(SCI_GETMULTIPLESELECTION, 0, 0);
}

void ScintillaEdit::setAdditionalSelectionTyping(bool additionalSelectionTyping) {
    send(SCI_SETADDITIONALSELECTIONTYPING, additionalSelectionTyping, 0);
}

bool ScintillaEdit::additionalSelectionTyping() const {
    return send(SCI_GETADDITIONALSELECTIONTYPING, 0, 0);
}

void ScintillaEdit::setAdditionalCaretsBlink(bool additionalCaretsBlink) {
    send(SCI_SETADDITIONALCARETSBLINK, additionalCaretsBlink, 0);
}

bool ScintillaEdit::additionalCaretsBlink() const {
    return send(SCI_GETADDITIONALCARETSBLINK, 0, 0);
}

void ScintillaEdit::setAdditionalCaretsVisible(bool additionalCaretsVisible) {
    send(SCI_SETADDITIONALCARETSVISIBLE, additionalCaretsVisible, 0);
}

bool ScintillaEdit::additionalCaretsVisible() const {
    return send(SCI_GETADDITIONALCARETSVISIBLE, 0, 0);
}

sptr_t ScintillaEdit::selections() const {
    return send(SCI_GETSELECTIONS, 0, 0);
}

bool ScintillaEdit::selectionEmpty() const {
    return send(SCI_GETSELECTIONEMPTY, 0, 0);
}

void ScintillaEdit::clearSelections() {
    send(SCI_CLEARSELECTIONS, 0, 0);
}

sptr_t ScintillaEdit::setSelection(sptr_t caret, sptr_t anchor) {
    return send(SCI_SETSELECTION, caret, anchor);
}

sptr_t ScintillaEdit::addSelection(sptr_t caret, sptr_t anchor) {
    return send(SCI_ADDSELECTION, caret, anchor);
}

void ScintillaEdit::dropSelectionN(sptr_t selection) {
    send(SCI_DROPSELECTIONN, selection, 0);
}

void ScintillaEdit::setMainSelection(sptr_t selection) {
    send(SCI_SETMAINSELECTION, selection, 0);
}

sptr_t ScintillaEdit::mainSelection() const {
    return send(SCI_GETMAINSELECTION, 0, 0);
}

void ScintillaEdit::setSelectionNCaret(sptr_t selection, sptr_t caret) {
    send(SCI_SETSELECTIONNCARET, selection, caret);
}

sptr_t ScintillaEdit::selectionNCaret(sptr_t selection) const {
    return send(SCI_GETSELECTIONNCARET, selection, 0);
}

void ScintillaEdit::setSelectionNAnchor(sptr_t selection, sptr_t anchor) {
    send(SCI_SETSELECTIONNANCHOR, selection, anchor);
}

sptr_t ScintillaEdit::selectionNAnchor(sptr_t selection) const {
    return send(SCI_GETSELECTIONNANCHOR, selection, 0);
}

void ScintillaEdit::setSelectionNCaretVirtualSpace(sptr_t selection, sptr_t space) {
    send(SCI_SETSELECTIONNCARETVIRTUALSPACE, selection, space);
}

sptr_t ScintillaEdit::selectionNCaretVirtualSpace(sptr_t selection) const {
    return send(SCI_GETSELECTIONNCARETVIRTUALSPACE, selection, 0);
}

void ScintillaEdit::setSelectionNAnchorVirtualSpace(sptr_t selection, sptr_t space) {
    send(SCI_SETSELECTIONNANCHORVIRTUALSPACE, selection, space);
}

sptr_t ScintillaEdit::selectionNAnchorVirtualSpace(sptr_t selection) const {
    return send(SCI_GETSELECTIONNANCHORVIRTUALSPACE, selection, 0);
}

void ScintillaEdit::setSelectionNStart(sptr_t selection, sptr_t anchor) {
    send(SCI_SETSELECTIONNSTART, selection, anchor);
}

sptr_t ScintillaEdit::selectionNStart(sptr_t selection) const {
    return send(SCI_GETSELECTIONNSTART, selection, 0);
}

void ScintillaEdit::setSelectionNEnd(sptr_t selection, sptr_t caret) {
    send(SCI_SETSELECTIONNEND, selection, caret);
}

sptr_t ScintillaEdit::selectionNEnd(sptr_t selection) const {
    return send(SCI_GETSELECTIONNEND, selection, 0);
}

void ScintillaEdit::setRectangularSelectionCaret(sptr_t caret) {
    send(SCI_SETRECTANGULARSELECTIONCARET, caret, 0);
}

sptr_t ScintillaEdit::rectangularSelectionCaret() const {
    return send(SCI_GETRECTANGULARSELECTIONCARET, 0, 0);
}

void ScintillaEdit::setRectangularSelectionAnchor(sptr_t anchor) {
    send(SCI_SETRECTANGULARSELECTIONANCHOR, anchor, 0);
}

sptr_t ScintillaEdit::rectangularSelectionAnchor() const {
    return send(SCI_GETRECTANGULARSELECTIONANCHOR, 0, 0);
}

void ScintillaEdit::setRectangularSelectionCaretVirtualSpace(sptr_t space) {
    send(SCI_SETRECTANGULARSELECTIONCARETVIRTUALSPACE, space, 0);
}

sptr_t ScintillaEdit::rectangularSelectionCaretVirtualSpace() const {
    return send(SCI_GETRECTANGULARSELECTIONCARETVIRTUALSPACE, 0, 0);
}

void ScintillaEdit::setRectangularSelectionAnchorVirtualSpace(sptr_t space) {
    send(SCI_SETRECTANGULARSELECTIONANCHORVIRTUALSPACE, space, 0);
}

sptr_t ScintillaEdit::rectangularSelectionAnchorVirtualSpace() const {
    return send(SCI_GETRECTANGULARSELECTIONANCHORVIRTUALSPACE, 0, 0);
}

void ScintillaEdit::setVirtualSpaceOptions(sptr_t virtualSpaceOptions) {
    send(SCI_SETVIRTUALSPACEOPTIONS, virtualSpaceOptions, 0);
}

sptr_t ScintillaEdit::virtualSpaceOptions() const {
    return send(SCI_GETVIRTUALSPACEOPTIONS, 0, 0);
}

void ScintillaEdit::setRectangularSelectionModifier(sptr_t modifier) {
    send(SCI_SETRECTANGULARSELECTIONMODIFIER, modifier, 0);
}

sptr_t ScintillaEdit::rectangularSelectionModifier() const {
    return send(SCI_GETRECTANGULARSELECTIONMODIFIER, 0, 0);
}

void ScintillaEdit::setAdditionalSelFore(sptr_t fore) {
    send(SCI_SETADDITIONALSELFORE, fore, 0);
}

void ScintillaEdit::setAdditionalSelBack(sptr_t back) {
    send(SCI_SETADDITIONALSELBACK, back, 0);
}

void ScintillaEdit::setAdditionalSelAlpha(sptr_t alpha) {
    send(SCI_SETADDITIONALSELALPHA, alpha, 0);
}

sptr_t ScintillaEdit::additionalSelAlpha() const {
    return send(SCI_GETADDITIONALSELALPHA, 0, 0);
}

void ScintillaEdit::setAdditionalCaretFore(sptr_t fore) {
    send(SCI_SETADDITIONALCARETFORE, fore, 0);
}

sptr_t ScintillaEdit::additionalCaretFore() const {
    return send(SCI_GETADDITIONALCARETFORE, 0, 0);
}

void ScintillaEdit::rotateSelection() {
    send(SCI_ROTATESELECTION, 0, 0);
}

void ScintillaEdit::swapMainAnchorCaret() {
    send(SCI_SWAPMAINANCHORCARET, 0, 0);
}

void ScintillaEdit::multipleSelectAddNext() {
    send(SCI_MULTIPLESELECTADDNEXT, 0, 0);
}

void ScintillaEdit::multipleSelectAddEach() {
    send(SCI_MULTIPLESELECTADDEACH, 0, 0);
}

sptr_t ScintillaEdit::changeLexerState(sptr_t start, sptr_t end) {
    return send(SCI_CHANGELEXERSTATE, start, end);
}

sptr_t ScintillaEdit::contractedFoldNext(sptr_t lineStart) {
    return send(SCI_CONTRACTEDFOLDNEXT, lineStart, 0);
}

void ScintillaEdit::verticalCentreCaret() {
    send(SCI_VERTICALCENTRECARET, 0, 0);
}

void ScintillaEdit::moveSelectedLinesUp() {
    send(SCI_MOVESELECTEDLINESUP, 0, 0);
}

void ScintillaEdit::moveSelectedLinesDown() {
    send(SCI_MOVESELECTEDLINESDOWN, 0, 0);
}

void ScintillaEdit::setIdentifier(sptr_t identifier) {
    send(SCI_SETIDENTIFIER, identifier, 0);
}

sptr_t ScintillaEdit::identifier() const {
    return send(SCI_GETIDENTIFIER, 0, 0);
}

void ScintillaEdit::rGBAImageSetWidth(sptr_t width) {
    send(SCI_RGBAIMAGESETWIDTH, width, 0);
}

void ScintillaEdit::rGBAImageSetHeight(sptr_t height) {
    send(SCI_RGBAIMAGESETHEIGHT, height, 0);
}

void ScintillaEdit::rGBAImageSetScale(sptr_t scalePercent) {
    send(SCI_RGBAIMAGESETSCALE, scalePercent, 0);
}

void ScintillaEdit::markerDefineRGBAImage(sptr_t markerNumber, const char * pixels) {
    send(SCI_MARKERDEFINERGBAIMAGE, markerNumber, (sptr_t)pixels);
}

void ScintillaEdit::registerRGBAImage(sptr_t type, const char * pixels) {
    send(SCI_REGISTERRGBAIMAGE, type, (sptr_t)pixels);
}

void ScintillaEdit::scrollToStart() {
    send(SCI_SCROLLTOSTART, 0, 0);
}

void ScintillaEdit::scrollToEnd() {
    send(SCI_SCROLLTOEND, 0, 0);
}

void ScintillaEdit::setTechnology(sptr_t technology) {
    send(SCI_SETTECHNOLOGY, technology, 0);
}

sptr_t ScintillaEdit::technology() const {
    return send(SCI_GETTECHNOLOGY, 0, 0);
}

sptr_t ScintillaEdit::createLoader(sptr_t bytes) {
    return send(SCI_CREATELOADER, bytes, 0);
}

void ScintillaEdit::findIndicatorShow(sptr_t start, sptr_t end) {
    send(SCI_FINDINDICATORSHOW, start, end);
}

void ScintillaEdit::findIndicatorFlash(sptr_t start, sptr_t end) {
    send(SCI_FINDINDICATORFLASH, start, end);
}

void ScintillaEdit::findIndicatorHide() {
    send(SCI_FINDINDICATORHIDE, 0, 0);
}

void ScintillaEdit::vCHomeDisplay() {
    send(SCI_VCHOMEDISPLAY, 0, 0);
}

void ScintillaEdit::vCHomeDisplayExtend() {
    send(SCI_VCHOMEDISPLAYEXTEND, 0, 0);
}

bool ScintillaEdit::caretLineVisibleAlways() const {
    return send(SCI_GETCARETLINEVISIBLEALWAYS, 0, 0);
}

void ScintillaEdit::setCaretLineVisibleAlways(bool alwaysVisible) {
    send(SCI_SETCARETLINEVISIBLEALWAYS, alwaysVisible, 0);
}

void ScintillaEdit::setLineEndTypesAllowed(sptr_t lineEndBitSet) {
    send(SCI_SETLINEENDTYPESALLOWED, lineEndBitSet, 0);
}

sptr_t ScintillaEdit::lineEndTypesAllowed() const {
    return send(SCI_GETLINEENDTYPESALLOWED, 0, 0);
}

sptr_t ScintillaEdit::lineEndTypesActive() const {
    return send(SCI_GETLINEENDTYPESACTIVE, 0, 0);
}

void ScintillaEdit::setRepresentation(const char * encodedCharacter, const char * representation) {
    send(SCI_SETREPRESENTATION, (sptr_t)encodedCharacter, (sptr_t)representation);
}

QByteArray ScintillaEdit::representation(const char * encodedCharacter) const {
    return TextReturner(SCI_GETREPRESENTATION, (sptr_t)encodedCharacter);
}

void ScintillaEdit::clearRepresentation(const char * encodedCharacter) {
    send(SCI_CLEARREPRESENTATION, (sptr_t)encodedCharacter, 0);
}

void ScintillaEdit::startRecord() {
    send(SCI_STARTRECORD, 0, 0);
}

void ScintillaEdit::stopRecord() {
    send(SCI_STOPRECORD, 0, 0);
}

void ScintillaEdit::setLexer(sptr_t lexer) {
    send(SCI_SETLEXER, lexer, 0);
}

sptr_t ScintillaEdit::lexer() const {
    return send(SCI_GETLEXER, 0, 0);
}

void ScintillaEdit::colourise(sptr_t start, sptr_t end) {
    send(SCI_COLOURISE, start, end);
}

void ScintillaEdit::setProperty(const char * key, const char * value) {
    send(SCI_SETPROPERTY, (sptr_t)key, (sptr_t)value);
}

void ScintillaEdit::setKeyWords(sptr_t keyWordSet, const char * keyWords) {
    send(SCI_SETKEYWORDS, keyWordSet, (sptr_t)keyWords);
}

void ScintillaEdit::setLexerLanguage(const char * language) {
    send(SCI_SETLEXERLANGUAGE, 0, (sptr_t)language);
}

void ScintillaEdit::loadLexerLibrary(const char * path) {
    send(SCI_LOADLEXERLIBRARY, 0, (sptr_t)path);
}

QByteArray ScintillaEdit::property(const char * key) const {
    return TextReturner(SCI_GETPROPERTY, (sptr_t)key);
}

QByteArray ScintillaEdit::propertyExpanded(const char * key) const {
    return TextReturner(SCI_GETPROPERTYEXPANDED, (sptr_t)key);
}

sptr_t ScintillaEdit::propertyInt(const char * key, sptr_t defaultValue) const {
    return send(SCI_GETPROPERTYINT, (sptr_t)key, defaultValue);
}

sptr_t ScintillaEdit::styleBitsNeeded() const {
    return send(SCI_GETSTYLEBITSNEEDED, 0, 0);
}

QByteArray ScintillaEdit::lexerLanguage() const {
    return TextReturner(SCI_GETLEXERLANGUAGE, 0);
}

sptr_t ScintillaEdit::privateLexerCall(sptr_t operation, sptr_t pointer) {
    return send(SCI_PRIVATELEXERCALL, operation, pointer);
}

QByteArray ScintillaEdit::propertyNames() {
    return TextReturner(SCI_PROPERTYNAMES, 0);
}

sptr_t ScintillaEdit::propertyType(const char * name) {
    return send(SCI_PROPERTYTYPE, (sptr_t)name, 0);
}

QByteArray ScintillaEdit::describeProperty(const char * name) {
    return TextReturner(SCI_DESCRIBEPROPERTY, (sptr_t)name);
}

QByteArray ScintillaEdit::describeKeyWordSets() {
    return TextReturner(SCI_DESCRIBEKEYWORDSETS, 0);
}

sptr_t ScintillaEdit::lineEndTypesSupported() const {
    return send(SCI_GETLINEENDTYPESSUPPORTED, 0, 0);
}

sptr_t ScintillaEdit::allocateSubStyles(sptr_t styleBase, sptr_t numberStyles) {
    return send(SCI_ALLOCATESUBSTYLES, styleBase, numberStyles);
}

sptr_t ScintillaEdit::subStylesStart(sptr_t styleBase) const {
    return send(SCI_GETSUBSTYLESSTART, styleBase, 0);
}

sptr_t ScintillaEdit::subStylesLength(sptr_t styleBase) const {
    return send(SCI_GETSUBSTYLESLENGTH, styleBase, 0);
}

sptr_t ScintillaEdit::styleFromSubStyle(sptr_t subStyle) const {
    return send(SCI_GETSTYLEFROMSUBSTYLE, subStyle, 0);
}

sptr_t ScintillaEdit::primaryStyleFromStyle(sptr_t style) const {
    return send(SCI_GETPRIMARYSTYLEFROMSTYLE, style, 0);
}

void ScintillaEdit::freeSubStyles() {
    send(SCI_FREESUBSTYLES, 0, 0);
}

void ScintillaEdit::setIdentifiers(sptr_t style, const char * identifiers) {
    send(SCI_SETIDENTIFIERS, style, (sptr_t)identifiers);
}

sptr_t ScintillaEdit::distanceToSecondaryStyles() const {
    return send(SCI_DISTANCETOSECONDARYSTYLES, 0, 0);
}

QByteArray ScintillaEdit::subStyleBases() const {
    return TextReturner(SCI_GETSUBSTYLEBASES, 0);
}

/* --Autogenerated -- end of section automatically generated from Scintilla.iface */
