% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/polyCub.midpoint.R
\name{polyCub.midpoint}
\alias{polyCub.midpoint}
\title{Two-Dimensional Midpoint Rule}
\usage{
polyCub.midpoint(polyregion, f, ..., eps = NULL, dimyx = NULL, plot = FALSE)
}
\arguments{
\item{polyregion}{a polygonal integration domain.
It can be any object coercible to the \pkg{spatstat.geom} class
\code{"\link[spatstat.geom]{owin}"} via a corresponding
\code{\link[spatstat.geom]{as.owin}}-method.
Note that this includes polygons of the classes \code{"gpc.poly"} and
\code{"\linkS4class{SpatialPolygons}"}, because \pkg{polyCub} defines
methods \code{\link{as.owin.gpc.poly}} and
\code{\link{as.owin.SpatialPolygons}}, respectively.
\pkg{sf} also registers suitable \code{as.owin} methods for its
\code{"\link[sf:st_polygon]{(MULTI)POLYGON}"} classes.}

\item{f}{a two-dimensional real-valued function.
As its first argument it must take a coordinate matrix, i.e., a
numeric matrix with two columns, and it must return a numeric vector of
length the number of coordinates.}

\item{...}{further arguments for \code{f}.}

\item{eps}{width and height of the pixels (squares),
see \code{\link[spatstat.geom]{as.mask}}.}

\item{dimyx}{number of subdivisions in each dimension,
see \code{\link[spatstat.geom]{as.mask}}.}

\item{plot}{logical indicating if an illustrative plot of the numerical
integration should be produced.}
}
\value{
The approximated value of the integral of \code{f} over
\code{polyregion}.
}
\description{
The surface is converted to a binary pixel image
using the \code{\link[spatstat.geom]{as.im.function}} method from package
\pkg{spatstat.geom} (Baddeley et al., 2015).
The integral under the surface is then approximated as the
sum over (pixel area * f(pixel midpoint)).
}
\examples{
## a function to integrate (here: isotropic zero-mean Gaussian density)
f <- function (s, sigma = 5)
    exp(-rowSums(s^2)/2/sigma^2) / (2*pi*sigma^2)

## a simple polygon as integration domain
hexagon <- list(
    list(x = c(7.33, 7.33, 3, -1.33, -1.33, 3),
         y = c(-0.5, 4.5, 7, 4.5, -0.5, -3))
)

if (require("spatstat.geom")) {
    hexagon.owin <- owin(poly = hexagon)

    show_midpoint <- function (eps)
    {
        plotpolyf(hexagon.owin, f, xlim = c(-8,8), ylim = c(-8,8),
                  use.lattice = FALSE)
        ## add evaluation points to plot
        with(as.mask(hexagon.owin, eps = eps),
             points(expand.grid(xcol, yrow), col = t(m), pch = 20))
        title(main = paste("2D midpoint rule with eps =", eps))
    }

    ## show nodes (eps = 0.5)
    show_midpoint(0.5)

    ## show pixel image (eps = 0.5)
    polyCub.midpoint(hexagon.owin, f, eps = 0.5, plot = TRUE)

    ## use a decreasing pixel size (increasing number of nodes)
    for (eps in c(5, 3, 1, 0.5, 0.3, 0.1))
        cat(sprintf("eps = \%.1f: \%.7f\n", eps,
                    polyCub.midpoint(hexagon.owin, f, eps = eps)))
}
}
\references{
Baddeley A, Rubak E, Turner R (2015).
\emph{Spatial Point Patterns: Methodology and Applications with R}.
Chapman and Hall/CRC Press, London.
}
\seealso{
Other polyCub-methods: 
\code{\link{polyCub.SV}()},
\code{\link{polyCub.exact.Gauss}()},
\code{\link{polyCub.iso}()},
\code{\link{polyCub}()}
}
\concept{polyCub-methods}
\keyword{math}
\keyword{spatial}
