% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posixt.R
\name{posixt-arithmetic}
\alias{posixt-arithmetic}
\alias{add_years.POSIXt}
\alias{add_quarters.POSIXt}
\alias{add_months.POSIXt}
\alias{add_weeks.POSIXt}
\alias{add_days.POSIXt}
\alias{add_hours.POSIXt}
\alias{add_minutes.POSIXt}
\alias{add_seconds.POSIXt}
\title{Arithmetic: date-time}
\usage{
\method{add_years}{POSIXt}(x, n, ..., invalid = NULL, nonexistent = NULL, ambiguous = x)

\method{add_quarters}{POSIXt}(x, n, ..., invalid = NULL, nonexistent = NULL, ambiguous = x)

\method{add_months}{POSIXt}(x, n, ..., invalid = NULL, nonexistent = NULL, ambiguous = x)

\method{add_weeks}{POSIXt}(x, n, ..., nonexistent = NULL, ambiguous = x)

\method{add_days}{POSIXt}(x, n, ..., nonexistent = NULL, ambiguous = x)

\method{add_hours}{POSIXt}(x, n, ...)

\method{add_minutes}{POSIXt}(x, n, ...)

\method{add_seconds}{POSIXt}(x, n, ...)
}
\arguments{
\item{x}{\verb{[POSIXct / POSIXlt]}

A date-time vector.}

\item{n}{\verb{[integer / clock_duration]}

An integer vector to be converted to a duration, or a duration
corresponding to the arithmetic function being used. This corresponds
to the number of duration units to add. \code{n} may be negative to subtract
units of duration.}

\item{...}{These dots are for future extensions and must be empty.}

\item{invalid}{\verb{[character(1) / NULL]}

One of the following invalid date resolution strategies:
\itemize{
\item \code{"previous"}: The previous valid instant in time.
\item \code{"previous-day"}: The previous valid day in time, keeping the time of
day.
\item \code{"next"}: The next valid instant in time.
\item \code{"next-day"}: The next valid day in time, keeping the time of day.
\item \code{"overflow"}: Overflow by the number of days that the input is invalid
by. Time of day is dropped.
\item \code{"overflow-day"}: Overflow by the number of days that the input is
invalid by. Time of day is kept.
\item \code{"NA"}: Replace invalid dates with \code{NA}.
\item \code{"error"}: Error on invalid dates.
}

Using either \code{"previous"} or \code{"next"} is generally recommended, as these
two strategies maintain the \emph{relative ordering} between elements of the
input.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{invalid} must be supplied and
cannot be \code{NULL}. This is a convenient way to make production code robust
to invalid dates.}

\item{nonexistent}{\verb{[character / NULL]}

One of the following nonexistent time resolution strategies, allowed to be
either length 1, or the same length as the input:
\itemize{
\item \code{"roll-forward"}: The next valid instant in time.
\item \code{"roll-backward"}: The previous valid instant in time.
\item \code{"shift-forward"}: Shift the nonexistent time forward by the size of
the daylight saving time gap.
\item \verb{"shift-backward}: Shift the nonexistent time backward by the size of
the daylight saving time gap.
\item \code{"NA"}: Replace nonexistent times with \code{NA}.
\item \code{"error"}: Error on nonexistent times.
}

Using either \code{"roll-forward"} or \code{"roll-backward"} is generally
recommended over shifting, as these two strategies maintain the
\emph{relative ordering} between elements of the input.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{nonexistent} must be supplied
and cannot be \code{NULL}. This is a convenient way to make production code
robust to nonexistent times.}

\item{ambiguous}{\verb{[character / zoned_time / POSIXct / list(2) / NULL]}

One of the following ambiguous time resolution strategies, allowed to be
either length 1, or the same length as the input:
\itemize{
\item \code{"earliest"}: Of the two possible times, choose the earliest one.
\item \code{"latest"}: Of the two possible times, choose the latest one.
\item \code{"NA"}: Replace ambiguous times with \code{NA}.
\item \code{"error"}: Error on ambiguous times.
}

Alternatively, \code{ambiguous} is allowed to be a zoned_time (or POSIXct) that
is either length 1, or the same length as the input. If an ambiguous time
is encountered, the zoned_time is consulted. If the zoned_time corresponds
to a naive_time that is also ambiguous \emph{and} uses the same daylight saving
time transition point as the original ambiguous time, then the offset of
the zoned_time is used to resolve the ambiguity. If the ambiguity cannot be
resolved by consulting the zoned_time, then this method falls back to
\code{NULL}.

Finally, \code{ambiguous} is allowed to be a list of size 2, where the first
element of the list is a zoned_time (as described above), and the second
element of the list is an ambiguous time resolution strategy to use when
the ambiguous time cannot be resolved by consulting the zoned_time.
Specifying a zoned_time on its own is identical to \verb{list(<zoned_time>, NULL)}.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{ambiguous} must be supplied and
cannot be \code{NULL}. Additionally, \code{ambiguous} cannot be specified as a
zoned_time on its own, as this implies \code{NULL} for ambiguous times that the
zoned_time cannot resolve. Instead, it must be specified as a list
alongside an ambiguous time resolution strategy as described above. This is
a convenient way to make production code robust to ambiguous times.}
}
\value{
\code{x} after performing the arithmetic.
}
\description{
These are POSIXct/POSIXlt methods for the
\link[=clock-arithmetic]{arithmetic generics}.

Calendrical based arithmetic:

These functions convert to a naive-time, then to a year-month-day, perform
the arithmetic, then convert back to a date-time.
\itemize{
\item \code{add_years()}
\item \code{add_quarters()}
\item \code{add_months()}
}

Naive-time based arithmetic:

These functions convert to a naive-time, perform the arithmetic, then
convert back to a date-time.
\itemize{
\item \code{add_weeks()}
\item \code{add_days()}
}

Sys-time based arithmetic:

These functions convert to a sys-time, perform the arithmetic, then
convert back to a date-time.
\itemize{
\item \code{add_hours()}
\item \code{add_minutes()}
\item \code{add_seconds()}
}
}
\details{
Adding a single quarter with \code{add_quarters()} is equivalent to adding
3 months.

\code{x} and \code{n} are recycled against each other.

Calendrical based arithmetic has the potential to generate invalid dates
(like the 31st of February), nonexistent times (due to daylight saving
time gaps), and ambiguous times (due to daylight saving time fallbacks).

Naive-time based arithmetic will never generate an invalid date, but
may generate a nonexistent or ambiguous time (i.e. you added 1 day and
landed in a daylight saving time gap).

Sys-time based arithmetic operates in the UTC time zone, which means
that it will never generate any invalid dates or nonexistent / ambiguous
times.

The conversion from POSIXct/POSIXlt to the corresponding clock type uses
a "best guess" about whether you want to do the arithmetic using a naive-time
or a sys-time. For example, when adding months, you probably want to
retain the printed time when converting to a year-month-day to perform
the arithmetic, so the conversion goes through naive-time. However,
when adding smaller units like seconds, you probably want
\verb{"2020-03-08 01:59:59" + 1 second} in the America/New_York time zone to
return \code{"2020-03-08 03:00:00"}, taking into account the fact that there
was a daylight saving time gap. This requires doing the arithmetic in
sys-time, so that is what clock converts to. If you disagree with this
heuristic for any reason, you can take control and perform the conversions
yourself. For example, you could convert the previous example to a
naive-time instead of a sys-time manually with \code{\link[=as_naive_time]{as_naive_time()}}, add
1 second giving \code{"2020-03-08 02:00:00"}, then convert back to a
POSIXct/POSIXlt, dealing with the nonexistent time that gets created by
using the \code{nonexistent} argument of \code{as.POSIXct()}.
}
\examples{
x <- as.POSIXct("2019-01-01", tz = "America/New_York")

add_years(x, 1:5)

y <- as.POSIXct("2019-01-31 00:30:00", tz = "America/New_York")

# Adding 1 month to `y` generates an invalid date. Unlike year-month-day
# types, R's native date-time types cannot handle invalid dates, so you must
# resolve them immediately. If you don't you get an error:
try(add_months(y, 1:2))
add_months(as_year_month_day(y), 1:2)

# Resolve invalid dates by specifying an invalid date resolution strategy
# with the `invalid` argument. Using `"previous"` here sets the date-time to
# the previous valid moment in time - i.e. the end of the month. The
# time is set to the last moment in the day to retain the relative ordering
# within your input. If you are okay with potentially losing this, and
# want to retain your time of day, you can use `"previous-day"` to set the
# date-time to the previous valid day, while keeping the time of day.
add_months(y, 1:2, invalid = "previous")
add_months(y, 1:2, invalid = "previous-day")
}
