/*
 * QEMU Crypto cipher algorithms
 *
 * Copyright (c) 2022 Bytedance
 * Author: lei he <helei.sig11@bytedance.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "qemu/osdep.h"

#include "crypto/init.h"
#include "crypto/akcipher.h"
#include "qapi/error.h"

static const uint8_t rsa1024_private_key[] = {
    0x30, 0x82, 0x02, 0x5c, 0x02, 0x01, 0x00, 0x02,
    0x81, 0x81, 0x00, 0xe6, 0x4d, 0x76, 0x4f, 0xb2,
    0x97, 0x09, 0xad, 0x9d, 0x17, 0x33, 0xf2, 0x30,
    0x42, 0x83, 0xa9, 0xcb, 0x49, 0xa4, 0x2e, 0x59,
    0x5e, 0x75, 0x51, 0xd1, 0xac, 0xc8, 0x86, 0x3e,
    0xdb, 0x72, 0x2e, 0xb2, 0xf7, 0xc3, 0x5b, 0xc7,
    0xea, 0xed, 0x30, 0xd1, 0xf7, 0x37, 0xee, 0x9d,
    0x36, 0x59, 0x6f, 0xf8, 0xce, 0xc0, 0x5c, 0x82,
    0x80, 0x37, 0x83, 0xd7, 0x45, 0x6a, 0xe9, 0xea,
    0xc5, 0x3a, 0x59, 0x6b, 0x34, 0x31, 0x44, 0x00,
    0x74, 0xa7, 0x29, 0xab, 0x79, 0x4a, 0xbd, 0xe8,
    0x25, 0x35, 0x01, 0x11, 0x40, 0xbf, 0x31, 0xbd,
    0xd3, 0xe0, 0x68, 0x1e, 0xd5, 0x5b, 0x2f, 0xe9,
    0x20, 0xf2, 0x9f, 0x46, 0x35, 0x30, 0xa8, 0xf1,
    0xfe, 0xef, 0xd8, 0x76, 0x23, 0x46, 0x34, 0x70,
    0xa1, 0xce, 0xc6, 0x65, 0x6d, 0xb0, 0x94, 0x7e,
    0xe5, 0x92, 0x45, 0x7b, 0xaa, 0xbb, 0x95, 0x97,
    0x77, 0xcd, 0xd3, 0x02, 0x03, 0x01, 0x00, 0x01,
    0x02, 0x81, 0x80, 0x30, 0x6a, 0xc4, 0x9e, 0xc8,
    0xba, 0xfc, 0x2b, 0xe5, 0xc4, 0xc5, 0x04, 0xfb,
    0xa4, 0x60, 0x2d, 0xc8, 0x31, 0x39, 0x35, 0x0d,
    0x50, 0xd0, 0x75, 0x5d, 0x11, 0x68, 0x2e, 0xe0,
    0xf4, 0x1d, 0xb3, 0x37, 0xa8, 0xe3, 0x07, 0x5e,
    0xa6, 0x43, 0x2b, 0x6a, 0x59, 0x01, 0x07, 0x47,
    0x41, 0xef, 0xd7, 0x9c, 0x85, 0x4a, 0xe7, 0xa7,
    0xff, 0xf0, 0xab, 0xe5, 0x0c, 0x11, 0x08, 0x10,
    0x75, 0x5a, 0x68, 0xa0, 0x08, 0x03, 0xc9, 0x40,
    0x79, 0x67, 0x1d, 0x65, 0x89, 0x2d, 0x08, 0xf9,
    0xb5, 0x1b, 0x7d, 0xd2, 0x41, 0x3b, 0x33, 0xf2,
    0x47, 0x2f, 0x9c, 0x0b, 0xd5, 0xaf, 0xcb, 0xdb,
    0xbb, 0x37, 0x63, 0x03, 0xf8, 0xe7, 0x2e, 0xc7,
    0x3c, 0x86, 0x9f, 0xc2, 0x9b, 0xb4, 0x70, 0x6a,
    0x4d, 0x7c, 0xe4, 0x1b, 0x3a, 0xa9, 0xae, 0xd7,
    0xce, 0x7f, 0x56, 0xc2, 0x73, 0x5e, 0x58, 0x63,
    0xd5, 0x86, 0x41, 0x02, 0x41, 0x00, 0xf6, 0x56,
    0x69, 0xec, 0xef, 0x65, 0x95, 0xdc, 0x25, 0x47,
    0xe0, 0x6f, 0xb0, 0x4f, 0x79, 0x77, 0x0a, 0x5e,
    0x46, 0xcb, 0xbd, 0x0b, 0x71, 0x51, 0x2a, 0xa4,
    0x65, 0x29, 0x18, 0xc6, 0x30, 0xa0, 0x95, 0x4c,
    0x4b, 0xbe, 0x8c, 0x40, 0xe3, 0x9c, 0x23, 0x02,
    0x14, 0x43, 0xe9, 0x64, 0xea, 0xe3, 0xa8, 0xe2,
    0x1a, 0xd5, 0xf9, 0x5c, 0xe0, 0x36, 0x2c, 0x97,
    0xda, 0xd5, 0xc7, 0x46, 0xce, 0x11, 0x02, 0x41,
    0x00, 0xef, 0x56, 0x08, 0xb8, 0x29, 0xa5, 0xa6,
    0x7c, 0xf7, 0x5f, 0xb4, 0xf5, 0x63, 0xe7, 0xeb,
    0x45, 0xfd, 0x89, 0xaa, 0x94, 0xa6, 0x3d, 0x0b,
    0xd9, 0x04, 0x6f, 0x78, 0xe0, 0xbb, 0xa2, 0xd4,
    0x29, 0x83, 0x17, 0x95, 0x6f, 0x50, 0x3d, 0x40,
    0x5d, 0xe5, 0x24, 0xda, 0xc2, 0x23, 0x50, 0x86,
    0xa8, 0x34, 0xc8, 0x6f, 0xec, 0x7f, 0xb6, 0x45,
    0x3a, 0xdd, 0x78, 0x9b, 0xee, 0xa1, 0xe4, 0x09,
    0xa3, 0x02, 0x40, 0x5c, 0xd6, 0x66, 0x67, 0x58,
    0x35, 0xc5, 0xcb, 0xc8, 0xf5, 0x14, 0xbd, 0xa3,
    0x09, 0xe0, 0xb2, 0x1f, 0x63, 0x36, 0x75, 0x34,
    0x52, 0xea, 0xaa, 0xf7, 0x52, 0x2b, 0x99, 0xd8,
    0x6f, 0x61, 0x06, 0x34, 0x1e, 0x23, 0xf1, 0xb5,
    0x34, 0x03, 0x53, 0xe5, 0xd1, 0xb3, 0xc7, 0x80,
    0x5f, 0x7b, 0x32, 0xbf, 0x84, 0x2f, 0x2e, 0xf3,
    0x22, 0xb0, 0x91, 0x5a, 0x2f, 0x04, 0xd7, 0x4a,
    0x9a, 0x01, 0xb1, 0x02, 0x40, 0x34, 0x0b, 0x26,
    0x4c, 0x3d, 0xaa, 0x2a, 0xc0, 0xe3, 0xdd, 0xe8,
    0xf0, 0xaf, 0x6f, 0xe0, 0x06, 0x51, 0x32, 0x9d,
    0x68, 0x43, 0x99, 0xe4, 0xb8, 0xa5, 0x31, 0x44,
    0x3c, 0xc2, 0x30, 0x8f, 0x28, 0x13, 0xbc, 0x8e,
    0x1f, 0x2d, 0x78, 0x94, 0x45, 0x96, 0xad, 0x63,
    0xf0, 0x71, 0x53, 0x72, 0x64, 0xa3, 0x4d, 0xae,
    0xa0, 0xe3, 0xc8, 0x93, 0xd7, 0x50, 0x0f, 0x89,
    0x00, 0xe4, 0x2d, 0x3d, 0x37, 0x02, 0x41, 0x00,
    0xbe, 0xa6, 0x08, 0xe0, 0xc8, 0x15, 0x2a, 0x47,
    0xcb, 0xd5, 0xec, 0x93, 0xd3, 0xaa, 0x12, 0x82,
    0xaf, 0xac, 0x51, 0x5a, 0x5b, 0xa7, 0x93, 0x4b,
    0xb9, 0xab, 0x00, 0xfa, 0x5a, 0xea, 0x34, 0xe4,
    0x80, 0xf1, 0x44, 0x6a, 0x65, 0xe4, 0x33, 0x99,
    0xfb, 0x54, 0xd7, 0x89, 0x5a, 0x1b, 0xd6, 0x2b,
    0xcc, 0x6e, 0x4b, 0x19, 0xa0, 0x6d, 0x93, 0x9f,
    0xc3, 0x91, 0x7a, 0xa5, 0xd8, 0x59, 0x0e, 0x9e,
};

static const uint8_t rsa1024_public_key[] = {
    0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xe6,
    0x4d, 0x76, 0x4f, 0xb2, 0x97, 0x09, 0xad, 0x9d,
    0x17, 0x33, 0xf2, 0x30, 0x42, 0x83, 0xa9, 0xcb,
    0x49, 0xa4, 0x2e, 0x59, 0x5e, 0x75, 0x51, 0xd1,
    0xac, 0xc8, 0x86, 0x3e, 0xdb, 0x72, 0x2e, 0xb2,
    0xf7, 0xc3, 0x5b, 0xc7, 0xea, 0xed, 0x30, 0xd1,
    0xf7, 0x37, 0xee, 0x9d, 0x36, 0x59, 0x6f, 0xf8,
    0xce, 0xc0, 0x5c, 0x82, 0x80, 0x37, 0x83, 0xd7,
    0x45, 0x6a, 0xe9, 0xea, 0xc5, 0x3a, 0x59, 0x6b,
    0x34, 0x31, 0x44, 0x00, 0x74, 0xa7, 0x29, 0xab,
    0x79, 0x4a, 0xbd, 0xe8, 0x25, 0x35, 0x01, 0x11,
    0x40, 0xbf, 0x31, 0xbd, 0xd3, 0xe0, 0x68, 0x1e,
    0xd5, 0x5b, 0x2f, 0xe9, 0x20, 0xf2, 0x9f, 0x46,
    0x35, 0x30, 0xa8, 0xf1, 0xfe, 0xef, 0xd8, 0x76,
    0x23, 0x46, 0x34, 0x70, 0xa1, 0xce, 0xc6, 0x65,
    0x6d, 0xb0, 0x94, 0x7e, 0xe5, 0x92, 0x45, 0x7b,
    0xaa, 0xbb, 0x95, 0x97, 0x77, 0xcd, 0xd3, 0x02,
    0x03, 0x01, 0x00, 0x01,
};

static const uint8_t rsa2048_private_key[] = {
    0x30, 0x82, 0x04, 0xa4, 0x02, 0x01, 0x00, 0x02,
    0x82, 0x01, 0x01, 0x00, 0xbd, 0x9c, 0x83, 0x6b,
    0x0e, 0x8e, 0xcf, 0xfa, 0xaa, 0x4f, 0x6a, 0xf4,
    0xe3, 0x52, 0x0f, 0xa5, 0xd0, 0xbe, 0x5e, 0x7f,
    0x08, 0x24, 0xba, 0x87, 0x46, 0xfb, 0x28, 0x93,
    0xe5, 0xe5, 0x81, 0x42, 0xc0, 0xf9, 0x17, 0xc7,
    0x81, 0x01, 0xf4, 0x18, 0x6a, 0x17, 0xf5, 0x57,
    0x20, 0x37, 0xcf, 0xf9, 0x74, 0x5e, 0xe1, 0x48,
    0x6a, 0x71, 0x0a, 0x0f, 0x79, 0x72, 0x2b, 0x46,
    0x10, 0x53, 0xdc, 0x14, 0x43, 0xbd, 0xbc, 0x6d,
    0x15, 0x6f, 0x15, 0x4e, 0xf0, 0x0d, 0x89, 0x39,
    0x02, 0xc3, 0x68, 0x5c, 0xa8, 0xfc, 0xed, 0x64,
    0x9d, 0x98, 0xb7, 0xcd, 0x83, 0x66, 0x93, 0xc3,
    0xd9, 0x57, 0xa0, 0x21, 0x93, 0xad, 0x5c, 0x75,
    0x69, 0x88, 0x9e, 0x81, 0xdc, 0x7f, 0x1d, 0xd5,
    0xbd, 0x1c, 0xc1, 0x30, 0x56, 0xa5, 0xda, 0x99,
    0x46, 0xa6, 0x6d, 0x0e, 0x6f, 0x5e, 0x51, 0x34,
    0x49, 0x73, 0xc3, 0x67, 0x49, 0x7e, 0x21, 0x2a,
    0x20, 0xa7, 0x2b, 0x92, 0x73, 0x1d, 0xa5, 0x25,
    0x2a, 0xd0, 0x3a, 0x89, 0x75, 0xb2, 0xbb, 0x19,
    0x37, 0x78, 0x48, 0xd2, 0xf2, 0x2a, 0x6d, 0x9e,
    0xc6, 0x26, 0xca, 0x46, 0x8c, 0xf1, 0x42, 0x2a,
    0x31, 0xb2, 0xfc, 0xe7, 0x55, 0x51, 0xff, 0x07,
    0x13, 0x5b, 0x36, 0x59, 0x2b, 0x43, 0x30, 0x4b,
    0x05, 0x5c, 0xd2, 0x45, 0xa0, 0xa0, 0x7c, 0x17,
    0x5b, 0x07, 0xbb, 0x5d, 0x83, 0x80, 0x92, 0x6d,
    0x87, 0x1a, 0x43, 0xac, 0xc7, 0x6b, 0x8d, 0x11,
    0x60, 0x27, 0xd2, 0xdf, 0xdb, 0x71, 0x02, 0x55,
    0x6e, 0xb5, 0xca, 0x4d, 0xda, 0x59, 0x0d, 0xb8,
    0x8c, 0xcd, 0xd3, 0x0e, 0x55, 0xa0, 0xa4, 0x8d,
    0xa0, 0x14, 0x10, 0x48, 0x42, 0x35, 0x56, 0x08,
    0xf7, 0x29, 0x5f, 0xa2, 0xea, 0xa4, 0x5e, 0x8e,
    0x99, 0x56, 0xaa, 0x5a, 0x8c, 0x23, 0x8f, 0x35,
    0x22, 0x8a, 0xff, 0xed, 0x02, 0x03, 0x01, 0x00,
    0x01, 0x02, 0x82, 0x01, 0x00, 0x4e, 0x4a, 0xf3,
    0x44, 0xe0, 0x64, 0xfd, 0xe1, 0xde, 0x33, 0x1e,
    0xd1, 0xf1, 0x8f, 0x6f, 0xe0, 0xa2, 0xfa, 0x08,
    0x60, 0xe1, 0xc6, 0xf0, 0xb2, 0x6d, 0x0f, 0xc6,
    0x28, 0x93, 0xb4, 0x19, 0x94, 0xab, 0xc3, 0xef,
    0x1a, 0xb4, 0xdd, 0x4e, 0xa2, 0x4a, 0x24, 0x8c,
    0x6c, 0xa6, 0x64, 0x05, 0x5f, 0x56, 0xba, 0xda,
    0xc1, 0x21, 0x1a, 0x7d, 0xf1, 0xf7, 0xce, 0xb9,
    0xa9, 0x9b, 0x92, 0x54, 0xfc, 0x95, 0x20, 0x22,
    0x4e, 0xd4, 0x9b, 0xe2, 0xab, 0x8e, 0x99, 0xb8,
    0x40, 0xaf, 0x30, 0x6a, 0xc6, 0x60, 0x0c, 0xd8,
    0x25, 0x44, 0xa1, 0xcb, 0xbb, 0x73, 0x77, 0x86,
    0xaa, 0x46, 0xf3, 0x54, 0xae, 0xa8, 0xa0, 0xdb,
    0xdd, 0xab, 0x6e, 0xfb, 0x2c, 0x5a, 0x14, 0xaf,
    0x08, 0x13, 0xa7, 0x6c, 0xe9, 0xfd, 0xcd, 0x4c,
    0x1f, 0x20, 0x3a, 0x16, 0x2b, 0xf0, 0xb6, 0x7c,
    0x47, 0x5f, 0xd1, 0x0a, 0x2c, 0xc4, 0xa5, 0x68,
    0xd0, 0x43, 0x75, 0x6b, 0x65, 0xaa, 0x32, 0xc6,
    0x99, 0x06, 0xcb, 0x8f, 0xe6, 0x8d, 0xce, 0xbf,
    0x4d, 0x0d, 0x7b, 0x22, 0x2a, 0x8a, 0xcb, 0x7d,
    0x7f, 0x16, 0x48, 0x85, 0xf1, 0x86, 0xcb, 0x54,
    0xb9, 0x39, 0xd4, 0xbc, 0xe3, 0x2d, 0x27, 0x59,
    0xf6, 0x81, 0x5e, 0x94, 0x45, 0xdf, 0xb9, 0x22,
    0xaf, 0x64, 0x0d, 0x14, 0xec, 0x8c, 0xeb, 0x71,
    0xac, 0xee, 0x09, 0x4c, 0xbf, 0x34, 0xf9, 0xf4,
    0x66, 0x77, 0x36, 0x3b, 0x41, 0x74, 0x01, 0x4f,
    0xfc, 0x56, 0x83, 0xba, 0x14, 0xb0, 0x2f, 0xdd,
    0x4d, 0xb9, 0x3f, 0xdf, 0x71, 0xbe, 0x7b, 0xba,
    0x66, 0xc8, 0xc5, 0x42, 0xc9, 0xba, 0x18, 0x63,
    0x45, 0x07, 0x2f, 0x84, 0x3e, 0xc3, 0xfb, 0x47,
    0xda, 0xd4, 0x1d, 0x0e, 0x9d, 0x96, 0xc0, 0xea,
    0xee, 0x45, 0x2f, 0xe1, 0x62, 0x23, 0xee, 0xef,
    0x3d, 0x5e, 0x55, 0xa1, 0x0d, 0x02, 0x81, 0x81,
    0x00, 0xeb, 0x76, 0x88, 0xd3, 0xae, 0x3f, 0x1d,
    0xf2, 0x49, 0xe0, 0x37, 0x49, 0x83, 0x82, 0x6c,
    0xf7, 0xf1, 0x17, 0x30, 0x75, 0x2e, 0x89, 0x06,
    0x88, 0x56, 0x32, 0xf6, 0xfa, 0x58, 0xcb, 0x3c,
    0x98, 0x67, 0xc3, 0xde, 0x10, 0x82, 0xe5, 0xfa,
    0xfa, 0x52, 0x47, 0x8d, 0xd7, 0x00, 0xc6, 0xcb,
    0xf7, 0xf6, 0x57, 0x9b, 0x6e, 0x0c, 0xac, 0xe8,
    0x3b, 0xd1, 0xde, 0xb5, 0x34, 0xaf, 0x8b, 0x2a,
    0xb0, 0x2d, 0x01, 0xeb, 0x7c, 0xa0, 0x42, 0x26,
    0xbb, 0x2b, 0x43, 0x0e, 0x1d, 0xe2, 0x4e, 0xc9,
    0xc1, 0x0a, 0x67, 0x1d, 0xfc, 0x83, 0x25, 0xce,
    0xb2, 0x18, 0xd9, 0x0d, 0x70, 0xf5, 0xa3, 0x5a,
    0x9c, 0x99, 0xdd, 0x47, 0xa1, 0x57, 0xe7, 0x20,
    0xde, 0xa1, 0x29, 0x8d, 0x96, 0x62, 0xf9, 0x26,
    0x95, 0x51, 0xa6, 0xe7, 0x09, 0x8b, 0xba, 0x16,
    0x8b, 0x19, 0x5b, 0xf9, 0x27, 0x0d, 0xc5, 0xd6,
    0x5f, 0x02, 0x81, 0x81, 0x00, 0xce, 0x26, 0x31,
    0xb5, 0x43, 0x53, 0x95, 0x39, 0xdd, 0x01, 0x98,
    0x8b, 0x3d, 0x27, 0xeb, 0x0b, 0x87, 0x1c, 0x95,
    0xfc, 0x3e, 0x36, 0x51, 0x31, 0xb5, 0xea, 0x59,
    0x56, 0xc0, 0x97, 0x62, 0xf0, 0x63, 0x2b, 0xb6,
    0x30, 0x9b, 0xdf, 0x19, 0x10, 0xe9, 0xa0, 0x3d,
    0xea, 0x54, 0x5a, 0xe6, 0xc6, 0x9e, 0x7e, 0xb5,
    0xf0, 0xb0, 0x54, 0xef, 0xc3, 0xe1, 0x47, 0xa6,
    0x95, 0xc7, 0xe4, 0xa3, 0x4a, 0x30, 0x68, 0x24,
    0x98, 0x7d, 0xc1, 0x34, 0xa9, 0xcb, 0xbc, 0x3c,
    0x08, 0x9c, 0x7d, 0x0c, 0xa2, 0xb7, 0x60, 0xaa,
    0x38, 0x08, 0x16, 0xa6, 0x7f, 0xdb, 0xd2, 0xb1,
    0x67, 0xe7, 0x93, 0x8e, 0xbb, 0x7e, 0xb9, 0xb5,
    0xd0, 0xd0, 0x9f, 0x7b, 0xcc, 0x46, 0xe6, 0x74,
    0x78, 0x1a, 0x96, 0xd6, 0xd7, 0x74, 0x34, 0x54,
    0x3b, 0x54, 0x55, 0x7f, 0x89, 0x81, 0xbc, 0x40,
    0x55, 0x87, 0x24, 0x95, 0x33, 0x02, 0x81, 0x81,
    0x00, 0xb0, 0x18, 0x5d, 0x2a, 0x1a, 0x95, 0x9f,
    0x9a, 0xd5, 0x3f, 0x37, 0x79, 0xe6, 0x3d, 0x83,
    0xab, 0x46, 0x86, 0x36, 0x3a, 0x5d, 0x0c, 0x23,
    0x73, 0x91, 0x2b, 0xda, 0x63, 0xce, 0x46, 0x68,
    0xd1, 0xfe, 0x40, 0x90, 0xf2, 0x3e, 0x43, 0x2b,
    0x19, 0x4c, 0xb1, 0xb0, 0xd5, 0x8c, 0x02, 0x21,
    0x07, 0x18, 0x17, 0xda, 0xe9, 0x49, 0xd7, 0x82,
    0x73, 0x42, 0x78, 0xd1, 0x82, 0x4e, 0x8a, 0xc0,
    0xe9, 0x33, 0x2f, 0xcd, 0x62, 0xce, 0x23, 0xca,
    0xfd, 0x8d, 0xd4, 0x3f, 0x59, 0x80, 0x27, 0xb6,
    0x61, 0x85, 0x9b, 0x2a, 0xe4, 0xef, 0x5c, 0x36,
    0x22, 0x21, 0xcd, 0x2a, 0x6d, 0x41, 0x77, 0xe2,
    0xcb, 0x5d, 0x93, 0x0d, 0x00, 0x10, 0x52, 0x8d,
    0xd5, 0x92, 0x28, 0x16, 0x78, 0xd3, 0x1a, 0x4c,
    0x8d, 0xbd, 0x9c, 0x1a, 0x0b, 0x9c, 0x91, 0x16,
    0x4c, 0xff, 0x31, 0x36, 0xbb, 0xcb, 0x64, 0x1a,
    0xf7, 0x02, 0x81, 0x80, 0x32, 0x65, 0x09, 0xdf,
    0xca, 0xee, 0xa2, 0xdb, 0x3b, 0x58, 0xc9, 0x86,
    0xb8, 0x53, 0x8a, 0xd5, 0x0d, 0x99, 0x82, 0x5c,
    0xe0, 0x84, 0x7c, 0xc2, 0xcf, 0x3a, 0xd3, 0xce,
    0x2e, 0x54, 0x93, 0xbe, 0x3a, 0x30, 0x14, 0x60,
    0xbb, 0xaa, 0x05, 0x41, 0xaa, 0x2b, 0x1f, 0x17,
    0xaa, 0xb9, 0x72, 0x12, 0xf9, 0xe9, 0xf5, 0xe6,
    0x39, 0xe4, 0xf9, 0x9c, 0x03, 0xf5, 0x75, 0x16,
    0xc6, 0x7f, 0xf1, 0x1f, 0x10, 0xc8, 0x54, 0xb1,
    0xe6, 0x84, 0x15, 0xb0, 0xb0, 0x7a, 0x7a, 0x9e,
    0x8c, 0x4a, 0xd1, 0x8c, 0xf1, 0x91, 0x32, 0xeb,
    0x71, 0xa6, 0xbf, 0xdb, 0x1f, 0xcc, 0xd8, 0xcb,
    0x92, 0xc3, 0xf2, 0xaf, 0x89, 0x22, 0x32, 0xfd,
    0x32, 0x12, 0xda, 0xbb, 0xac, 0x55, 0x68, 0x01,
    0x78, 0x56, 0x89, 0x7c, 0xb0, 0x0e, 0x9e, 0xcc,
    0xc6, 0x28, 0x04, 0x7e, 0x83, 0xf5, 0x96, 0x30,
    0x92, 0x51, 0xf2, 0x1b, 0x02, 0x81, 0x81, 0x00,
    0x83, 0x6d, 0xd1, 0x98, 0x90, 0x41, 0x8c, 0xa7,
    0x92, 0x83, 0xac, 0x89, 0x05, 0x0c, 0x79, 0x67,
    0x90, 0xb6, 0xa1, 0xf3, 0x2f, 0xca, 0xf0, 0x15,
    0xe0, 0x30, 0x58, 0xe9, 0x4f, 0xcb, 0x4c, 0x56,
    0x56, 0x56, 0x14, 0x3f, 0x1b, 0x79, 0xb6, 0xef,
    0x57, 0x4b, 0x28, 0xbd, 0xb0, 0xe6, 0x0c, 0x49,
    0x4b, 0xbe, 0xe1, 0x57, 0x28, 0x2a, 0x23, 0x5e,
    0xc4, 0xa2, 0x19, 0x4b, 0x00, 0x67, 0x78, 0xd9,
    0x26, 0x6e, 0x17, 0x25, 0xce, 0xe4, 0xfd, 0xde,
    0x86, 0xa8, 0x5a, 0x67, 0x47, 0x6b, 0x15, 0x09,
    0xe1, 0xec, 0x8e, 0x62, 0x98, 0x91, 0x6f, 0xc0,
    0x98, 0x0c, 0x70, 0x0e, 0x7d, 0xbe, 0x63, 0xbd,
    0x12, 0x5a, 0x98, 0x1c, 0xe3, 0x0c, 0xfb, 0xc7,
    0xfb, 0x1b, 0xbd, 0x02, 0x87, 0xcc, 0x0c, 0xbb,
    0xc2, 0xd4, 0xb6, 0xc1, 0xa1, 0x23, 0xd3, 0x1e,
    0x21, 0x6f, 0x48, 0xba, 0x0e, 0x2e, 0xc7, 0x42
};

static const uint8_t rsa2048_public_key[] = {
    0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01,
    0x00, 0xbd, 0x9c, 0x83, 0x6b, 0x0e, 0x8e, 0xcf,
    0xfa, 0xaa, 0x4f, 0x6a, 0xf4, 0xe3, 0x52, 0x0f,
    0xa5, 0xd0, 0xbe, 0x5e, 0x7f, 0x08, 0x24, 0xba,
    0x87, 0x46, 0xfb, 0x28, 0x93, 0xe5, 0xe5, 0x81,
    0x42, 0xc0, 0xf9, 0x17, 0xc7, 0x81, 0x01, 0xf4,
    0x18, 0x6a, 0x17, 0xf5, 0x57, 0x20, 0x37, 0xcf,
    0xf9, 0x74, 0x5e, 0xe1, 0x48, 0x6a, 0x71, 0x0a,
    0x0f, 0x79, 0x72, 0x2b, 0x46, 0x10, 0x53, 0xdc,
    0x14, 0x43, 0xbd, 0xbc, 0x6d, 0x15, 0x6f, 0x15,
    0x4e, 0xf0, 0x0d, 0x89, 0x39, 0x02, 0xc3, 0x68,
    0x5c, 0xa8, 0xfc, 0xed, 0x64, 0x9d, 0x98, 0xb7,
    0xcd, 0x83, 0x66, 0x93, 0xc3, 0xd9, 0x57, 0xa0,
    0x21, 0x93, 0xad, 0x5c, 0x75, 0x69, 0x88, 0x9e,
    0x81, 0xdc, 0x7f, 0x1d, 0xd5, 0xbd, 0x1c, 0xc1,
    0x30, 0x56, 0xa5, 0xda, 0x99, 0x46, 0xa6, 0x6d,
    0x0e, 0x6f, 0x5e, 0x51, 0x34, 0x49, 0x73, 0xc3,
    0x67, 0x49, 0x7e, 0x21, 0x2a, 0x20, 0xa7, 0x2b,
    0x92, 0x73, 0x1d, 0xa5, 0x25, 0x2a, 0xd0, 0x3a,
    0x89, 0x75, 0xb2, 0xbb, 0x19, 0x37, 0x78, 0x48,
    0xd2, 0xf2, 0x2a, 0x6d, 0x9e, 0xc6, 0x26, 0xca,
    0x46, 0x8c, 0xf1, 0x42, 0x2a, 0x31, 0xb2, 0xfc,
    0xe7, 0x55, 0x51, 0xff, 0x07, 0x13, 0x5b, 0x36,
    0x59, 0x2b, 0x43, 0x30, 0x4b, 0x05, 0x5c, 0xd2,
    0x45, 0xa0, 0xa0, 0x7c, 0x17, 0x5b, 0x07, 0xbb,
    0x5d, 0x83, 0x80, 0x92, 0x6d, 0x87, 0x1a, 0x43,
    0xac, 0xc7, 0x6b, 0x8d, 0x11, 0x60, 0x27, 0xd2,
    0xdf, 0xdb, 0x71, 0x02, 0x55, 0x6e, 0xb5, 0xca,
    0x4d, 0xda, 0x59, 0x0d, 0xb8, 0x8c, 0xcd, 0xd3,
    0x0e, 0x55, 0xa0, 0xa4, 0x8d, 0xa0, 0x14, 0x10,
    0x48, 0x42, 0x35, 0x56, 0x08, 0xf7, 0x29, 0x5f,
    0xa2, 0xea, 0xa4, 0x5e, 0x8e, 0x99, 0x56, 0xaa,
    0x5a, 0x8c, 0x23, 0x8f, 0x35, 0x22, 0x8a, 0xff,
    0xed, 0x02, 0x03, 0x01, 0x00, 0x01
};

static const uint8_t test_sha1_dgst[] = {
    0x3c, 0x05, 0x19, 0x34, 0x29, 0x19, 0xc7, 0xe0,
    0x87, 0xb6, 0x24, 0xf9, 0x58, 0xac, 0xa4, 0xd4,
    0xb2, 0xd9, 0x03, 0x9e,
};

static const uint8_t exp_signature_rsa2048_pkcs1[] = {
    0x4e, 0x82, 0x56, 0x4c, 0x84, 0x66, 0xca, 0x1e,
    0xc6, 0x92, 0x46, 0x20, 0x02, 0x6b, 0x64, 0x46,
    0x15, 0x6b, 0x24, 0xf2, 0xbb, 0xfa, 0x44, 0x3c,
    0xaf, 0x42, 0xc8, 0x41, 0xfd, 0xce, 0xed, 0x95,
    0x34, 0xaf, 0x25, 0x09, 0xd1, 0x06, 0x94, 0xaa,
    0x52, 0xd4, 0x29, 0xc8, 0x52, 0x34, 0x67, 0x59,
    0x4f, 0x5a, 0xfd, 0x23, 0x30, 0x5e, 0xc7, 0x1e,
    0xa6, 0xe0, 0x1b, 0x23, 0xca, 0x82, 0x47, 0x9a,
    0x2e, 0x2c, 0x66, 0x45, 0x5a, 0x12, 0xa9, 0x15,
    0xbf, 0xd6, 0xd6, 0xfa, 0x8d, 0x60, 0x99, 0x89,
    0x91, 0x39, 0x06, 0xb7, 0xd3, 0x9a, 0xef, 0x15,
    0x7b, 0x95, 0x87, 0x77, 0x2c, 0x41, 0xd4, 0x71,
    0xd5, 0xdf, 0x22, 0x7b, 0x01, 0xe2, 0xc1, 0xfb,
    0xb9, 0x4e, 0x0c, 0x9b, 0xd5, 0x04, 0xed, 0x2b,
    0x7e, 0x73, 0x53, 0xaa, 0x33, 0x89, 0x9d, 0x95,
    0x28, 0x8f, 0x8b, 0x80, 0x34, 0x7a, 0xea, 0xe3,
    0x66, 0x8a, 0xa8, 0xad, 0xed, 0x91, 0x43, 0xdd,
    0x77, 0xe5, 0xd7, 0x16, 0xda, 0xa8, 0x00, 0x29,
    0x3f, 0x9f, 0xe0, 0x1d, 0x42, 0x9d, 0x35, 0x5d,
    0x0f, 0xf3, 0x90, 0x27, 0x3a, 0x8c, 0x46, 0x13,
    0x53, 0x3e, 0x3b, 0x38, 0x77, 0xf8, 0x57, 0x61,
    0xbc, 0xc4, 0x54, 0x68, 0x48, 0xae, 0x58, 0x03,
    0x33, 0x94, 0x3f, 0x18, 0x1e, 0xb3, 0x3f, 0x79,
    0xa7, 0x26, 0x92, 0x5d, 0x32, 0x2a, 0xdb, 0xe6,
    0x3a, 0xe8, 0xd7, 0xaa, 0x91, 0xfe, 0x9f, 0x06,
    0x26, 0x68, 0x8c, 0x27, 0x31, 0xb0, 0x04, 0x9e,
    0x94, 0x79, 0x63, 0xa1, 0xc7, 0xe8, 0x5b, 0x8c,
    0xd3, 0xf1, 0x88, 0x58, 0x31, 0x2f, 0x4e, 0x11,
    0x00, 0xfe, 0x29, 0xad, 0x2c, 0xa9, 0x8e, 0x63,
    0xd8, 0x7d, 0xc5, 0xa1, 0x71, 0xfa, 0x08, 0x29,
    0xea, 0xd6, 0x6c, 0x53, 0x00, 0x52, 0xa0, 0xed,
    0x6b, 0x7c, 0x67, 0x50, 0x71, 0x2d, 0x96, 0x7a,
};

static const uint8_t exp_signature_rsa1024_pkcs1[] = {
    0x6b, 0x5b, 0xbb, 0x3b, 0x1f, 0x08, 0xd8, 0xc0,
    0x4a, 0xf1, 0x5a, 0x12, 0xc2, 0x39, 0x14, 0x65,
    0x4f, 0xda, 0x79, 0x67, 0xf2, 0x89, 0x25, 0xad,
    0x9e, 0x7e, 0xba, 0xa8, 0x34, 0x15, 0x03, 0xdd,
    0x80, 0x6b, 0x01, 0xd7, 0x4a, 0xf3, 0xd6, 0xef,
    0x1e, 0x48, 0xf3, 0xbc, 0x75, 0x1a, 0xc4, 0x2c,
    0x90, 0x15, 0x9f, 0x21, 0x24, 0x98, 0x21, 0xef,
    0x6d, 0x3b, 0xf3, 0x82, 0x8f, 0x8d, 0xd8, 0x48,
    0x37, 0x16, 0x19, 0x8e, 0x3c, 0x64, 0xa0, 0x9e,
    0xf7, 0x0c, 0xd9, 0x5c, 0xc6, 0x13, 0xc4, 0x5f,
    0xf8, 0xf3, 0x59, 0x5b, 0xd0, 0x33, 0x95, 0x98,
    0xde, 0x67, 0x25, 0x58, 0x46, 0xba, 0xee, 0x0f,
    0x47, 0x7a, 0x7f, 0xd0, 0xe4, 0x77, 0x09, 0x17,
    0xe9, 0x81, 0x6e, 0x2d, 0x33, 0x9b, 0x13, 0x0b,
    0xc9, 0xb2, 0x0c, 0x2c, 0xb5, 0xdf, 0x52, 0x8f,
    0xab, 0x0d, 0xc6, 0x59, 0x1d, 0xc7, 0x33, 0x7b,
};

static const uint8_t test_plaintext[] = {
    0x00, 0x44, 0xbc, 0x6f, 0x77, 0xfb, 0xe2, 0xa4,
    0x98, 0x9e, 0xf5, 0x33, 0xa0, 0xbd, 0x81, 0xb9,
    0xf1, 0x44, 0x7f, 0x79, 0x89, 0x23, 0xe5, 0x46,
    0x66, 0x9f, 0x98, 0x95, 0x6f, 0x56, 0x78, 0xf6,
    0xf5, 0xac, 0x9c, 0xda, 0xc2, 0x79, 0x59, 0xf0,
    0x1b, 0x03, 0xfa, 0x46, 0x1c, 0x1f, 0x18, 0x07,
    0xce, 0xad, 0xed, 0x3d, 0x11, 0xf9, 0x1b, 0x26,
    0x4a, 0x97, 0x28, 0x71, 0x5f, 0x2c, 0x5e, 0x58,
    0xf0, 0xd6, 0xbf, 0xa4, 0x12, 0xd0, 0x1d, 0x07,
    0xcb, 0x73, 0x66, 0xb6, 0xa4, 0x09, 0xaf, 0x5d,
    0xe9, 0x14, 0x14, 0xaf, 0x69, 0xd6, 0xee, 0x0a,
    0xfc, 0xca, 0xac, 0x94, 0x47, 0xd5, 0x9d, 0x5b,
    0x2b, 0xfb, 0xce, 0x9d, 0x04, 0xc1, 0xaf, 0xa5,
    0xa1, 0x8d, 0xa9, 0x48, 0xa8, 0x65, 0xe6, 0x9f,
    0x74, 0x78, 0x16, 0x32, 0x93, 0xb5, 0x21, 0xb9,
    0x9f, 0x3f, 0xc1, 0xe5, 0xa2, 0x50, 0x8b, 0x12,
    0xfb, 0x3e, 0xb0, 0x8a, 0x00, 0xc7, 0x20, 0x56,
    0xb3, 0xb1, 0x29, 0x95, 0x89, 0xd6, 0x50, 0xf5,
    0x37, 0x38, 0x8e, 0x12, 0xf1, 0xba, 0x82, 0x37,
    0x34, 0x68, 0x4b, 0xe8, 0xe3, 0x11, 0x1c, 0x46,
    0xf9, 0x63, 0x3a, 0xd6, 0xf3, 0x3f, 0x55, 0xa6,
    0xbd, 0x89, 0xf1, 0x2d, 0x38, 0x91, 0x7c, 0xc2,
    0x4d, 0xf1, 0x69, 0x82, 0x6d, 0x71, 0x77, 0xf4,
    0xfc, 0x43, 0x20, 0x6f, 0x43, 0xb9, 0x43, 0xd1,
    0x65, 0xbd, 0xca, 0xb1, 0x43, 0x87, 0xf8, 0xc8,
    0x76, 0x21, 0xa9, 0xeb, 0x3e, 0x9a, 0xef, 0xc9,
    0x0e, 0x79, 0xbc, 0xf0, 0xf8, 0xc8, 0xe2, 0xbc,
    0x33, 0x35, 0x3e, 0xfc, 0xf9, 0x44, 0x69, 0x06,
    0x7c, 0x7f, 0x5d, 0xa2, 0x9e, 0xab, 0xc2, 0x82,
    0xa0, 0xfb, 0xc5, 0x79, 0x57, 0x8c, 0xf1, 0x1c,
    0x51, 0x64, 0x4c, 0x56, 0x08, 0x80, 0x32, 0xf4,
    0x97, 0x8f, 0x6f, 0xb2, 0x16, 0xa6, 0x9d, 0x71,
};

static const uint8_t exp_ciphertext_rsa1024_raw[] = {
    0x01, 0xa0, 0xc2, 0x94, 0x9f, 0xd6, 0xbe, 0x8d,
    0xe9, 0x24, 0xaa, 0x9c, 0x67, 0xd7, 0xe3, 0x04,
    0x34, 0xbf, 0xd3, 0x27, 0xa1, 0x43, 0xeb, 0x60,
    0x6b, 0x5b, 0x64, 0x15, 0x55, 0x16, 0x98, 0x35,
    0xc2, 0x59, 0xa7, 0xf7, 0x24, 0xf7, 0x05, 0xb9,
    0xe8, 0x56, 0x6f, 0xf2, 0x7d, 0x8b, 0x3c, 0xcb,
    0xa6, 0xc2, 0xac, 0x0c, 0x37, 0x8c, 0x70, 0x70,
    0x55, 0x05, 0x07, 0x0d, 0x63, 0x6b, 0x7d, 0x5f,
    0xae, 0x03, 0x1e, 0x55, 0x05, 0xbb, 0xa8, 0xe7,
    0xff, 0xa0, 0x8c, 0x5b, 0x6b, 0x01, 0x48, 0x2e,
    0x4f, 0x7f, 0xe2, 0x74, 0xc6, 0x32, 0xa7, 0x2d,
    0xdb, 0x91, 0x9b, 0x67, 0x4d, 0x71, 0xf9, 0x8c,
    0x42, 0x43, 0x75, 0x4e, 0xd0, 0x0e, 0x7c, 0xa0,
    0x97, 0x1a, 0x5f, 0x8e, 0x6f, 0xe4, 0xfa, 0x16,
    0x1d, 0x59, 0x0e, 0x0b, 0x11, 0x12, 0xa3, 0x0c,
    0xa6, 0x55, 0xe6, 0xdb, 0xa7, 0x71, 0xa6, 0xff,
};

static const uint8_t exp_ciphertext_rsa1024_pkcs1[] = {
    0x93, 0x78, 0x6a, 0x76, 0xb8, 0x94, 0xea, 0xe4,
    0x32, 0x79, 0x01, 0x8b, 0xc1, 0xcb, 0x2e, 0x2d,
    0xfe, 0xdc, 0x9b, 0xe3, 0xe9, 0x23, 0xe4, 0x0a,
    0xb0, 0x6b, 0x9f, 0x6b, 0x62, 0xf5, 0x3d, 0xf0,
    0x78, 0x84, 0x77, 0x21, 0xad, 0x0b, 0x30, 0x30,
    0x94, 0xe2, 0x18, 0xc4, 0x9b, 0x12, 0x06, 0xc8,
    0xaa, 0xf7, 0x30, 0xe4, 0xc8, 0x64, 0xe7, 0x51,
    0xf1, 0x6a, 0xe1, 0xa2, 0x58, 0x7a, 0x02, 0x9c,
    0x8e, 0xf0, 0x2d, 0x25, 0x6b, 0xb7, 0x25, 0x5e,
    0x05, 0xaf, 0x38, 0xb2, 0x69, 0x5e, 0x6c, 0x75,
    0x6e, 0x27, 0xba, 0x5d, 0x7d, 0x35, 0x72, 0xb7,
    0x25, 0xd4, 0xaa, 0xb2, 0x4b, 0x9e, 0x6b, 0x82,
    0xb2, 0x32, 0xe2, 0x13, 0x1d, 0x00, 0x21, 0x08,
    0xae, 0x14, 0xbb, 0xc0, 0x40, 0xb7, 0x0d, 0xd5,
    0x0e, 0x4d, 0x6d, 0x9a, 0x70, 0x86, 0xe9, 0xfc,
    0x67, 0x2b, 0xa4, 0x11, 0x45, 0xb6, 0xc4, 0x2f,
};

static const uint8_t exp_ciphertext_rsa2048_raw[] = {
    0x09, 0x7b, 0x9e, 0x7c, 0x10, 0x1f, 0x73, 0xb4,
    0x5f, 0xdb, 0x4f, 0x05, 0xe7, 0xfc, 0x9e, 0x35,
    0x48, 0xd8, 0xc8, 0xf5, 0xac, 0x6d, 0xb4, 0xb0,
    0xd4, 0xf7, 0x69, 0x0f, 0x30, 0x78, 0xbb, 0x55,
    0x67, 0x66, 0x66, 0x05, 0xf4, 0x77, 0xe2, 0x30,
    0xa5, 0x94, 0x10, 0xa3, 0xcb, 0xee, 0x13, 0x9f,
    0x47, 0x1b, 0x2e, 0xf9, 0xfd, 0x94, 0x09, 0xbd,
    0x26, 0x6e, 0x84, 0xc7, 0x5c, 0x42, 0x20, 0x76,
    0x72, 0x83, 0x75, 0x68, 0xa4, 0x18, 0x2d, 0x76,
    0x62, 0xc3, 0xab, 0xc0, 0xc9, 0x36, 0x59, 0xe0,
    0xa9, 0x70, 0x1f, 0xff, 0x97, 0x07, 0x0d, 0x88,
    0xc2, 0xd8, 0x51, 0x35, 0xf7, 0xb0, 0x50, 0xe4,
    0x9f, 0x3d, 0xd4, 0x71, 0x8b, 0x40, 0x89, 0x71,
    0x6c, 0xd8, 0xc2, 0x63, 0xb6, 0x3a, 0xce, 0xb1,
    0x32, 0xf1, 0xc6, 0x11, 0x31, 0x25, 0x48, 0xcf,
    0xeb, 0xbc, 0xd3, 0x9b, 0xc5, 0xbd, 0xd2, 0x57,
    0x73, 0x9b, 0x20, 0xb8, 0xdf, 0xbe, 0xb8, 0x40,
    0xb6, 0xac, 0x24, 0xdb, 0x94, 0x6a, 0x93, 0x43,
    0x4a, 0xa8, 0xa3, 0xcf, 0xd5, 0x61, 0x1b, 0x46,
    0x1d, 0x6f, 0x57, 0xec, 0xa6, 0xd0, 0x44, 0x05,
    0x48, 0xb8, 0x90, 0x80, 0x23, 0x8e, 0x5f, 0xb0,
    0x4b, 0x6f, 0xe3, 0xf9, 0xb0, 0x04, 0x60, 0xae,
    0x80, 0xcf, 0xa5, 0x5c, 0x11, 0xe4, 0xce, 0x57,
    0x5b, 0xbb, 0xde, 0x92, 0xfc, 0xe7, 0x3f, 0xe0,
    0xfc, 0x06, 0xc8, 0xf3, 0x8c, 0xac, 0x86, 0x09,
    0x31, 0xe5, 0x7e, 0xfb, 0x5d, 0xa7, 0x57, 0xf8,
    0x1d, 0x23, 0x9d, 0xa3, 0xeb, 0x53, 0x28, 0xde,
    0xbf, 0x53, 0xef, 0x35, 0x3c, 0x7e, 0x3c, 0x1b,
    0x76, 0x9d, 0x09, 0x25, 0x43, 0xd4, 0x8b, 0xca,
    0xda, 0x45, 0x5b, 0xdc, 0x9f, 0x57, 0x5a, 0x30,
    0x2e, 0xe9, 0x73, 0x68, 0x28, 0xfa, 0x40, 0xb0,
    0x7c, 0x31, 0xd7, 0x8b, 0x4e, 0x99, 0x94, 0xf1,
};

static const uint8_t exp_ciphertext_rsa2048_pkcs1[] = {
    0xa5, 0x19, 0x19, 0x34, 0xad, 0xf6, 0xd2, 0xbe,
    0xed, 0x8f, 0xe5, 0xfe, 0xa2, 0xa5, 0x20, 0x08,
    0x15, 0x53, 0x7c, 0x68, 0x28, 0xae, 0x07, 0xb2,
    0x4c, 0x5d, 0xee, 0xc1, 0xc6, 0xdc, 0xd6, 0x8b,
    0xc6, 0xba, 0x46, 0xe1, 0x16, 0xa9, 0x04, 0x72,
    0xdf, 0x8f, 0x1e, 0x97, 0x2a, 0x55, 0xe7, 0xac,
    0x08, 0x0d, 0x61, 0xe8, 0x64, 0x8b, 0x6f, 0x96,
    0x0e, 0xbb, 0x8a, 0x30, 0xb3, 0x73, 0x28, 0x61,
    0x16, 0x89, 0x90, 0x88, 0x8e, 0xda, 0x22, 0xe6,
    0x42, 0x16, 0xc7, 0xe8, 0x30, 0x0d, 0x7f, 0x44,
    0x1e, 0xef, 0xe6, 0xdb, 0x78, 0x54, 0x89, 0xa5,
    0x60, 0x67, 0xb3, 0x35, 0x2d, 0x79, 0x49, 0xcf,
    0xe6, 0x8f, 0xf3, 0x64, 0x52, 0x1c, 0x6c, 0x43,
    0x7e, 0xb0, 0xde, 0x55, 0xdf, 0xbe, 0xb7, 0xb1,
    0xdb, 0x02, 0xee, 0x76, 0x96, 0xcc, 0x0b, 0x97,
    0x8c, 0x23, 0xaa, 0x7d, 0x4c, 0x47, 0x28, 0x41,
    0x7a, 0x20, 0x39, 0x1f, 0x64, 0x0b, 0xf1, 0x74,
    0xf1, 0x29, 0xda, 0xe9, 0x3a, 0x36, 0xa6, 0x88,
    0xb8, 0xc0, 0x21, 0xb8, 0x9b, 0x5d, 0x90, 0x85,
    0xa3, 0x30, 0x61, 0x17, 0x8c, 0x74, 0x63, 0xd5,
    0x0f, 0x95, 0xdc, 0xc8, 0x4f, 0xa7, 0x24, 0x55,
    0x40, 0xe2, 0x84, 0x57, 0x65, 0x06, 0x11, 0x30,
    0x2b, 0x9e, 0x32, 0x95, 0x39, 0xf2, 0x1a, 0x3f,
    0xab, 0xcd, 0x7b, 0x7f, 0x9c, 0xf0, 0x00, 0x50,
    0x7c, 0xf4, 0xbe, 0xcb, 0x80, 0xea, 0x66, 0xba,
    0x0e, 0x7b, 0x46, 0x0b, 0x25, 0xe0, 0xc1, 0x03,
    0x29, 0x11, 0x2d, 0x69, 0x4f, 0x21, 0xa2, 0x58,
    0x37, 0x4b, 0x84, 0x15, 0xb3, 0x65, 0x3a, 0xac,
    0xd4, 0xd0, 0xf6, 0xdf, 0x4b, 0x82, 0xca, 0x9e,
    0xbb, 0xbe, 0x3c, 0x4d, 0xd5, 0xbf, 0x00, 0xd6,
    0x12, 0x48, 0x72, 0x0b, 0xc7, 0xf8, 0xe1, 0xcd,
    0xd0, 0x28, 0x03, 0x19, 0xa6, 0x06, 0x13, 0x45,
};

static const uint8_t rsa_private_key_lack_element[] = {
    /* RSAPrivateKey, offset: 0, length: 176 */
    0x30, 0x81, 0xb0,
    /* version, offset: 4, length: 1 */
    0x02, 0x01, 0x00,
    /* n, offset: 7, length: 65 */
    0x02, 0x41,
    0x00, 0xb9, 0xe1, 0x22, 0xdb, 0x56, 0x2f, 0xb6,
    0xf7, 0xf0, 0x0a, 0x87, 0x43, 0x07, 0x12, 0xdb,
    0x6d, 0xb6, 0x2b, 0x41, 0x8d, 0x2c, 0x3c, 0xa5,
    0xdd, 0x78, 0x9a, 0x8f, 0xab, 0x8e, 0xf2, 0x4a,
    0xc8, 0x34, 0x0c, 0x12, 0x4f, 0x11, 0x90, 0xc6,
    0xc2, 0xa5, 0xd0, 0xcd, 0xfb, 0xfc, 0x2c, 0x95,
    0x56, 0x82, 0xdf, 0x39, 0xf3, 0x3b, 0x1d, 0x62,
    0x26, 0x97, 0xb7, 0x93, 0x25, 0xc7, 0xec, 0x7e,
    0xf7,
    /* e, offset: 74, length: 3 */
    0x02, 0x03, 0x01, 0x00, 0x01,
    /* d, offset: 79, length: 64 */
    0x02, 0x40,
    0x1e, 0x80, 0xfe, 0xda, 0x65, 0xdb, 0x70, 0xb8,
    0x61, 0x91, 0x28, 0xbf, 0x6c, 0x32, 0xc1, 0x05,
    0xd1, 0x26, 0x6a, 0x1c, 0x83, 0xcc, 0xf4, 0x1f,
    0x53, 0x42, 0x72, 0x1f, 0x62, 0x57, 0x0a, 0xc4,
    0x66, 0x76, 0x30, 0x87, 0xb9, 0xb1, 0xb9, 0x6a,
    0x63, 0xfd, 0x8f, 0x3e, 0xfc, 0x35, 0x3f, 0xd6,
    0x2e, 0x6c, 0xc8, 0x70, 0x8a, 0x17, 0xc1, 0x28,
    0x6a, 0xfe, 0x51, 0x56, 0xb3, 0x92, 0x6f, 0x09,
    /* p, offset: 145, length: 33 */
    0x02, 0x21,
    0x00, 0xe3, 0x2e, 0x2d, 0x8d, 0xba, 0x1c, 0x34,
    0x4c, 0x49, 0x9f, 0xc1, 0xa6, 0xdd, 0xd7, 0x13,
    0x8d, 0x05, 0x48, 0xdd, 0xff, 0x5c, 0x30, 0xbc,
    0x6b, 0xc4, 0x18, 0x9d, 0xfc, 0xa2, 0xd0, 0x9b,
    0x4d,
    /* q, offset: 180, length: 33 */
    0x02, 0x21,
    0x00, 0xd1, 0x75, 0xaf, 0x4b, 0xc6, 0x1a, 0xb0,
    0x98, 0x14, 0x42, 0xae, 0x33, 0xf3, 0x44, 0xde,
    0x21, 0xcb, 0x04, 0xda, 0xfb, 0x1e, 0x35, 0x92,
    0xcd, 0x69, 0xc0, 0x83, 0x06, 0x83, 0x8e, 0x39,
    0x53,
    /* lack element: dp, dq, u */
};

static const uint8_t rsa_public_key_lack_element[] = {
    /* RSAPublicKey, offset: 0, length: 67 */
    0x30, 0x81, 0x43,
    /* n, offset: 7, length: 65 */
    0x02, 0x41,
    0x00, 0xb9, 0xe1, 0x22, 0xdb, 0x56, 0x2f, 0xb6,
    0xf7, 0xf0, 0x0a, 0x87, 0x43, 0x07, 0x12, 0xdb,
    0x6d, 0xb6, 0x2b, 0x41, 0x8d, 0x2c, 0x3c, 0xa5,
    0xdd, 0x78, 0x9a, 0x8f, 0xab, 0x8e, 0xf2, 0x4a,
    0xc8, 0x34, 0x0c, 0x12, 0x4f, 0x11, 0x90, 0xc6,
    0xc2, 0xa5, 0xd0, 0xcd, 0xfb, 0xfc, 0x2c, 0x95,
    0x56, 0x82, 0xdf, 0x39, 0xf3, 0x3b, 0x1d, 0x62,
    0x26, 0x97, 0xb7, 0x93, 0x25, 0xc7, 0xec, 0x7e,
    0xf7,
    /* lack element: e */
};

static const uint8_t rsa_public_key_empty_element[] = {
    /* RSAPublicKey, offset: 0, length: 69 */
    0x30, 0x81, 0x45,
    /* n, offset: 7, length: 65 */
    0x02, 0x41,
    0x00, 0xb9, 0xe1, 0x22, 0xdb, 0x56, 0x2f, 0xb6,
    0xf7, 0xf0, 0x0a, 0x87, 0x43, 0x07, 0x12, 0xdb,
    0x6d, 0xb6, 0x2b, 0x41, 0x8d, 0x2c, 0x3c, 0xa5,
    0xdd, 0x78, 0x9a, 0x8f, 0xab, 0x8e, 0xf2, 0x4a,
    0xc8, 0x34, 0x0c, 0x12, 0x4f, 0x11, 0x90, 0xc6,
    0xc2, 0xa5, 0xd0, 0xcd, 0xfb, 0xfc, 0x2c, 0x95,
    0x56, 0x82, 0xdf, 0x39, 0xf3, 0x3b, 0x1d, 0x62,
    0x26, 0x97, 0xb7, 0x93, 0x25, 0xc7, 0xec, 0x7e,
    0xf7,
    /* e: empty element */
    0x02, 0x00,
};

static const uint8_t rsa_private_key_empty_element[] = {
    /* RSAPrivateKey, offset: 0, length: 19 */
    0x30, 0x81, 0x13,
    /* version, offset: 4, length: 1 */
    0x02, 0x01, 0x00,
    /* n: empty element */
    0x02, 0x00,
    /* e: empty element */
    0x02, 0x00,
    /* d: empty element */
    0x02, 0x00,
    /* p: empty element */
    0x02, 0x00,
    /* q: empty element */
    0x02, 0x00,
    /* dp: empty element */
    0x02, 0x00,
    /* dq: empty element */
    0x02, 0x00,
    /* u: empty element */
    0x02, 0x00,
};

static const uint8_t rsa_public_key_invalid_length_val[] = {
    /* RSAPublicKey, INVALID length: 313 */
    0x30, 0x82, 0x01, 0x39,
    /* n, offset: 7, length: 65 */
    0x02, 0x41,
    0x00, 0xb9, 0xe1, 0x22, 0xdb, 0x56, 0x2f, 0xb6,
    0xf7, 0xf0, 0x0a, 0x87, 0x43, 0x07, 0x12, 0xdb,
    0x6d, 0xb6, 0x2b, 0x41, 0x8d, 0x2c, 0x3c, 0xa5,
    0xdd, 0x78, 0x9a, 0x8f, 0xab, 0x8e, 0xf2, 0x4a,
    0xc8, 0x34, 0x0c, 0x12, 0x4f, 0x11, 0x90, 0xc6,
    0xc2, 0xa5, 0xd0, 0xcd, 0xfb, 0xfc, 0x2c, 0x95,
    0x56, 0x82, 0xdf, 0x39, 0xf3, 0x3b, 0x1d, 0x62,
    0x26, 0x97, 0xb7, 0x93, 0x25, 0xc7, 0xec, 0x7e,
    0xf7,
    /* e, */
    0x02, 0x03, 0x01, 0x00, 0x01,  /* INTEGER, offset: 74, length: 3 */
};

static const uint8_t rsa_public_key_extra_elem[] = {
    /* RSAPublicKey, length: 80 */
    0x30, 0x81, 0x50,
    /* n, offset: 7, length: 65 */
    0x02, 0x41,
    0x00, 0xb9, 0xe1, 0x22, 0xdb, 0x56, 0x2f, 0xb6,
    0xf7, 0xf0, 0x0a, 0x87, 0x43, 0x07, 0x12, 0xdb,
    0x6d, 0xb6, 0x2b, 0x41, 0x8d, 0x2c, 0x3c, 0xa5,
    0xdd, 0x78, 0x9a, 0x8f, 0xab, 0x8e, 0xf2, 0x4a,
    0xc8, 0x34, 0x0c, 0x12, 0x4f, 0x11, 0x90, 0xc6,
    0xc2, 0xa5, 0xd0, 0xcd, 0xfb, 0xfc, 0x2c, 0x95,
    0x56, 0x82, 0xdf, 0x39, 0xf3, 0x3b, 0x1d, 0x62,
    0x26, 0x97, 0xb7, 0x93, 0x25, 0xc7, 0xec, 0x7e,
    0xf7,
    /* e, offset: 74, length: 3 */
    0x02, 0x03, 0x01, 0x00, 0x01,
    /* Additional integer field, length 3 */
    0x02, 0x06, 0xe1, 0x22, 0xdb, 0xe1, 0x22, 0xdb,
};

typedef struct QCryptoRSAKeyTestData QCryptoRSAKeyTestData;
struct QCryptoRSAKeyTestData {
    const char *path;
    QCryptoAkCipherKeyType key_type;
    QCryptoAkCipherOptions opt;
    const uint8_t *key;
    size_t keylen;
    bool is_valid_key;
    size_t exp_key_len;
};

typedef struct QCryptoAkCipherTestData QCryptoAkCipherTestData;
struct QCryptoAkCipherTestData {
    const char *path;
    QCryptoAkCipherOptions opt;

    const uint8_t *priv_key;
    size_t priv_key_len;
    const uint8_t *pub_key;
    size_t pub_key_len;

    const uint8_t *plaintext;
    size_t plen;
    const uint8_t *ciphertext;
    size_t clen;
    const uint8_t *dgst;
    size_t dlen;
    const uint8_t *signature;
    size_t slen;
};

static QCryptoRSAKeyTestData rsakey_test_data[] = {
    {
        .path = "/crypto/akcipher/rsakey-1024-public",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PUBLIC,
        .key = rsa1024_public_key,
        .keylen = sizeof(rsa1024_public_key),
        .is_valid_key = true,
        .exp_key_len = 128,
    },
    {
        .path = "/crypto/akcipher/rsakey-1024-private",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PRIVATE,
        .key = rsa1024_private_key,
        .keylen = sizeof(rsa1024_private_key),
        .is_valid_key = true,
        .exp_key_len = 128,
    },
    {
        .path = "/crypto/akcipher/rsakey-2048-public",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PUBLIC,
        .key = rsa2048_public_key,
        .keylen = sizeof(rsa2048_public_key),
        .is_valid_key = true,
        .exp_key_len = 256,
    },
    {
        .path = "/crypto/akcipher/rsakey-2048-private",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PRIVATE,
        .key = rsa2048_private_key,
        .keylen = sizeof(rsa2048_private_key),
        .is_valid_key = true,
        .exp_key_len = 256,
    },
    {
        .path = "/crypto/akcipher/rsakey-public-lack-elem",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PUBLIC,
        .key = rsa_public_key_lack_element,
        .keylen = sizeof(rsa_public_key_lack_element),
        .is_valid_key = false,
    },
    {
        .path = "/crypto/akcipher/rsakey-private-lack-elem",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PRIVATE,
        .key = rsa_private_key_lack_element,
        .keylen = sizeof(rsa_private_key_lack_element),
        .is_valid_key = false,
    },
    {
        .path = "/crypto/akcipher/rsakey-public-empty-elem",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PUBLIC,
        .key = rsa_public_key_empty_element,
        .keylen = sizeof(rsa_public_key_empty_element),
        .is_valid_key = false,
    },
    {
        .path = "/crypto/akcipher/rsakey-private-empty-elem",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PRIVATE,
        .key = rsa_private_key_empty_element,
        .keylen = sizeof(rsa_private_key_empty_element),
        .is_valid_key = false,
    },
    {
        .path = "/crypto/akcipher/rsakey-public-empty-key",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PUBLIC,
        .key = NULL,
        .keylen = 0,
        .is_valid_key = false,
    },
    {
        .path = "/crypto/akcipher/rsakey-private-empty-key",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PRIVATE,
        .key = NULL,
        .keylen = 0,
        .is_valid_key = false,
    },
    {
        .path = "/crypto/akcipher/rsakey-public-invalid-length-val",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PUBLIC,
        .key = rsa_public_key_invalid_length_val,
        .keylen = sizeof(rsa_public_key_invalid_length_val),
        .is_valid_key = false,
    },
    {
        .path = "/crypto/akcipher/rsakey-public-extra-elem",
        .key_type = QCRYPTO_AK_CIPHER_KEY_TYPE_PUBLIC,
        .key = rsa_public_key_extra_elem,
        .keylen = sizeof(rsa_public_key_extra_elem),
        .is_valid_key = false,
    },
};

static QCryptoAkCipherTestData akcipher_test_data[] = {
    /* rsa1024 with raw padding */
    {
        .path = "/crypto/akcipher/rsa1024-raw",
        .opt = {
            .alg = QCRYPTO_AK_CIPHER_ALGO_RSA,
            .u.rsa = {
                .padding_alg = QCRYPTO_RSA_PADDING_ALGO_RAW,
            },
        },
        .pub_key = rsa1024_public_key,
        .pub_key_len = sizeof(rsa1024_public_key),
        .priv_key = rsa1024_private_key,
        .priv_key_len = sizeof(rsa1024_private_key),

        .plaintext = test_plaintext,
        .plen = 128,
        .ciphertext = exp_ciphertext_rsa1024_raw,
        .clen = sizeof(exp_ciphertext_rsa1024_raw),
    },

    /* rsa1024 with pkcs1 padding */
    {
        .path = "/crypto/akcipher/rsa1024-pkcs1",
        .opt = {
            .alg = QCRYPTO_AK_CIPHER_ALGO_RSA,
            .u.rsa = {
                .padding_alg = QCRYPTO_RSA_PADDING_ALGO_PKCS1,
                .hash_alg = QCRYPTO_HASH_ALGO_SHA1,
            },
        },
        .pub_key = rsa1024_public_key,
        .pub_key_len = sizeof(rsa1024_public_key),
        .priv_key = rsa1024_private_key,
        .priv_key_len = sizeof(rsa1024_private_key),

        .plaintext = test_plaintext,
        .plen = 64,
        .ciphertext = exp_ciphertext_rsa1024_pkcs1,
        .clen = sizeof(exp_ciphertext_rsa1024_pkcs1),
        .dgst = test_sha1_dgst,
        .dlen = sizeof(test_sha1_dgst),
        .signature = exp_signature_rsa1024_pkcs1,
        .slen = sizeof(exp_signature_rsa1024_pkcs1),
    },

    /* rsa2048 with raw padding */
    {
        .path = "/crypto/akcipher/rsa2048-raw",
        .opt = {
            .alg = QCRYPTO_AK_CIPHER_ALGO_RSA,
            .u.rsa = {
                .padding_alg = QCRYPTO_RSA_PADDING_ALGO_RAW,
            },
        },
        .pub_key = rsa2048_public_key,
        .pub_key_len = sizeof(rsa2048_public_key),
        .priv_key = rsa2048_private_key,
        .priv_key_len = sizeof(rsa2048_private_key),

        .plaintext = test_plaintext,
        .plen = 256,
        .ciphertext = exp_ciphertext_rsa2048_raw,
        .clen = sizeof(exp_ciphertext_rsa2048_raw),
    },

    /* rsa2048 with pkcs1 padding */
    {
        .path = "/crypto/akcipher/rsa2048-pkcs1",
        .opt = {
            .alg = QCRYPTO_AK_CIPHER_ALGO_RSA,
            .u.rsa = {
                .padding_alg = QCRYPTO_RSA_PADDING_ALGO_PKCS1,
                .hash_alg = QCRYPTO_HASH_ALGO_SHA1,
            },
        },
        .pub_key = rsa2048_public_key,
        .pub_key_len = sizeof(rsa2048_public_key),
        .priv_key = rsa2048_private_key,
        .priv_key_len = sizeof(rsa2048_private_key),

        .plaintext = test_plaintext,
        .plen = 128,
        .ciphertext = exp_ciphertext_rsa2048_pkcs1,
        .clen = sizeof(exp_ciphertext_rsa2048_pkcs1),
        .dgst = test_sha1_dgst,
        .dlen = sizeof(test_sha1_dgst),
        .signature = exp_signature_rsa2048_pkcs1,
        .slen = sizeof(exp_signature_rsa2048_pkcs1),
    },

};

static void test_akcipher(const void *opaque)
{
    const QCryptoAkCipherTestData *data = opaque;
    g_autofree uint8_t *plaintext = NULL;
    g_autofree uint8_t *ciphertext = NULL;
    g_autofree uint8_t *signature = NULL;
    QCryptoAkCipher *pub_key, *priv_key;

    if (!qcrypto_akcipher_supports((QCryptoAkCipherOptions *)&data->opt)) {
        return;
    }
    pub_key = qcrypto_akcipher_new(&data->opt,
                                   QCRYPTO_AK_CIPHER_KEY_TYPE_PUBLIC,
                                   data->pub_key, data->pub_key_len,
                                   &error_abort);
    g_assert(pub_key != NULL);
    priv_key = qcrypto_akcipher_new(&data->opt,
                                    QCRYPTO_AK_CIPHER_KEY_TYPE_PRIVATE,
                                    data->priv_key, data->priv_key_len,
                                    &error_abort);
    g_assert(priv_key != NULL);

    if (data->plaintext != NULL) {

        ciphertext = g_new0(uint8_t, data->clen);
        g_assert(qcrypto_akcipher_encrypt(pub_key, data->plaintext, data->plen,
                                          ciphertext, data->clen,
                                          &error_abort) > 0);

        /**
         * In the asymmetric encryption algorithms, the ciphertext generated
         * each time may be different, here only compare the decrypted
         * plaintext
         */
        plaintext = g_new0(uint8_t, data->clen);
        g_assert(qcrypto_akcipher_decrypt(priv_key, ciphertext,
                                          data->clen, plaintext,
                                          data->plen,
                                          &error_abort) == data->plen);
        g_assert(!memcmp(plaintext, data->plaintext, data->plen));
    }

    if (data->signature != NULL) {
        signature = g_new(uint8_t, data->slen);
        g_assert(qcrypto_akcipher_sign(priv_key, data->dgst, data->dlen,
                                       signature, data->slen,
                                       &error_abort) > 0);
        /**
         * The signature generated each time may be different, here only check
         * the verification.
         */
        g_assert(qcrypto_akcipher_verify(pub_key, data->signature, data->slen,
                                         data->dgst, data->dlen,
                                         &error_abort) == 0);
        g_assert(qcrypto_akcipher_verify(pub_key, signature, data->slen,
                                         data->dgst, data->dlen,
                                         &error_abort) == 0);
        ++signature[0];
        /* Here error should be ignored */
        g_assert(qcrypto_akcipher_verify(pub_key, signature, data->slen,
                                         data->dgst, data->dlen, NULL) != 0);
    }

    qcrypto_akcipher_free(pub_key);
    qcrypto_akcipher_free(priv_key);
}

static void test_rsakey(const void *opaque)
{
    const QCryptoRSAKeyTestData *data = (const QCryptoRSAKeyTestData *)opaque;
    QCryptoAkCipherOptions opt = {
        .alg = QCRYPTO_AK_CIPHER_ALGO_RSA,
        .u.rsa = {
            .padding_alg = QCRYPTO_RSA_PADDING_ALGO_PKCS1,
            .hash_alg = QCRYPTO_HASH_ALGO_SHA1,
        }
    };
    g_autoptr(QCryptoAkCipher) key = qcrypto_akcipher_new(
        &opt, data->key_type, data->key, data->keylen, NULL);

    if (!qcrypto_akcipher_supports(&opt)) {
        return;
    }

    if (!data->is_valid_key) {
        g_assert(key == NULL);
        return;
    }

    g_assert(key != NULL);
    g_assert(qcrypto_akcipher_max_ciphertext_len(key) == data->exp_key_len);
    g_assert(qcrypto_akcipher_max_plaintext_len(key) == data->exp_key_len);
    g_assert(qcrypto_akcipher_max_signature_len(key) == data->exp_key_len);
    g_assert(qcrypto_akcipher_max_dgst_len(key) == data->exp_key_len);
}

int main(int argc, char **argv)
{
    size_t i;
    g_test_init(&argc, &argv, NULL);
    g_assert(qcrypto_init(NULL) == 0);

    for (i = 0; i < G_N_ELEMENTS(akcipher_test_data); i++) {
        g_test_add_data_func(akcipher_test_data[i].path,
                             &akcipher_test_data[i],
                             test_akcipher);
    }
    for (i = 0; i < G_N_ELEMENTS(rsakey_test_data); i++) {
        g_test_add_data_func(rsakey_test_data[i].path,
                             &rsakey_test_data[i],
                             test_rsakey);
    }

    return g_test_run();
}
