%define OT_EnclosingSimplexAlgorithm_doc
"Point location algorithm.

Base class to define an algorithm to find simplices containing
a list of points.

Available constructors:
    EnclosingSimplexAlgorithm(*vertices, simplices*)

Parameters
----------
vertices : :class:`~openturns.Sample`
    Vertices.

simplices : :class:`~openturns.IndicesCollection`
    Simplices.

Notes
-----

Two algorithms can be selected in any dimension:

* :class:`~openturns.NaiveEnclosingSimplex` loops over all simplices
  and stops when the given point is inside.
* :class:`~openturns.BoundingVolumeHierarchy` builds a binary tree based
  on bounding box of simplices.

Two algorithms are specific to 1D meshes, and much more efficient:

* :class:`~openturns.RegularGridEnclosingSimplex` is the most efficient
  algorithm when mesh corresponds to a :class:`~openturns.RegularGrid`,
  :meth:`query` works in constant time.
* :class:`~openturns.EnclosingSimplexMonotonic1D` looks for enclosing
  simplex by dichotomy, it only works for sorted 1D meshes.

It is recommended to use derived classes in order to select the best algorithm
according to your data.  If you create a generic :class:`EnclosingSimplexAlgorithm`,
here is how the derived class is selected:

* If vertices and simplices correspond to a :class:`~openturns.RegularGrid`,
  :class:`~openturns.RegularGridEnclosingSimplex` algorithm is selected.
* If vertices and simplices correspond to a simple 1D mesh with sorted vertices,
  :class:`~openturns.EnclosingSimplexMonotonic1D` algorithm is selected.
* Otherwise, 'EnclosingSimplexAlgorithm-LargeDimension' :class:`~openturns.ResourceMap`
  entry is read.  If input dimension is lower than this value,
  :class:`~openturns.BoundingVolumeHierarchy` algorithm is chosen, otherwise
  it is :class:`~openturns.NaiveEnclosingSimplex`.

By default, the algorithm is :class:`~openturns.BoundingVolumeHierarchy`.

See also
--------
NaiveEnclosingSimplex, BoundingVolumeHierarchy, RegularGridEnclosingSimplex,
EnclosingSimplexMonotonic1D

Examples
--------
>>> import openturns as ot
>>> mesher = ot.IntervalMesher([5, 10])
>>> lowerbound = [0.0, 0.0]
>>> upperBound = [2.0, 4.0]
>>> interval = ot.Interval(lowerbound, upperBound)
>>> mesh = mesher.build(interval)
>>> locator = ot.EnclosingSimplexAlgorithm(mesh.getVertices(), mesh.getSimplices())
>>> simplex = locator.query([0.1, 0.2])"
%enddef

%feature("docstring") OT::EnclosingSimplexAlgorithmImplementation
OT_EnclosingSimplexAlgorithm_doc

// ---------------------------------------------------------------------

%define OT_EnclosingSimplexAlgorithm_setVerticesAndSimplices_doc
"Rebuild a new data structure for these vertices and simplices.

Parameters
----------
vertices : :class:`~openturns.Sample`
    Vertices.

simplices : :class:`~openturns.IndicesCollection`
    Simplices."
%enddef

%feature("docstring") OT::EnclosingSimplexAlgorithmImplementation::setVerticesAndSimplices
OT_EnclosingSimplexAlgorithm_setVerticesAndSimplices_doc

// ---------------------------------------------------------------------

%define OT_EnclosingSimplexAlgorithm_query_doc
"Get the index of the enclosing simplex of the given point.

Available usages:
    query(*point*)

    query(*sample*)

Parameters
----------
point : sequence of float
    Given point.
sample : 2-d sequence of float
    Given points.

Returns
-------
index : int
    If point is enclosed in a simplex, return its index; otherwise return an
    int which is at least greater than the number of simplices.
indices : :class:`~openturns.Indices`
    Index of enclosing simplex of each point of the sample.  If there is no
    enclosing simplex, value is an int which is at least greater than the
    number of simplices."
%enddef

%feature("docstring") OT::EnclosingSimplexAlgorithmImplementation::query
OT_EnclosingSimplexAlgorithm_query_doc

// ---------------------------------------------------------------------

%define OT_EnclosingSimplexAlgorithm_getVertices_doc
"Collection of vertices accessor.

Returns
-------
vertices : :class:`~openturns.Sample`
    Collection of points."
%enddef

%feature("docstring") OT::EnclosingSimplexAlgorithmImplementation::getVertices
OT_EnclosingSimplexAlgorithm_getVertices_doc

// ---------------------------------------------------------------------

%define OT_EnclosingSimplexAlgorithm_getSimplices_doc
"Collection of simplex accessor.

Returns
-------
simplices : :class:`~openturns.IndicesCollection`
    Collection of simplices."
%enddef

%feature("docstring") OT::EnclosingSimplexAlgorithmImplementation::getSimplices
OT_EnclosingSimplexAlgorithm_getSimplices_doc

// ---------------------------------------------------------------------

%define OT_EnclosingSimplexAlgorithm_setBarycentricCoordinatesEpsilon_doc
R"RAW(Accessor to the tolerance for membership test.

Parameters
----------
epsilon : float
    Tolerance for the membership. A point is in a simplex if its barycentric coordinates :math:`\xi_i` are
    all in :math:`[-\varepsilon,1+\varepsilon]` and :math:`\sum_{i=1}^d\xi_i\in[-\varepsilon,1+\varepsilon]`.)RAW"
%enddef

%feature("docstring") OT::EnclosingSimplexAlgorithmImplementation::setBarycentricCoordinatesEpsilon
OT_EnclosingSimplexAlgorithm_setBarycentricCoordinatesEpsilon_doc

// ---------------------------------------------------------------------

%define OT_EnclosingSimplexAlgorithm_getBarycentricCoordinatesEpsilon_doc
R"RAW(Accessor to the tolerance for membership test.

Returns
-------
epsilon : float
    Tolerance for the membership. A point is in a simplex if its barycentric coordinates :math:`\xi_i` are
    all in :math:`[-\varepsilon,1+\varepsilon]` and :math:`\sum_{i=1}^d\xi_i\in[-\varepsilon,1+\varepsilon]`.)RAW"
%enddef

%feature("docstring") OT::EnclosingSimplexAlgorithmImplementation::getBarycentricCoordinatesEpsilon
OT_EnclosingSimplexAlgorithm_getBarycentricCoordinatesEpsilon_doc
