package x25519

import (
	"bytes"
	"crypto"
	"crypto/ed25519"
	"crypto/rand"
	"io"
	"reflect"
	"testing"

	"golang.org/x/crypto/curve25519"
)

func mustTeeReader(t *testing.T) *bytes.Buffer {
	t.Helper()
	reader := rand.Reader
	t.Cleanup(func() {
		rand.Reader = reader
	})
	buf := new(bytes.Buffer)
	rand.Reader = io.TeeReader(reader, buf)
	return buf
}

func BenchmarkGenerateKey(b *testing.B) {
	for i := 0; i < b.N; i++ {
		if _, _, err := GenerateKey(rand.Reader); err != nil {
			b.Fatal(err)
		}
	}
}

func BenchmarkSignVerify(b *testing.B) {
	pub, priv, err := GenerateKey(rand.Reader)
	if err != nil {
		b.Fatal(err)
	}

	data := make([]byte, 1024)
	if _, err := io.ReadFull(rand.Reader, data); err != nil {
		b.Fatal(err)
	}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		sig, err := Sign(rand.Reader, priv, data)
		if err != nil {
			b.Fatal(err)
		}
		if !Verify(pub, data, sig) {
			b.Fatalf("failed to verify: %d", i)
		}
	}
}

func BenchmarkSignVerifyMethod(b *testing.B) {
	pub, priv, err := GenerateKey(rand.Reader)
	if err != nil {
		b.Fatal(err)
	}

	data := make([]byte, 1024)
	if _, err := io.ReadFull(rand.Reader, data); err != nil {
		b.Fatal(err)
	}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		sig, err := priv.Sign(rand.Reader, data, crypto.Hash(0))
		if err != nil {
			b.Fatal(err)
		}
		if !Verify(pub, data, sig) {
			b.Fatalf("failed to verify: %d", i)
		}
	}
}

func TestVectors(t *testing.T) {
	buf := mustTeeReader(t)
	buf.Write(make([]byte, 64))
	buf.Write([]byte{
		0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
		0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
		0x58, 0x4b, 0x44, 0xef, 0xbc, 0xf6, 0xac, 0x10,
		0x97, 0x67, 0x60, 0xcc, 0x41, 0xf1, 0x82, 0xb,
		0xb8, 0x69, 0x6, 0xf3, 0xad, 0xa5, 0x46, 0x6f,
		0xc2, 0xf3, 0x9, 0x78, 0x7b, 0x3c, 0x6f, 0xd6,
		0x39, 0x19, 0xb1, 0x48, 0xd6, 0x6, 0xb7, 0x79,
		0xfb, 0xf7, 0xda, 0x61, 0xee, 0x57, 0x7b, 0xe6,
	})
	buf.Write([]byte{
		0x56, 0x53, 0xd8, 0x94, 0x41, 0x20, 0x4d, 0x9a,
		0xd6, 0x3c, 0xfa, 0x3a, 0x78, 0x4a, 0x9e, 0x2e,
		0x4d, 0x13, 0x07, 0xa1, 0x66, 0xde, 0xf7, 0xa7,
		0xb4, 0x3a, 0xc8, 0x4f, 0x23, 0x2c, 0x9a, 0x6d,
		0x16, 0x17, 0x10, 0xe6, 0xe7, 0x91, 0x2e, 0x38,
		0x61, 0xe3, 0xe9, 0x9f, 0x90, 0x2c, 0x6d, 0x9,
		0x82, 0xc6, 0x44, 0xd6, 0xeb, 0x12, 0xd0, 0x53,
		0x66, 0x27, 0x61, 0xd0, 0x35, 0x33, 0x31, 0x9d,
	})

	type args struct {
		rand    io.Reader
		message []byte
		opts    crypto.SignerOpts
	}
	tests := []struct {
		name          string
		p             PrivateKey
		args          args
		wantSignature []byte
		wantErr       bool
	}{
		// Test vector from
		// https://github.com/signalapp/libsignal-protocol-c/blob/3a83a4f4ed2302ff6e68ab569c88793b50c22d28/src/curve25519/ed25519/tests/internal_fast_tests.c#L324
		{"ok", []byte{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xbd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
		}, args{buf, make([]byte, 200), crypto.Hash(0)}, []byte{
			0x11, 0xc7, 0xf3, 0xe6, 0xc4, 0xdf, 0x9e, 0x8a,
			0x51, 0x50, 0xe1, 0xdb, 0x3b, 0x30, 0xf9, 0x2d,
			0xe3, 0xa3, 0xb3, 0xaa, 0x43, 0x86, 0x56, 0x54,
			0x5f, 0xa7, 0x39, 0x0f, 0x4b, 0xcc, 0x7b, 0xb2,
			0x6c, 0x43, 0x1d, 0x9e, 0x90, 0x64, 0x3e, 0x4f,
			0x0e, 0xaa, 0x0e, 0x9c, 0x55, 0x77, 0x66, 0xfa,
			0x69, 0xad, 0xa5, 0x76, 0xd6, 0x3d, 0xca, 0xf2,
			0xac, 0x32, 0x6c, 0x11, 0xd0, 0xb9, 0x77, 0x02,
		}, false},
		// Test vector from
		// https://github.com/signalapp/libsignal-protocol-c/blob/3a83a4f4ed2302ff6e68ab569c88793b50c22d28/src/curve25519/ed25519/tests/internal_slow_tests.c#L98
		{"ok", []byte{
			0xb0, 0x3d, 0x85, 0x79, 0x6d, 0x92, 0x89, 0x78,
			0x26, 0xaf, 0x9d, 0xb9, 0x13, 0x98, 0xf3, 0xf9,
			0x73, 0x7d, 0x5f, 0x5c, 0xde, 0x76, 0xd1, 0xc4,
			0x4c, 0x3a, 0x3f, 0xa9, 0x6e, 0xe5, 0x19, 0x46,
		}, args{buf, []byte{
			0x00, 0x01, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x01, 0x01, 0x00, 0x01,
			0x01, 0x01, 0x01, 0x00, 0x01, 0x01, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01, 0x01,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x01,
			0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x01,
			0x01, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x01,
			0x01, 0x00, 0x01, 0x01, 0x00, 0x01, 0x00, 0x01,
			0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01,
			0x01, 0x01, 0x00, 0x00, 0x01, 0x01, 0x01, 0x00,
			0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x01, 0x01,
			0x01, 0x00, 0x01, 0x01, 0x00, 0x01, 0x01, 0x01,
			0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01,
			0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x01,
			0x00, 0x00, 0x01, 0x01, 0x01, 0x00, 0x01, 0x00,
		}, crypto.Hash(0)}, []byte{
			0x15, 0x29, 0x03, 0x38, 0x66, 0x16, 0xcd, 0x26,
			0xbb, 0x3e, 0xec, 0xe2, 0x9f, 0x72, 0xa2, 0x5c,
			0x7d, 0x05, 0xc9, 0xcb, 0x84, 0x3f, 0x92, 0x96,
			0xb3, 0xfb, 0xb9, 0xdd, 0xd6, 0xed, 0x99, 0x04,
			0xc1, 0xa8, 0x02, 0x16, 0xcf, 0x49, 0x3f, 0xf1,
			0xbe, 0x69, 0xf9, 0xf1, 0xcc, 0x16, 0xd7, 0xdc,
			0x6e, 0xd3, 0x78, 0xaa, 0x04, 0xeb, 0x71, 0x51,
			0x9d, 0xe8, 0x7a, 0x5b, 0xd8, 0x49, 0x7b, 0x05,
		}, false},
		// Test vector from
		// https://github.com/signalapp/libsignal-protocol-c/blob/3a83a4f4ed2302ff6e68ab569c88793b50c22d28/src/curve25519/ed25519/tests/internal_slow_tests.c#L165
		//
		// Note that byte 16 is altered before comparing it with the expected
		// signature, so the real value for b[16] is 0xb4 instead of 0xb5.
		// https://github.com/signalapp/libsignal-protocol-c/blob/3a83a4f4ed2302ff6e68ab569c88793b50c22d28/src/curve25519/ed25519/tests/internal_slow_tests.c#L211
		{"ok", []byte{
			0xb8, 0x96, 0x4, 0xb2, 0xcc, 0xe1, 0x1b, 0xe9,
			0xd5, 0x4a, 0xe5, 0x1, 0x1e, 0x3c, 0x2b, 0xfe,
			0x24, 0x35, 0x3f, 0x6, 0x6, 0x73, 0x3e, 0xb7,
			0xe4, 0xa1, 0x6f, 0xb9, 0xe9, 0xbf, 0xff, 0x64,
		}, args{buf, []byte{
			0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x00, 0x01,
			0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x00,
			0x01, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
			0x01, 0x01, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x01, 0x01, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x01, 0x00, 0x01, 0x01, 0x00, 0x01,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01,
			0x00, 0x01, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00,
			0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00,
			0x01, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00,
			0x01, 0x01, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00,
			0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x01, 0x01,
			0x00, 0x00, 0x01, 0x01, 0x01, 0x00, 0x00, 0x01,
			0x01, 0x00, 0x01, 0x01, 0x00, 0x01, 0x01, 0x01,
			0x01, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x01,
			0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00,
		}, crypto.Hash(0)}, []byte{
			0x33, 0x50, 0xa8, 0x68, 0xcd, 0x9e, 0x74, 0x99,
			0xa3, 0x5c, 0x33, 0x75, 0x2b, 0x22, 0x03, 0xf8,
			0xb4, 0x0f, 0xea, 0x8c, 0x33, 0x1c, 0x68, 0x8b,
			0xbb, 0xf3, 0x31, 0xcf, 0x7c, 0x42, 0x37, 0x35,
			0xa0, 0x0e, 0x15, 0xb8, 0x5d, 0x2b, 0xe1, 0xa2,
			0x03, 0x77, 0x94, 0x3d, 0x13, 0x5c, 0xd4, 0x9b,
			0x6a, 0x31, 0xf4, 0xdc, 0xfe, 0x24, 0xad, 0x54,
			0xeb, 0xd2, 0x98, 0x47, 0xf1, 0xcc, 0xbf, 0x0d,
		}, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			gotSignature, err := tt.p.Sign(tt.args.rand, tt.args.message, tt.args.opts)
			if (err != nil) != tt.wantErr {
				t.Errorf("PrivateKey.Sign() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(gotSignature, tt.wantSignature) {
				t.Errorf("PrivateKey.Sign() = %x, want %x", gotSignature, tt.wantSignature)
			}

			// XEdDSA Verify
			publicKey, err := tt.p.PublicKey()
			if err != nil {
				t.Errorf("PrivateKey.Public() error = %v", err)
				return
			}
			if !Verify(publicKey, tt.args.message, gotSignature) {
				t.Error("Verify() = false, want true")
			}

			// Ed265519 Verify
			ed, err := publicKey.ToEd25519()
			if err != nil {
				t.Errorf("PublicKey.ToEd25519() error = %v", err)
			}
			if !ed25519.Verify(ed, tt.args.message, gotSignature) {
				t.Error("ed25519.Verify() = false, want true")
			}
		})
	}
}

func TestSignVerify(t *testing.T) {
	iterations := 1000
	for i := 0; i < iterations; i++ {
		pub, priv, err := GenerateKey(rand.Reader)
		if err != nil {
			t.Fatal(err)
		}

		data := make([]byte, 1024)
		if _, err := io.ReadFull(rand.Reader, data); err != nil {
			t.Fatal(err)
		}

		sig, err := Sign(rand.Reader, priv, data)
		if err != nil {
			t.Fatal(err)
		}
		if !Verify(pub, data, sig) {
			t.Fatalf("failed to verify: %d", i)
		}

		// Ed265519 Verify
		ed, err := pub.ToEd25519()
		if err != nil {
			t.Errorf("PublicKey.ToEd25519() error = %v", err)
		}
		if !ed25519.Verify(ed, data, sig) {
			t.Error("ed25519.Verify() = false, want true")
		}
	}
}

func TestGenerateKey(t *testing.T) {
	buf := mustTeeReader(t)
	buf.Write([]byte{
		0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
		0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
		0x58, 0x4b, 0x44, 0xef, 0xbc, 0xf6, 0xac, 0x10,
		0x97, 0x67, 0x60, 0xcc, 0x41, 0xf1, 0x82, 0xb,
	})

	type args struct {
		rand io.Reader
	}
	tests := []struct {
		name    string
		args    args
		want    PublicKey
		want1   PrivateKey
		wantErr bool
	}{
		{"ok", args{buf}, []byte{
			0x51, 0xf3, 0xfe, 0x72, 0xb4, 0x35, 0x66, 0x52,
			0x16, 0x16, 0x38, 0xfb, 0x9b, 0xf7, 0x17, 0x06,
			0x87, 0xb6, 0x35, 0x53, 0xc1, 0x63, 0x9c, 0x73,
			0xa7, 0x79, 0x1e, 0xd2, 0xba, 0xf8, 0xcb, 0x67,
		}, []byte{
			0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
			0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
			0x58, 0x4b, 0x44, 0xef, 0xbc, 0xf6, 0xac, 0x10,
			0x97, 0x67, 0x60, 0xcc, 0x41, 0xf1, 0x82, 0x0b,
		}, false},
		{"fail", args{buf}, nil, nil, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, got1, err := GenerateKey(tt.args.rand)
			if (err != nil) != tt.wantErr {
				t.Errorf("GenerateKey() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("GenerateKey() got = %x, want %v", got, tt.want)
			}
			if !reflect.DeepEqual(got1, tt.want1) {
				t.Errorf("GenerateKey() got1 = %v, want %v", got1, tt.want1)
			}
		})
	}
}

func TestPublicKey_ToEd25519(t *testing.T) {
	tests := []struct {
		name    string
		p       PublicKey
		want    ed25519.PublicKey
		wantErr bool
	}{

		{"ok", []byte{
			0x59, 0x95, 0xf4, 0x64, 0xe9, 0xd3, 0x4d, 0x5c,
			0xa5, 0x6b, 0x99, 0x05, 0xb9, 0xa3, 0xcc, 0x37,
			0xc4, 0x56, 0xb2, 0xd8, 0xd3, 0x13, 0xed, 0xbc,
			0xf5, 0x84, 0xb7, 0x05, 0xb5, 0xc0, 0x49, 0x55,
		}, []byte{
			0x9c, 0xdc, 0xad, 0xd4, 0x70, 0xc0, 0x55, 0xcc,
			0xbf, 0x70, 0x83, 0xba, 0x09, 0x56, 0xc2, 0x45,
			0xc4, 0x2c, 0xad, 0x21, 0xbd, 0x28, 0xad, 0xb8,
			0x5f, 0x25, 0x7f, 0x9a, 0x9b, 0xb9, 0x31, 0x1c,
		}, false},
		{"fail", []byte{
			0x59, 0x95, 0xf4, 0x64, 0xe9, 0xd3, 0x4d, 0x5c,
			0xa5, 0x6b, 0x99, 0x05, 0xb9, 0xa3, 0xcc, 0x37,
		}, nil, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := tt.p.ToEd25519()
			if (err != nil) != tt.wantErr {
				t.Errorf("PublicKey.ToEd25519() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("PublicKey.ToEd25519() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestPrivateKey_Public(t *testing.T) {
	tests := []struct {
		name string
		p    PrivateKey
		want PublicKey
	}{
		{"ok", []byte{
			0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
			0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
			0x58, 0x4b, 0x44, 0xef, 0xbc, 0xf6, 0xac, 0x10,
			0x97, 0x67, 0x60, 0xcc, 0x41, 0xf1, 0x82, 0x0b,
		}, []byte{
			0x51, 0xf3, 0xfe, 0x72, 0xb4, 0x35, 0x66, 0x52,
			0x16, 0x16, 0x38, 0xfb, 0x9b, 0xf7, 0x17, 0x06,
			0x87, 0xb6, 0x35, 0x53, 0xc1, 0x63, 0x9c, 0x73,
			0xa7, 0x79, 0x1e, 0xd2, 0xba, 0xf8, 0xcb, 0x67,
		}},
		{"fail", []byte{
			0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
			0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
		}, nil},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := tt.p.Public(); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("PrivateKey.Public() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestPrivateKey_SharedKey(t *testing.T) {
	type args struct {
		peerPublicKey []byte
	}
	tests := []struct {
		name    string
		p       PrivateKey
		args    args
		want    []byte
		wantErr bool
	}{
		{"ok basepoint", []byte{
			0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
			0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
			0x58, 0x4b, 0x44, 0xef, 0xbc, 0xf6, 0xac, 0x10,
			0x97, 0x67, 0x60, 0xcc, 0x41, 0xf1, 0x82, 0x0b,
		}, args{curve25519.Basepoint}, []byte{
			0x51, 0xf3, 0xfe, 0x72, 0xb4, 0x35, 0x66, 0x52,
			0x16, 0x16, 0x38, 0xfb, 0x9b, 0xf7, 0x17, 0x06,
			0x87, 0xb6, 0x35, 0x53, 0xc1, 0x63, 0x9c, 0x73,
			0xa7, 0x79, 0x1e, 0xd2, 0xba, 0xf8, 0xcb, 0x67,
		}, false},
		{"ok other", []byte{
			0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
			0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
			0x58, 0x4b, 0x44, 0xef, 0xbc, 0xf6, 0xac, 0x10,
			0x97, 0x67, 0x60, 0xcc, 0x41, 0xf1, 0x82, 0x0b,
		}, args{[]byte{
			0x51, 0xf3, 0xfe, 0x72, 0xb4, 0x35, 0x66, 0x52,
			0x16, 0x16, 0x38, 0xfb, 0x9b, 0xf7, 0x17, 0x06,
			0x87, 0xb6, 0x35, 0x53, 0xc1, 0x63, 0x9c, 0x73,
			0xa7, 0x79, 0x1e, 0xd2, 0xba, 0xf8, 0xcb, 0x67,
		}}, []byte{
			0x73, 0xdd, 0xf1, 0x92, 0x71, 0x49, 0xdc, 0x21,
			0xd6, 0x9e, 0xc1, 0x7b, 0xf9, 0x82, 0x57, 0x15,
			0x53, 0xc0, 0xc7, 0xf9, 0x26, 0x0c, 0xa4, 0x24,
			0x00, 0x29, 0x4b, 0xae, 0x9f, 0xc7, 0x77, 0x60,
		}, false},
		{"zero", []byte{
			0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
			0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
			0x58, 0x4b, 0x44, 0xef, 0xbc, 0xf6, 0xac, 0x10,
			0x97, 0x67, 0x60, 0xcc, 0x41, 0xf1, 0x82, 0x0b,
		}, args{make([]byte, 32)}, nil, true},
		{"fail", []byte{
			0x4e, 0x6c, 0xd9, 0xe8, 0x40, 0x4d, 0xe1, 0xe0,
			0xc5, 0x22, 0x72, 0x8f, 0x78, 0xde, 0x88, 0x26,
		}, args{[]byte{
			0x51, 0xf3, 0xfe, 0x72, 0xb4, 0x35, 0x66, 0x52,
			0x16, 0x16, 0x38, 0xfb, 0x9b, 0xf7, 0x17, 0x06,
			0x87, 0xb6, 0x35, 0x53, 0xc1, 0x63, 0x9c, 0x73,
			0xa7, 0x79, 0x1e, 0xd2, 0xba, 0xf8, 0xcb, 0x67,
		}}, nil, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := tt.p.SharedKey(tt.args.peerPublicKey)
			if (err != nil) != tt.wantErr {
				t.Errorf("PrivateKey.SharedKey() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("PrivateKey.SharedKey() = %x, want %v", got, tt.want)
			}
		})
	}
}

func TestPrivateKey_Sign_error(t *testing.T) {
	type args struct {
		rand    io.Reader
		message []byte
		opts    crypto.SignerOpts
	}
	tests := []struct {
		name          string
		p             PrivateKey
		args          args
		wantSignature []byte
		wantErr       bool
	}{
		{"fail SignerOpts", []byte{
			0x51, 0xf3, 0xfe, 0x72, 0xb4, 0x35, 0x66, 0x52,
			0x16, 0x16, 0x38, 0xfb, 0x9b, 0xf7, 0x17, 0x06,
			0x87, 0xb6, 0x35, 0x53, 0xc1, 0x63, 0x9c, 0x73,
			0xa7, 0x79, 0x1e, 0xd2, 0xba, 0xf8, 0xcb, 0x67,
		}, args{rand.Reader, []byte("message"), crypto.SHA512}, nil, true},
		{"panic", []byte{
			0x51, 0xf3, 0xfe, 0x72, 0xb4, 0x35, 0x66, 0x52,
			0x16, 0x16, 0x38, 0xfb, 0x9b, 0xf7, 0x17, 0x06,
		}, args{rand.Reader, []byte("message"), crypto.Hash(0)}, nil, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if tt.name == "panic" {
				defer func() {
					err := recover()
					if (err != nil) != tt.wantErr {
						t.Errorf("PrivateKey.Sign() recover error = %v, wantErr %v", err, tt.wantErr)
						return
					}
				}()
			}

			gotSignature, err := tt.p.Sign(tt.args.rand, tt.args.message, tt.args.opts)
			if (err != nil) != tt.wantErr {
				t.Errorf("PrivateKey.Sign() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(gotSignature, tt.wantSignature) {
				t.Errorf("PrivateKey.Sign() = %v, want %v", gotSignature, tt.wantSignature)
			}
		})
	}
}

func TestVerify_error(t *testing.T) {
	type args struct {
		publicKey PublicKey
		message   []byte
		sig       []byte
	}
	tests := []struct {
		name string
		args args
		want bool
	}{
		{"fail", args{[]byte{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xbd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
		}, make([]byte, 199), []byte{
			0x11, 0xc7, 0xf3, 0xe6, 0xc4, 0xdf, 0x9e, 0x8a,
			0x51, 0x50, 0xe1, 0xdb, 0x3b, 0x30, 0xf9, 0x2d,
			0xe3, 0xa3, 0xb3, 0xaa, 0x43, 0x86, 0x56, 0x54,
			0x5f, 0xa7, 0x39, 0x0f, 0x4b, 0xcc, 0x7b, 0xb2,
			0x6c, 0x43, 0x1d, 0x9e, 0x90, 0x64, 0x3e, 0x4f,
			0x0e, 0xaa, 0x0e, 0x9c, 0x55, 0x77, 0x66, 0xfa,
			0x69, 0xad, 0xa5, 0x76, 0xd6, 0x3d, 0xca, 0xf2,
			0xac, 0x32, 0x6c, 0x11, 0xd0, 0xb9, 0x77, 0x02,
		}}, false},
		{"panic", args{[]byte{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0xbd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		}, make([]byte, 200), []byte{
			0x11, 0xc7, 0xf3, 0xe6, 0xc4, 0xdf, 0x9e, 0x8a,
			0x51, 0x50, 0xe1, 0xdb, 0x3b, 0x30, 0xf9, 0x2d,
			0xe3, 0xa3, 0xb3, 0xaa, 0x43, 0x86, 0x56, 0x54,
			0x5f, 0xa7, 0x39, 0x0f, 0x4b, 0xcc, 0x7b, 0xb2,
			0x6c, 0x43, 0x1d, 0x9e, 0x90, 0x64, 0x3e, 0x4f,
			0x0e, 0xaa, 0x0e, 0x9c, 0x55, 0x77, 0x66, 0xfa,
			0x69, 0xad, 0xa5, 0x76, 0xd6, 0x3d, 0xca, 0xf2,
			0xac, 0x32, 0x6c, 0x11, 0xd0, 0xb9, 0x77, 0x02,
		}}, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if tt.name == "panic" {
				defer func() {
					err := recover()
					if err == nil {
						t.Errorf("Verify() recover error = %v, wantErr true", err)
						return
					}
				}()
			}
			if got := Verify(tt.args.publicKey, tt.args.message, tt.args.sig); got != tt.want {
				t.Errorf("Verify() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestPublicKey_Equal(t *testing.T) {
	pub1, _, err := GenerateKey(rand.Reader)
	if err != nil {
		t.Fatal(err)
	}
	pub2, _, err := GenerateKey(rand.Reader)
	if err != nil {
		t.Fatal(err)
	}
	type args struct {
		x crypto.PublicKey
	}
	tests := []struct {
		name string
		p    PublicKey
		args args
		want bool
	}{
		{"true", pub1, args{pub1}, true},
		{"false", pub1, args{pub2}, false},
		{"false type", pub1, args{[]byte(pub1)}, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := tt.p.Equal(tt.args.x); got != tt.want {
				t.Errorf("PublicKey.Equal() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestPrivateKey_Equal(t *testing.T) {
	_, priv1, err := GenerateKey(rand.Reader)
	if err != nil {
		t.Fatal(err)
	}
	_, priv2, err := GenerateKey(rand.Reader)
	if err != nil {
		t.Fatal(err)
	}

	type args struct {
		x crypto.PrivateKey
	}
	tests := []struct {
		name string
		p    PrivateKey
		args args
		want bool
	}{
		{"true", priv1, args{priv1}, true},
		{"false", priv1, args{priv2}, false},
		{"false type", priv1, args{[]byte(priv1)}, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := tt.p.Equal(tt.args.x); got != tt.want {
				t.Errorf("PrivateKey.Equal() = %v, want %v", got, tt.want)
			}
		})
	}
}
