/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2024 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "ExtractSelection.h"

#include <MeshComponent.h>
#include <Log.h>
#include <Application.h>

// disable warning generated by clang about the surrounded headers
#include <CamiTKDisableWarnings>
#include <vtkExtractSelection.h>
#include <CamiTKReEnableWarnings>

#include <vtkCallbackCommand.h>
#include <vtkUnstructuredGrid.h>
#include <vtkSelection.h>
#include <vtkDataSet.h>

using namespace camitk;

// -------------------- ExtractSelection --------------------
ExtractSelection::ExtractSelection(ActionExtension* extension) : Action(extension) {
    this->setName("Extract Selection");
    this->setDescription(tr("Extract the current selection"));
    this->setComponentClassName("MeshComponent");
    this->setFamily("Basic Mesh");
    this->addTag(tr("selection"));
}

// --------------- apply -------------------
Action::ApplyStatus ExtractSelection::apply() {
    // set waiting cursor and status bar
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    Application::showStatusBarMessage(tr("Extracting Surface..."));
    Application::resetProgressBar();
    vtkSmartPointer<vtkCallbackCommand> progressCallback = vtkSmartPointer<vtkCallbackCommand>::New();
    progressCallback->SetCallback(&Application::vtkProgressFunction);

    // use the last target
    MeshComponent* targetMesh = dynamic_cast<MeshComponent*>(getTargets().last());

    if (targetMesh == nullptr) {
        QApplication::restoreOverrideCursor();
        CAMITK_WARNING(tr("Mesh \"%1\" is not a MeshComponent. Action aborted.").arg(getTargets().first()->getName()))
        return ABORTED;
    }

    CAMITK_TRACE(tr("Extract selection of \"%1\"").arg(getTargets().first()->getName()))

    vtkSmartPointer<vtkExtractSelection> extractSelection = vtkSmartPointer<vtkExtractSelection>::New();

    if (targetMesh->getPointSet() == nullptr) {
        QApplication::restoreOverrideCursor();
        CAMITK_WARNING(tr("Mesh \"%1\" does not have any vtkPointSet. Action aborted.").arg(getTargets().first()->getName()))
        return ABORTED;
    }

    if (targetMesh->getActiveSelection()->GetNumberOfNodes() == 0) {
        QApplication::restoreOverrideCursor();
        CAMITK_WARNING(tr("Mesh \"%1\" does not have anything currently selected. Action aborted.").arg(getTargets().first()->getName()))
        return ABORTED;
    }

    extractSelection->SetInputData(0, targetMesh->getPointSet());
    extractSelection->SetInputData(1, targetMesh->getActiveSelection());
    extractSelection->AddObserver(vtkCommand::ProgressEvent, progressCallback);
    extractSelection->Update();
    vtkSmartPointer<vtkUnstructuredGrid> grid = vtkUnstructuredGrid::SafeDownCast(extractSelection->GetOutput());

    // create a new MeshComponent with the extracted selection
    new MeshComponent(grid, targetMesh->getName() + " extracted");

    // restore the normal cursor and progress bar
    Application::refresh();
    Application::resetProgressBar();
    Application::showStatusBarMessage("");
    QApplication::restoreOverrideCursor();

    return SUCCESS;
}
