# Usage: python gen.py

import json
import os
from collections import defaultdict

script_dir = os.path.dirname(os.path.abspath(__file__))
usage_file = os.path.join(script_dir, 'usage.json')

with open(usage_file, 'r') as file:
  data = json.load(file)
  
  with open(os.path.join(script_dir, '../USAGE.md'), 'w') as md_file:
    md_file.write("""<!--- This file is generated by gen.py, do not edit it directly. -->
Usage
----

There are many environment variables to control Box64's behaviour, which will be listed below by category.

### Configuration files

In addition to environment variables, by default, Box64 also looks for 2 places for rcfile: the system-wide `/etc/box64.box64rc` and user-specific `~/.box64rc`.
Settings priority follows this order (from highest to lowest): `~/.box64rc` > `/etc/box64.box64rc` > environment variables.

Example configuration:

```
[factorio]
BOX64_DYNAREC_SAFEFLAGS=0
BOX64_DYNAREC_BIGBLOCK=2
BOX64_DYNAREC_FORWARD=1024
BOX64_DYNAREC_CALLRET=1
```

This configuration will apply the specified settings application-wide to any executable named `factorio`.

### Advanced usage

1. **Wildcard Matching**

   Asterisks (`*`) can be used for basic pattern matching in application names. For instance, `[*setup*]` will match any program containing "setup" in its name. Note this implements simple wildcard matching rather than full regex support.
2. **Custom Configuration File**

   The `BOX64_RCFILE` environment variable can specify an alternative configuration file instead of the default `/etc/box64.box64rc`.
3. **Per-File Settings**

   Sections starting with `/` apply to specific files. For example:
   ```
   [/d3d9.dll]
   BOX64_DYNAREC_SAFEFLAGS=0
   ```
   These settings will only affect the `d3d9.dll` file. This syntax also works for **emulated** Linux libraries, e.g., `[/libstdc++.so.6]`.

----

""")

    categories = defaultdict(list)
    for entry in data:
      categories[entry["category"]].append(entry)

    for category, entries in categories.items():
      md_file.write(f"## {category}\n\n")
      for entry in entries:
        md_file.write(f"### {entry['name']}\n\n{entry['description']}\n\n")
        for option in entry['options']:
          md_file.write(f" * {option['key']}: {option['description']} {"[Default]" if option['default'] else ""}\n")
        md_file.write("\n")


  with open(os.path.join(script_dir, '../box64.pod'), 'w') as pod_file:
    pod_file.write("""=head1 NAME

box64 - Linux Userspace x86_64 Emulator with a twist

=head1 SYNOPSIS

B<box64> [B<--help>] [B<--version>] I<executable>

=head1 DESCRIPTION

B<Box64> lets you run x86_64 Linux programs (such as games) on non-x86_64 Linux
systems, like ARM (host system needs to be 64-bit little-endian). Since B<Box64>
uses the native versions of some "system" libraries, like libc, libm, SDL, and
OpenGL, it's easy to integrate and use with most applications, and performance
can be surprisingly high in many cases. B<Box64> integrates with DynaRec (dynamic
recompiler) for the ARM64 platform, providing a speed boost between 5 to 10
times faster than using only the interpreter.

=head1 OPTIONS

=over 8

=item B<-h,--help>

Print box64 help and quit.

=item B<-v,--version>

Print box64 version and quit.

=back

=head1 BRIEF USAGE

There are many environment variables to control B<Box64>'s behaviour. In
addition to environment variables, B<Box64> also looks for 2 places for rcfile:
F</etc/box64.box64rc> and F<~/.box64rc>, in the format of .ini files.
Settings priority: F<~/.box64rc> > F</etc/box64.box64rc> > environment variables.
Example:

    [factorio]
    BOX64_DYNAREC_SAFEFLAGS=0
    BOX64_DYNAREC_BIGBLOCK=2
    BOX64_DYNAREC_FORWARD=1024
    BOX64_DYNAREC_CALLRET=1

=head1 ENVIRONMENT VARIABLES

=over 8

""")

    for entry in data:
      pod_file.write(f"\n=item B<{entry['name']}> =I<{ '|'.join(option['key'] for option in entry['options']) }>\n\n{entry['description']}\n\n")
      for option in entry['options']:
        pod_file.write(f" * {option['key']} : {option['description']} {"[Default]" if option['default'] else ""}\n")
      pod_file.write("\n")

    pod_file.write("""
=back

=cut
""")
