﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace APIGateway {
namespace Model {

/**
 * <p>Represents a method response of a given HTTP status code returned to the
 * client. The method response is passed from the back end through the associated
 * integration response that can be transformed using a mapping template.
 * </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/MethodResponse">AWS
 * API Reference</a></p>
 */
class MethodResponse {
 public:
  AWS_APIGATEWAY_API MethodResponse() = default;
  AWS_APIGATEWAY_API MethodResponse(Aws::Utils::Json::JsonView jsonValue);
  AWS_APIGATEWAY_API MethodResponse& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_APIGATEWAY_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The method response's status code.</p>
   */
  inline const Aws::String& GetStatusCode() const { return m_statusCode; }
  inline bool StatusCodeHasBeenSet() const { return m_statusCodeHasBeenSet; }
  template <typename StatusCodeT = Aws::String>
  void SetStatusCode(StatusCodeT&& value) {
    m_statusCodeHasBeenSet = true;
    m_statusCode = std::forward<StatusCodeT>(value);
  }
  template <typename StatusCodeT = Aws::String>
  MethodResponse& WithStatusCode(StatusCodeT&& value) {
    SetStatusCode(std::forward<StatusCodeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A key-value map specifying required or optional response parameters that API
   * Gateway can send back to the caller. A key defines a method response header and
   * the value specifies whether the associated method response header is required or
   * not. The expression of the key must match the pattern
   * <code>method.response.header.{name}</code>, where <code>name</code> is a valid
   * and unique header name. API Gateway passes certain integration response data to
   * the method response headers specified here according to the mapping you
   * prescribe in the API's IntegrationResponse. The integration response data that
   * can be mapped include an integration response header expressed in
   * <code>integration.response.header.{name}</code>, a static value enclosed within
   * a pair of single quotes (e.g., <code>'application/json'</code>), or a JSON
   * expression from the back-end response payload in the form of
   * <code>integration.response.body.{JSON-expression}</code>, where
   * <code>JSON-expression</code> is a valid JSON expression without the
   * <code>$</code> prefix.)</p>
   */
  inline const Aws::Map<Aws::String, bool>& GetResponseParameters() const { return m_responseParameters; }
  inline bool ResponseParametersHasBeenSet() const { return m_responseParametersHasBeenSet; }
  template <typename ResponseParametersT = Aws::Map<Aws::String, bool>>
  void SetResponseParameters(ResponseParametersT&& value) {
    m_responseParametersHasBeenSet = true;
    m_responseParameters = std::forward<ResponseParametersT>(value);
  }
  template <typename ResponseParametersT = Aws::Map<Aws::String, bool>>
  MethodResponse& WithResponseParameters(ResponseParametersT&& value) {
    SetResponseParameters(std::forward<ResponseParametersT>(value));
    return *this;
  }
  inline MethodResponse& AddResponseParameters(Aws::String key, bool value) {
    m_responseParametersHasBeenSet = true;
    m_responseParameters.emplace(key, value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the Model resources used for the response's content-type. Response
   * models are represented as a key/value map, with a content-type as the key and a
   * Model name as the value.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetResponseModels() const { return m_responseModels; }
  inline bool ResponseModelsHasBeenSet() const { return m_responseModelsHasBeenSet; }
  template <typename ResponseModelsT = Aws::Map<Aws::String, Aws::String>>
  void SetResponseModels(ResponseModelsT&& value) {
    m_responseModelsHasBeenSet = true;
    m_responseModels = std::forward<ResponseModelsT>(value);
  }
  template <typename ResponseModelsT = Aws::Map<Aws::String, Aws::String>>
  MethodResponse& WithResponseModels(ResponseModelsT&& value) {
    SetResponseModels(std::forward<ResponseModelsT>(value));
    return *this;
  }
  template <typename ResponseModelsKeyT = Aws::String, typename ResponseModelsValueT = Aws::String>
  MethodResponse& AddResponseModels(ResponseModelsKeyT&& key, ResponseModelsValueT&& value) {
    m_responseModelsHasBeenSet = true;
    m_responseModels.emplace(std::forward<ResponseModelsKeyT>(key), std::forward<ResponseModelsValueT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  inline bool RequestIdHasBeenSet() const { return m_requestIdHasBeenSet; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  MethodResponse& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_statusCode;
  bool m_statusCodeHasBeenSet = false;

  Aws::Map<Aws::String, bool> m_responseParameters;
  bool m_responseParametersHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_responseModels;
  bool m_responseModelsHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
