/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkMultiBlockDataGroupFilter.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkMultiBlockDataGroupFilter.h"

#include "vtkCellData.h"
#include "vtkDataSet.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMultiBlockDataSet.h"
#include "vtkObjectFactory.h"
#include "vtkStreamingDemandDrivenPipeline.h"

vtkStandardNewMacro(vtkMultiBlockDataGroupFilter);
//-----------------------------------------------------------------------------
vtkMultiBlockDataGroupFilter::vtkMultiBlockDataGroupFilter() = default;

//-----------------------------------------------------------------------------
vtkMultiBlockDataGroupFilter::~vtkMultiBlockDataGroupFilter() = default;

//-----------------------------------------------------------------------------
int vtkMultiBlockDataGroupFilter::RequestInformation(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** vtkNotUsed(inputVector), vtkInformationVector* outputVector)
{
  vtkInformation* info = outputVector->GetInformationObject(0);
  info->Remove(vtkStreamingDemandDrivenPipeline::WHOLE_EXTENT());
  return 1;
}

//-----------------------------------------------------------------------------
int vtkMultiBlockDataGroupFilter::RequestUpdateExtent(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* vtkNotUsed(outputVector))
{
  int numInputs = inputVector[0]->GetNumberOfInformationObjects();
  for (int i = 0; i < numInputs; i++)
  {
    vtkInformation* inInfo = inputVector[0]->GetInformationObject(i);
    if (inInfo->Has(vtkStreamingDemandDrivenPipeline::WHOLE_EXTENT()))
    {
      inInfo->Set(vtkStreamingDemandDrivenPipeline::UPDATE_EXTENT(),
        inInfo->Get(vtkStreamingDemandDrivenPipeline::WHOLE_EXTENT()), 6);
    }
  }
  return 1;
}

//-----------------------------------------------------------------------------
int vtkMultiBlockDataGroupFilter::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  vtkInformation* info = outputVector->GetInformationObject(0);
  vtkMultiBlockDataSet* output =
    vtkMultiBlockDataSet::SafeDownCast(info->Get(vtkDataObject::DATA_OBJECT()));
  if (!output)
  {
    return 0;
  }

  /*
  unsigned int updatePiece = static_cast<unsigned int>(
    info->Get(vtkStreamingDemandDrivenPipeline::UPDATE_PIECE_NUMBER()));
  unsigned int updateNumPieces =  static_cast<unsigned int>(
    info->Get(vtkStreamingDemandDrivenPipeline::UPDATE_NUMBER_OF_PIECES()));
  */

  int numInputs = inputVector[0]->GetNumberOfInformationObjects();
  output->SetNumberOfBlocks(numInputs);
  for (int idx = 0; idx < numInputs; ++idx)
  {
    /*
    // This can be a vtkMultiPieceDataSet if we ever support it.
    vtkMultiBlockDataSet* block = vtkMultiBlockDataSet::New();
    block->SetNumberOfBlocks(updateNumPieces);
    block->Delete();
    */

    vtkDataObject* input = nullptr;
    vtkInformation* inInfo = inputVector[0]->GetInformationObject(idx);
    if (inInfo)
    {
      input = inInfo->Get(vtkDataObject::DATA_OBJECT());
    }
    if (input)
    {
      vtkDataObject* dsCopy = input->NewInstance();
      dsCopy->ShallowCopy(input);
      output->SetBlock(idx, dsCopy);
      dsCopy->Delete();
    }
    else
    {
      output->SetBlock(idx, nullptr);
    }
  }

  if (output->GetNumberOfBlocks() == 1 && output->GetBlock(0) &&
    output->GetBlock(0)->IsA("vtkMultiBlockDataSet"))
  {
    vtkMultiBlockDataSet* block = vtkMultiBlockDataSet::SafeDownCast(output->GetBlock(0));
    block->Register(this);
    output->ShallowCopy(block);
    block->UnRegister(this);
  }

  return 1;
}

//-----------------------------------------------------------------------------
void vtkMultiBlockDataGroupFilter::AddInputData(vtkDataObject* input)
{
  this->AddInputData(0, input);
}

//-----------------------------------------------------------------------------
void vtkMultiBlockDataGroupFilter::AddInputData(int index, vtkDataObject* input)
{
  this->AddInputDataInternal(index, input);
}

//-----------------------------------------------------------------------------
int vtkMultiBlockDataGroupFilter::FillInputPortInformation(int, vtkInformation* info)
{
  info->Set(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkDataObject");
  info->Set(vtkAlgorithm::INPUT_IS_REPEATABLE(), 1);
  info->Set(vtkAlgorithm::INPUT_IS_OPTIONAL(), 1);
  return 1;
}

//-----------------------------------------------------------------------------
void vtkMultiBlockDataGroupFilter::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}
