#!/usr/bin/env python

"""
Managing and presenting periods of time.

Copyright (C) 2014, 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from bisect import bisect_left, bisect_right, insort_left
from datetime import date, datetime, timedelta
from imiptools.dates import check_permitted_values, correct_datetime, \
                            format_datetime, get_datetime, \
                            get_datetime_attributes, \
                            get_recurrence_start, get_recurrence_start_point, \
                            get_start_of_day, \
                            get_tzid, \
                            to_timezone, to_utc_datetime
from imiptools.sql import DatabaseOperations

def ifnone(x, y):
    if x is None: return y
    else: return x

def from_string(s, encoding):
    if s:
        return unicode(s, encoding)
    else:
        return s

def to_string(s, encoding):
    if s:
        return s.encode(encoding)
    else:
        return s

class Comparable:

    "A date/datetime wrapper that allows comparisons with other types."

    def __init__(self, dt):
        self.dt = dt

    def __cmp__(self, other):
        dt = None
        odt = None

        # Find any dates/datetimes.

        if isinstance(self.dt, date):
            dt = self.dt
        if isinstance(other, date):
            odt = other
        elif isinstance(other, Comparable):
            if isinstance(other.dt, date):
                odt = other.dt
            else:
                other = other.dt

        if dt and odt:
            return cmp(dt, odt)
        elif dt:
            return other.__rcmp__(dt)
        elif odt:
            return self.dt.__cmp__(odt)
        else:
            return self.dt.__cmp__(other)

class PointInTime:

    "A base class for special values."

    pass

class StartOfTime(PointInTime):

    "A special value that compares earlier than other values."

    def __cmp__(self, other):
        if isinstance(other, StartOfTime):
            return 0
        else:
            return -1

    def __rcmp__(self, other):
        return -self.__cmp__(other)

    def __nonzero__(self):
        return False

    def __hash__(self):
        return 0

class EndOfTime(PointInTime):

    "A special value that compares later than other values."

    def __cmp__(self, other):
        if isinstance(other, EndOfTime):
            return 0
        else:
            return 1

    def __rcmp__(self, other):
        return -self.__cmp__(other)

    def __nonzero__(self):
        return False

    def __hash__(self):
        return 0

class Endless:

    "A special value indicating an endless period."

    def __cmp__(self, other):
        if isinstance(other, Endless):
            return 0
        else:
            return 1

    def __rcmp__(self, other):
        return -self.__cmp__(other)

    def __nonzero__(self):
        return True

class PeriodBase:

    "A basic period abstraction."

    def __init__(self, start, end):

        """
        Define a period according to 'start' and 'end' which may be special
        start/end of time values or iCalendar-format datetime strings.
        """

        if isinstance(start, (date, PointInTime)):
            self.start = start
        else:
            self.start = get_datetime(start) or StartOfTime()

        if isinstance(end, (date, PointInTime)):
            self.end = end
        else:
            self.end = get_datetime(end) or EndOfTime()

    def as_tuple(self):
        return self.start, self.end

    def __hash__(self):
        return hash((self.get_start(), self.get_end()))

    def __cmp__(self, other):

        "Return a comparison result against 'other' using points in time."

        if isinstance(other, PeriodBase):
            return cmp(
                (Comparable(ifnone(self.get_start_point(), StartOfTime())), Comparable(ifnone(self.get_end_point(), EndOfTime()))),
                (Comparable(ifnone(other.get_start_point(), StartOfTime())), Comparable(ifnone(other.get_end_point(), EndOfTime())))
                )
        else:
            return 1

    def overlaps(self, other):
        return Comparable(ifnone(self.get_end_point(), EndOfTime())) > Comparable(ifnone(other.get_start_point(), StartOfTime())) and \
               Comparable(ifnone(self.get_start_point(), StartOfTime())) < Comparable(ifnone(other.get_end_point(), EndOfTime()))

    def within(self, other):
        return Comparable(ifnone(self.get_start_point(), StartOfTime())) >= Comparable(ifnone(other.get_start_point(), StartOfTime())) and \
               Comparable(ifnone(self.get_end_point(), EndOfTime())) <= Comparable(ifnone(other.get_end_point(), EndOfTime()))

    def common(self, other):
        start = max(Comparable(ifnone(self.get_start_point(), StartOfTime())), Comparable(ifnone(other.get_start_point(), StartOfTime())))
        end = min(Comparable(ifnone(self.get_end_point(), EndOfTime())), Comparable(ifnone(other.get_end_point(), EndOfTime())))
        if start <= end:
            return self.make_corrected(start.dt, end.dt)
        else:
            return None

    def get_key(self):
        return self.get_start(), self.get_end()

    # Datetime and metadata methods.

    def get_start(self):
        return self.start

    def get_end(self):
        return self.end

    def get_start_attr(self):
        return get_datetime_attributes(self.start, self.tzid)

    def get_end_attr(self):
        return get_datetime_attributes(self.end, self.tzid)

    def get_start_item(self):
        return self.get_start(), self.get_start_attr()

    def get_end_item(self):
        return self.get_end(), self.get_end_attr()

    def get_start_point(self):
        return self.start

    def get_end_point(self):
        return self.end

    def get_duration(self):
        start = self.get_start_point()
        end = self.get_end_point()
        if start and end:
            return end - start
        else:
            return Endless()

class Period(PeriodBase):

    "A simple period abstraction."

    def __init__(self, start, end, tzid=None, origin=None):

        """
        Initialise a period with the given 'start' and 'end', having a
        contextual 'tzid', if specified, and an indicated 'origin'.

        All metadata from the start and end points are derived from the supplied
        dates/datetimes.
        """

        PeriodBase.__init__(self, start, end)
        self.tzid = tzid
        self.origin = origin

    def as_tuple(self):
        return self.start, self.end, self.tzid, self.origin

    def __repr__(self):
        return "Period%r" % (self.as_tuple(),)

    # Datetime and metadata methods.

    def get_tzid(self):
        return get_tzid(self.get_start_attr(), self.get_end_attr()) or self.tzid

    def get_start_point(self):
        start = self.get_start()
        if isinstance(start, PointInTime): return start
        else: return to_utc_datetime(start, self.get_tzid())

    def get_end_point(self):
        end = self.get_end()
        if isinstance(end, PointInTime): return end
        else: return to_utc_datetime(end, self.get_tzid())

    # Period and event recurrence logic.

    def is_replaced(self, recurrenceids):

        """
        Return whether this period refers to one of the 'recurrenceids'.
        The 'recurrenceids' should be normalised to UTC datetimes according to
        time zone information provided by their objects or be floating dates or
        datetimes requiring conversion using contextual time zone information.
        """

        for recurrenceid in recurrenceids:
            if self.is_affected(recurrenceid):
                return recurrenceid
        return None

    def is_affected(self, recurrenceid):

        """
        Return whether this period refers to 'recurrenceid'. The 'recurrenceid'
        should be normalised to UTC datetimes according to time zone information
        provided by their objects. Otherwise, this period's contextual time zone
        information is used to convert any date or floating datetime
        representation to a point in time.
        """

        if not recurrenceid:
            return None
        d = get_recurrence_start(recurrenceid)
        dt = get_recurrence_start_point(recurrenceid, self.tzid)

        # Compare the start to dates only, using the normalised start datetime
        # for comparisons with the start point.

        if not isinstance(d, datetime) and self.get_start() == d or self.get_start_point() == dt:
            return recurrenceid

        return None

    # Value correction methods.

    def with_duration(self, duration):

        """
        Return a version of this period with the same start point but with the
        given 'duration'.
        """

        return self.make_corrected(self.get_start(), self.get_start() + duration)

    def check_permitted(self, permitted_values):

        "Check the period against the given 'permitted_values'."

        start = self.get_start()
        end = self.get_end()
        start_errors = check_permitted_values(start, permitted_values)
        end_errors = check_permitted_values(end, permitted_values)

        if not (start_errors or end_errors):
            return None

        return start_errors, end_errors

    def get_corrected(self, permitted_values):

        "Return a corrected version of this period."

        errors = self.check_permitted(permitted_values)

        if not errors:
            return self

        start_errors, end_errors = errors

        start = self.get_start()
        end = self.get_end()

        if start_errors:
            start = correct_datetime(start, permitted_values)
        if end_errors:
            end = correct_datetime(end, permitted_values)

        return self.make_corrected(start, end)

    def make_corrected(self, start, end):
        return self.__class__(start, end, self.tzid, self.origin)

class FreeBusyPeriod(PeriodBase):

    "A free/busy record abstraction."

    def __init__(self, start, end, uid=None, transp=None, recurrenceid=None,
        summary=None, organiser=None):

        """
        Initialise a free/busy period with the given 'start' and 'end' points,
        plus any 'uid', 'transp', 'recurrenceid', 'summary' and 'organiser'
        details.
        """

        PeriodBase.__init__(self, start, end)
        self.uid = uid
        self.transp = transp or None
        self.recurrenceid = recurrenceid or None
        self.summary = summary or None
        self.organiser = organiser or None

    def as_tuple(self, strings_only=False, string_datetimes=False):

        """
        Return the initialisation parameter tuple, converting datetimes and
        false value parameters to strings if 'strings_only' is set to a true
        value. Otherwise, if 'string_datetimes' is set to a true value, only the
        datetime values are converted to strings.
        """

        null = lambda x: (strings_only and [""] or [x])[0]
        return (
            (strings_only or string_datetimes) and format_datetime(self.get_start_point()) or self.start,
            (strings_only or string_datetimes) and format_datetime(self.get_end_point()) or self.end,
            self.uid or null(self.uid),
            self.transp or strings_only and "OPAQUE" or None,
            self.recurrenceid or null(self.recurrenceid),
            self.summary or null(self.summary),
            self.organiser or null(self.organiser)
            )

    def __cmp__(self, other):

        """
        Compare this object to 'other', employing the uid if the periods
        involved are the same.
        """

        result = PeriodBase.__cmp__(self, other)
        if result == 0 and isinstance(other, FreeBusyPeriod):
            return cmp((self.uid, self.recurrenceid), (other.uid, other.recurrenceid))
        else:
            return result

    def get_key(self):
        return self.uid, self.recurrenceid, self.get_start()

    def __repr__(self):
        return "FreeBusyPeriod%r" % (self.as_tuple(),)

    def get_tzid(self):
        return "UTC"

    # Period and event recurrence logic.

    def is_replaced(self, recurrences):

        """
        Return whether this period refers to one of the 'recurrences'.
        The 'recurrences' must be UTC datetimes corresponding to the start of
        the period described by a recurrence.
        """

        for recurrence in recurrences:
            if self.is_affected(recurrence):
                return True
        return False

    def is_affected(self, recurrence):

        """
        Return whether this period refers to 'recurrence'. The 'recurrence' must
        be a UTC datetime corresponding to the start of the period described by
        a recurrence.
        """

        return recurrence and self.get_start_point() == recurrence

    # Value correction methods.

    def make_corrected(self, start, end):
        return self.__class__(start, end)

class FreeBusyOfferPeriod(FreeBusyPeriod):

    "A free/busy record abstraction for an offer period."

    def __init__(self, start, end, uid=None, transp=None, recurrenceid=None,
        summary=None, organiser=None, expires=None):

        """
        Initialise a free/busy period with the given 'start' and 'end' points,
        plus any 'uid', 'transp', 'recurrenceid', 'summary' and 'organiser'
        details.

        An additional 'expires' parameter can be used to indicate an expiry
        datetime in conjunction with free/busy offers made when countering
        event proposals.
        """

        FreeBusyPeriod.__init__(self, start, end, uid, transp, recurrenceid,
            summary, organiser)
        self.expires = expires or None

    def as_tuple(self, strings_only=False, string_datetimes=False):

        """
        Return the initialisation parameter tuple, converting datetimes and
        false value parameters to strings if 'strings_only' is set to a true
        value. Otherwise, if 'string_datetimes' is set to a true value, only the
        datetime values are converted to strings.
        """

        null = lambda x: (strings_only and [""] or [x])[0]
        return FreeBusyPeriod.as_tuple(self, strings_only, string_datetimes) + (
            self.expires or null(self.expires),)

    def __repr__(self):
        return "FreeBusyOfferPeriod%r" % (self.as_tuple(),)

class FreeBusyGroupPeriod(FreeBusyPeriod):

    "A free/busy record abstraction for a quota group period."

    def __init__(self, start, end, uid=None, transp=None, recurrenceid=None,
        summary=None, organiser=None, attendee=None):

        """
        Initialise a free/busy period with the given 'start' and 'end' points,
        plus any 'uid', 'transp', 'recurrenceid', 'summary' and 'organiser'
        details.

        An additional 'attendee' parameter can be used to indicate the identity
        of the attendee recording the period.
        """

        FreeBusyPeriod.__init__(self, start, end, uid, transp, recurrenceid,
            summary, organiser)
        self.attendee = attendee or None

    def as_tuple(self, strings_only=False, string_datetimes=False):

        """
        Return the initialisation parameter tuple, converting datetimes and
        false value parameters to strings if 'strings_only' is set to a true
        value. Otherwise, if 'string_datetimes' is set to a true value, only the
        datetime values are converted to strings.
        """

        null = lambda x: (strings_only and [""] or [x])[0]
        return FreeBusyPeriod.as_tuple(self, strings_only, string_datetimes) + (
            self.attendee or null(self.attendee),)

    def __cmp__(self, other):

        """
        Compare this object to 'other', employing the uid if the periods
        involved are the same.
        """

        result = FreeBusyPeriod.__cmp__(self, other)
        if isinstance(other, FreeBusyGroupPeriod) and result == 0:
            return cmp(self.attendee, other.attendee)
        else:
            return result

    def __repr__(self):
        return "FreeBusyGroupPeriod%r" % (self.as_tuple(),)

class RecurringPeriod(Period):
    
    """
    A period with iCalendar metadata attributes and origin information from an
    object.
    """
    
    def __init__(self, start, end, tzid=None, origin=None, start_attr=None, end_attr=None):
        Period.__init__(self, start, end, tzid, origin)
        self.start_attr = start_attr
        self.end_attr = end_attr

    def get_start_attr(self):
        return self.start_attr

    def get_end_attr(self):
        return self.end_attr

    def as_tuple(self):
        return self.start, self.end, self.tzid, self.origin, self.start_attr, self.end_attr
    
    def __repr__(self):
        return "RecurringPeriod%r" % (self.as_tuple(),)

    def make_corrected(self, start, end):
        return self.__class__(start, end, self.tzid, self.origin, self.get_start_attr(), self.get_end_attr())

class FreeBusyCollectionBase:

    "Common operations on free/busy period collections."

    period_columns = [
        "start", "end", "object_uid", "transp", "object_recurrenceid",
        "summary", "organiser"
        ]

    period_class = FreeBusyPeriod

    def __init__(self, mutable=True):
        self.mutable = mutable

    def _check_mutable(self):
        if not self.mutable:
            raise TypeError, "Cannot mutate this collection."

    def copy(self):

        "Make an independent mutable copy of the collection."

        return FreeBusyCollection(list(self), True)

    def make_period(self, t):

        """
        Make a period using the given tuple of arguments and the collection's
        column details.
        """

        args = []
        for arg, column in zip(t, self.period_columns):
            args.append(from_string(arg, "utf-8"))
        return self.period_class(*args)

    def make_tuple(self, t):

        """
        Return a tuple from the given tuple 't' conforming to the collection's
        column details.
        """

        args = []
        for arg, column in zip(t, self.period_columns):
            args.append(arg)
        return tuple(args)

    # List emulation methods.

    def __iadd__(self, periods):
        for period in periods:
            self.insert_period(period)
        return self

    def append(self, period):
        self.insert_period(period)

    # Operations.

    def can_schedule(self, periods, uid, recurrenceid):

        """
        Return whether the collection can accommodate the given 'periods'
        employing the specified 'uid' and 'recurrenceid'.
        """

        for conflict in self.have_conflict(periods, True):
            if conflict.uid != uid or conflict.recurrenceid != recurrenceid:
                return False

        return True

    def have_conflict(self, periods, get_conflicts=False):

        """
        Return whether any period in the collection overlaps with the given
        'periods', returning a collection of such overlapping periods if
        'get_conflicts' is set to a true value.
        """

        conflicts = set()
        for p in periods:
            overlapping = self.period_overlaps(p, get_conflicts)
            if overlapping:
                if get_conflicts:
                    conflicts.update(overlapping)
                else:
                    return True

        if get_conflicts:
            return conflicts
        else:
            return False

    def period_overlaps(self, period, get_periods=False):

        """
        Return whether any period in the collection overlaps with the given
        'period', returning a collection of overlapping periods if 'get_periods'
        is set to a true value.
        """

        overlapping = self.get_overlapping([period])

        if get_periods:
            return overlapping
        else:
            return len(overlapping) != 0

    def replace_overlapping(self, period, replacements):

        """
        Replace existing periods in the collection within the given 'period',
        using the given 'replacements'.
        """

        self._check_mutable()

        self.remove_overlapping(period)
        for replacement in replacements:
            self.insert_period(replacement)

    def coalesce_freebusy(self):

        "Coalesce the periods in the collection, returning a new collection."

        if not self:
            return FreeBusyCollection()

        fb = []

        it = iter(self)
        period = it.next()

        start = period.get_start_point()
        end = period.get_end_point()

        try:
            while True:
                period = it.next()
                if period.get_start_point() > end:
                    fb.append(self.period_class(start, end))
                    start = period.get_start_point()
                    end = period.get_end_point()
                else:
                    end = max(end, period.get_end_point())
        except StopIteration:
            pass

        fb.append(self.period_class(start, end))
        return FreeBusyCollection(fb)

    def invert_freebusy(self):

        "Return the free periods from the collection as a new collection."

        if not self:
            return FreeBusyCollection([self.period_class(None, None)])

        # Coalesce periods that overlap or are adjacent.

        fb = self.coalesce_freebusy()
        free = []

        # Add a start-of-time period if appropriate.

        first = fb[0].get_start_point()
        if first:
            free.append(self.period_class(None, first))

        start = fb[0].get_end_point()

        for period in fb[1:]:
            free.append(self.period_class(start, period.get_start_point()))
            start = period.get_end_point()

        # Add an end-of-time period if appropriate.

        if start:
            free.append(self.period_class(start, None))

        return FreeBusyCollection(free)

    def _update_freebusy(self, periods, uid, recurrenceid):

        """
        Update the free/busy details with the given 'periods', using the given
        'uid' plus 'recurrenceid' to remove existing periods.
        """

        self._check_mutable()

        self.remove_specific_event_periods(uid, recurrenceid)

        for p in periods:
            self.insert_period(p)

    def update_freebusy(self, periods, transp, uid, recurrenceid, summary, organiser):

        """
        Update the free/busy details with the given 'periods', 'transp' setting,
        'uid' plus 'recurrenceid' and 'summary' and 'organiser' details.
        """

        new_periods = []

        for p in periods:
            new_periods.append(
                self.period_class(p.get_start_point(), p.get_end_point(), uid, transp, recurrenceid, summary, organiser)
                )

        self._update_freebusy(new_periods, uid, recurrenceid)

class SupportAttendee:

    "A mix-in that supports the affected attendee in free/busy periods."

    period_columns = FreeBusyCollectionBase.period_columns + ["attendee"]
    period_class = FreeBusyGroupPeriod

    def _update_freebusy(self, periods, uid, recurrenceid, attendee=None):

        """
        Update the free/busy details with the given 'periods', using the given
        'uid' plus 'recurrenceid' and 'attendee' to remove existing periods.
        """

        self._check_mutable()

        self.remove_specific_event_periods(uid, recurrenceid, attendee)

        for p in periods:
            self.insert_period(p)

    def update_freebusy(self, periods, transp, uid, recurrenceid, summary, organiser, attendee=None):

        """
        Update the free/busy details with the given 'periods', 'transp' setting,
        'uid' plus 'recurrenceid' and 'summary' and 'organiser' details.

        An optional 'attendee' indicates the attendee affected by the period.
        """

        new_periods = []

        for p in periods:
            new_periods.append(
                self.period_class(p.get_start_point(), p.get_end_point(), uid, transp, recurrenceid, summary, organiser, attendee)
                )

        self._update_freebusy(new_periods, uid, recurrenceid, attendee)

class SupportExpires:

    "A mix-in that supports the expiry datetime in free/busy periods."

    period_columns = FreeBusyCollectionBase.period_columns + ["expires"]
    period_class = FreeBusyOfferPeriod

    def update_freebusy(self, periods, transp, uid, recurrenceid, summary, organiser, expires=None):

        """
        Update the free/busy details with the given 'periods', 'transp' setting,
        'uid' plus 'recurrenceid' and 'summary' and 'organiser' details.

        An optional 'expires' datetime string indicates the expiry time of any
        free/busy offer.
        """

        new_periods = []

        for p in periods:
            new_periods.append(
                self.period_class(p.get_start_point(), p.get_end_point(), uid, transp, recurrenceid, summary, organiser, expires)
                )

        self._update_freebusy(new_periods, uid, recurrenceid)

class FreeBusyCollection(FreeBusyCollectionBase):

    "An abstraction for a collection of free/busy periods."

    def __init__(self, periods=None, mutable=True):

        """
        Initialise the collection with the given list of 'periods', or start an
        empty collection if no list is given. If 'mutable' is indicated, the
        collection may be changed; otherwise, an exception will be raised.
        """

        FreeBusyCollectionBase.__init__(self, mutable)
        self.periods = periods or []

    # List emulation methods.

    def __nonzero__(self):
        return bool(self.periods)

    def __iter__(self):
        return iter(self.periods)

    def __len__(self):
        return len(self.periods)

    def __getitem__(self, i):
        return self.periods[i]

    # Operations.

    def insert_period(self, period):

        "Insert the given 'period' into the collection."

        self._check_mutable()

        i = bisect_left(self.periods, period)
        if i == len(self.periods):
            self.periods.append(period)
        elif self.periods[i] != period:
            self.periods.insert(i, period)

    def remove_periods(self, periods):

        "Remove the given 'periods' from the collection."

        self._check_mutable()

        for period in periods:
            i = bisect_left(self.periods, period)
            if i < len(self.periods) and self.periods[i] == period:
                del self.periods[i]

    def remove_event_periods(self, uid, recurrenceid=None):

        """
        Remove from the collection all periods associated with 'uid' and
        'recurrenceid' (which if omitted causes the "parent" object's periods to
        be referenced).

        Return the removed periods.
        """

        self._check_mutable()

        removed = []
        i = 0
        while i < len(self.periods):
            fb = self.periods[i]
            if fb.uid == uid and fb.recurrenceid == recurrenceid:
                removed.append(self.periods[i])
                del self.periods[i]
            else:
                i += 1

        return removed

    # Specific period removal when updating event details.

    remove_specific_event_periods = remove_event_periods

    def remove_additional_periods(self, uid, recurrenceids=None):

        """
        Remove from the collection all periods associated with 'uid' having a
        recurrence identifier indicating an additional or modified period.

        If 'recurrenceids' is specified, remove all periods associated with
        'uid' that do not have a recurrence identifier in the given list.

        Return the removed periods.
        """

        self._check_mutable()

        removed = []
        i = 0
        while i < len(self.periods):
            fb = self.periods[i]
            if fb.uid == uid and fb.recurrenceid and (
                recurrenceids is None or
                recurrenceids is not None and fb.recurrenceid not in recurrenceids
                ):
                removed.append(self.periods[i])
                del self.periods[i]
            else:
                i += 1

        return removed

    def remove_affected_period(self, uid, start):

        """
        Remove from the collection the period associated with 'uid' that
        provides an occurrence starting at the given 'start' (provided by a
        recurrence identifier, converted to a datetime). A recurrence identifier
        is used to provide an alternative time period whilst also acting as a
        reference to the originally-defined occurrence.

        Return any removed period in a list.
        """

        self._check_mutable()

        removed = []

        search = Period(start, start)
        found = bisect_left(self.periods, search)

        while found < len(self.periods):
            fb = self.periods[found]

            # Stop looking if the start no longer matches the recurrence identifier.

            if fb.get_start_point() != search.get_start_point():
                break

            # If the period belongs to the parent object, remove it and return.

            if not fb.recurrenceid and uid == fb.uid:
                removed.append(self.periods[found])
                del self.periods[found]
                break

            # Otherwise, keep looking for a matching period.

            found += 1

        return removed

    def periods_from(self, period):

        "Return the entries in the collection at or after 'period'."

        first = bisect_left(self.periods, period)
        return self.periods[first:]

    def periods_until(self, period):

        "Return the entries in the collection before 'period'."

        last = bisect_right(self.periods, Period(period.get_end(), period.get_end(), period.get_tzid()))
        return self.periods[:last]

    def get_overlapping(self, periods):

        """
        Return the entries in the collection providing periods overlapping with
        the given sorted collection of 'periods'.
        """

        return get_overlapping(self.periods, periods)

    def remove_overlapping(self, period):

        "Remove all periods overlapping with 'period' from the collection."

        self._check_mutable()

        overlapping = self.get_overlapping([period])

        if overlapping:
            for fb in overlapping:
                self.periods.remove(fb)

class FreeBusyGroupCollection(SupportAttendee, FreeBusyCollection):

    "A collection of quota group free/busy objects."

    def remove_specific_event_periods(self, uid, recurrenceid=None, attendee=None):

        """
        Remove from the collection all periods associated with 'uid' and
        'recurrenceid' (which if omitted causes the "parent" object's periods to
        be referenced) and any 'attendee'.

        Return the removed periods.
        """

        self._check_mutable()

        removed = []
        i = 0
        while i < len(self.periods):
            fb = self.periods[i]
            if fb.uid == uid and fb.recurrenceid == recurrenceid and fb.attendee == attendee:
                removed.append(self.periods[i])
                del self.periods[i]
            else:
                i += 1

        return removed

class FreeBusyOffersCollection(SupportExpires, FreeBusyCollection):

    "A collection of offered free/busy objects."

    pass

class FreeBusyDatabaseCollection(FreeBusyCollectionBase, DatabaseOperations):

    """
    An abstraction for a collection of free/busy periods stored in a database
    system.
    """

    def __init__(self, cursor, table_name, column_names=None, filter_values=None,
        mutable=True, paramstyle=None):

        """
        Initialise the collection with the given 'cursor' and with the
        'table_name', 'column_names' and 'filter_values' configuring the
        selection of data. If 'mutable' is indicated, the collection may be
        changed; otherwise, an exception will be raised.
        """

        FreeBusyCollectionBase.__init__(self, mutable)
        DatabaseOperations.__init__(self, column_names, filter_values, paramstyle)
        self.cursor = cursor
        self.table_name = table_name

    # List emulation methods.

    def __nonzero__(self):
        return len(self) and True or False

    def __iter__(self):
        query, values = self.get_query(
            "select %(columns)s from %(table)s :condition" % {
                "columns" : self.columnlist(self.period_columns),
                "table" : self.table_name
                })
        self.cursor.execute(query, values)
        return iter(map(lambda t: self.make_period(t), self.cursor.fetchall()))

    def __len__(self):
        query, values = self.get_query(
            "select count(*) from %(table)s :condition" % {
                "table" : self.table_name
                })
        self.cursor.execute(query, values)
        result = self.cursor.fetchone()
        return result and int(result[0]) or 0

    def __getitem__(self, i):
        return list(iter(self))[i]

    # Operations.

    def insert_period(self, period):

        "Insert the given 'period' into the collection."

        self._check_mutable()

        columns, values = self.period_columns, period.as_tuple(string_datetimes=True)

        query, values = self.get_query(
            "insert into %(table)s (:columns) values (:values)" % {
                "table" : self.table_name
                },
            columns, [to_string(v, "utf-8") for v in values])

        self.cursor.execute(query, values)

    def remove_periods(self, periods):

        "Remove the given 'periods' from the collection."

        self._check_mutable()

        for period in periods:
            values = period.as_tuple(string_datetimes=True)

            query, values = self.get_query(
                "delete from %(table)s :condition" % {
                    "table" : self.table_name
                    },
                self.period_columns, [to_string(v, "utf-8") for v in values])

            self.cursor.execute(query, values)

    def remove_event_periods(self, uid, recurrenceid=None):

        """
        Remove from the collection all periods associated with 'uid' and
        'recurrenceid' (which if omitted causes the "parent" object's periods to
        be referenced).

        Return the removed periods.
        """

        self._check_mutable()

        if recurrenceid:
            columns, values = ["object_uid", "object_recurrenceid"], [uid, recurrenceid]
        else:
            columns, values = ["object_uid", "object_recurrenceid is null"], [uid]

        query, _values = self.get_query(
            "select %(columns)s from %(table)s :condition" % {
                "columns" : self.columnlist(self.period_columns),
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, _values)
        removed = self.cursor.fetchall()

        query, values = self.get_query(
            "delete from %(table)s :condition" % {
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, values)

        return map(lambda t: self.make_period(t), removed)

    # Specific period removal when updating event details.

    remove_specific_event_periods = remove_event_periods

    def remove_additional_periods(self, uid, recurrenceids=None):

        """
        Remove from the collection all periods associated with 'uid' having a
        recurrence identifier indicating an additional or modified period.

        If 'recurrenceids' is specified, remove all periods associated with
        'uid' that do not have a recurrence identifier in the given list.

        Return the removed periods.
        """

        self._check_mutable()

        if not recurrenceids:
            columns, values = ["object_uid", "object_recurrenceid is not null"], [uid]
        else:
            columns, values = ["object_uid", "object_recurrenceid not in ?", "object_recurrenceid is not null"], [uid, tuple(recurrenceids)]

        query, _values = self.get_query(
            "select %(columns)s from %(table)s :condition" % {
                "columns" : self.columnlist(self.period_columns),
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, _values)
        removed = self.cursor.fetchall()

        query, values = self.get_query(
            "delete from %(table)s :condition" % {
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, values)

        return map(lambda t: self.make_period(t), removed)

    def remove_affected_period(self, uid, start):

        """
        Remove from the collection the period associated with 'uid' that
        provides an occurrence starting at the given 'start' (provided by a
        recurrence identifier, converted to a datetime). A recurrence identifier
        is used to provide an alternative time period whilst also acting as a
        reference to the originally-defined occurrence.

        Return any removed period in a list.
        """

        self._check_mutable()

        start = format_datetime(start)

        columns, values = ["object_uid", "start", "object_recurrenceid is null"], [uid, start]

        query, _values = self.get_query(
            "select %(columns)s from %(table)s :condition" % {
                "columns" : self.columnlist(self.period_columns),
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, _values)
        removed = self.cursor.fetchall()

        query, values = self.get_query(
            "delete from %(table)s :condition" % {
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, values)

        return map(lambda t: self.make_period(t), removed)

    def periods_from(self, period):

        "Return the entries in the collection at or after 'period'."

        start = format_datetime(period.get_start_point())

        columns, values = [], []

        if start:
            columns.append("start >= ?")
            values.append(start)

        query, values = self.get_query(
            "select %(columns)s from %(table)s :condition" % {
                "columns" : self.columnlist(self.period_columns),
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, values)

        return map(lambda t: self.make_period(t), self.cursor.fetchall())

    def periods_until(self, period):

        "Return the entries in the collection before 'period'."

        end = format_datetime(period.get_end_point())

        columns, values = [], []

        if end:
            columns.append("start < ?")
            values.append(end)

        query, values = self.get_query(
            "select %(columns)s from %(table)s :condition" % {
                "columns" : self.columnlist(self.period_columns),
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, values)

        return map(lambda t: self.make_period(t), self.cursor.fetchall())

    def get_overlapping(self, periods):

        """
        Return the entries in the collection providing periods overlapping with
        the given sorted collection of 'periods'.
        """

        overlapping = set()

        for period in periods:
            columns, values = self._get_period_values(period)

            query, values = self.get_query(
                "select %(columns)s from %(table)s :condition" % {
                    "columns" : self.columnlist(self.period_columns),
                    "table" : self.table_name
                    },
                columns, values)

            self.cursor.execute(query, values)

            overlapping.update(map(lambda t: self.make_period(t), self.cursor.fetchall()))

        overlapping = list(overlapping)
        overlapping.sort()
        return overlapping

    def remove_overlapping(self, period):

        "Remove all periods overlapping with 'period' from the collection."

        self._check_mutable()

        columns, values = self._get_period_values(period)

        query, values = self.get_query(
            "delete from %(table)s :condition" % {
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, values)

    def _get_period_values(self, period):

        start = format_datetime(period.get_start_point())
        end = format_datetime(period.get_end_point())

        columns, values = [], []

        if end:
            columns.append("start < ?")
            values.append(end)
        if start:
            columns.append("end > ?")
            values.append(start)

        return columns, values

class FreeBusyGroupDatabaseCollection(SupportAttendee, FreeBusyDatabaseCollection):

    "A collection of quota group free/busy objects."

    def remove_specific_event_periods(self, uid, recurrenceid=None, attendee=None):

        """
        Remove from the collection all periods associated with 'uid' and
        'recurrenceid' (which if omitted causes the "parent" object's periods to
        be referenced) and any 'attendee'.

        Return the removed periods.
        """

        self._check_mutable()

        columns, values = ["object_uid"], [uid]

        if recurrenceid:
            columns.append("object_recurrenceid")
            values.append(recurrenceid)
        else:
            columns.append("object_recurrenceid is null")

        if attendee:
            columns.append("attendee")
            values.append(attendee)
        else:
            columns.append("attendee is null")

        query, _values = self.get_query(
            "select %(columns)s from %(table)s :condition" % {
                "columns" : self.columnlist(self.period_columns),
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, _values)
        removed = self.cursor.fetchall()

        query, values = self.get_query(
            "delete from %(table)s :condition" % {
                "table" : self.table_name
                },
            columns, values)

        self.cursor.execute(query, values)

        return map(lambda t: self.make_period(t), removed)

class FreeBusyOffersDatabaseCollection(SupportExpires, FreeBusyDatabaseCollection):

    "A collection of offered free/busy objects."

    pass

def get_overlapping(first, second):

    """
    Return the entries in the sorted 'first' collection that are overlapping
    with the given sorted 'second' collection.
    """

    if not first or not second:
        return []

    # Examine each period in the second collection, attempting to match periods
    # in the first collection.

    overlapping = set()

    for p2 in second:
        last_point = p2.get_end_point()

        # Examine the first collection up to the point where no matches will
        # occur.

        for p1 in first:
            if p1.get_start_point() > last_point:
                break
            elif p1.overlaps(p2):
                overlapping.add(p1)

    overlapping = list(overlapping)
    overlapping.sort()
    return overlapping

# Period layout.

def get_scale(periods, tzid, view_period=None):

    """
    Return a time scale from the given list of 'periods'.

    The given 'tzid' is used to make sure that the times are defined according
    to the chosen time zone.

    An optional 'view_period' is used to constrain the scale to the given
    period.

    The returned scale is a mapping from time to (starting, ending) tuples,
    where starting and ending are collections of periods.
    """

    scale = {}
    view_start = view_period and to_timezone(view_period.get_start_point(), tzid) or None
    view_end = view_period and to_timezone(view_period.get_end_point(), tzid) or None

    for p in periods:

        # Add a point and this event to the starting list.

        start = to_timezone(p.get_start(), tzid)
        start = view_start and max(start, view_start) or start
        if not scale.has_key(start):
            scale[start] = [], []
        scale[start][0].append(p)

        # Add a point and this event to the ending list.

        end = to_timezone(p.get_end(), tzid)
        end = view_end and min(end, view_end) or end
        if not scale.has_key(end):
            scale[end] = [], []
        scale[end][1].append(p)

    return scale

class Point:

    "A qualified point in time."

    PRINCIPAL, REPEATED = 0, 1

    def __init__(self, point, indicator=None):
        self.point = point
        self.indicator = indicator or self.PRINCIPAL

    def __hash__(self):
        return hash((self.point, self.indicator))

    def __cmp__(self, other):
        if isinstance(other, Point):
            return cmp((self.point, self.indicator), (other.point, other.indicator))
        elif isinstance(other, datetime):
            return cmp(self.point, other)
        else:
            return 1

    def __eq__(self, other):
        return self.__cmp__(other) == 0

    def __ne__(self, other):
        return not self == other

    def __lt__(self, other):
        return self.__cmp__(other) < 0

    def __le__(self, other):
        return self.__cmp__(other) <= 0

    def __gt__(self, other):
        return not self <= other

    def __ge__(self, other):
        return not self < other

    def __repr__(self):
        return "Point(%r, Point.%s)" % (self.point, self.indicator and "REPEATED" or "PRINCIPAL")

def get_slots(scale):

    """
    Return an ordered list of time slots from the given 'scale'.

    Each slot is a tuple containing details of a point in time for the start of
    the slot, together with a list of parallel event periods.

    Each point in time is described as a Point representing the actual point in
    time together with an indicator of the nature of the point in time (as a
    principal point in a time scale or as a repeated point used to terminate
    events occurring for an instant in time).
    """

    slots = []
    active = []

    points = scale.items()
    points.sort()

    for point, (starting, ending) in points:
        ending = set(ending)
        instants = ending.intersection(starting)

        # Discard all active events ending at or before this start time.
        # Free up the position in the active list.

        for t in ending.difference(instants):
            i = active.index(t)
            active[i] = None

        # For each event starting at the current point, fill any newly-vacated
        # position or add to the end of the active list.

        for t in starting:
            try:
                i = active.index(None)
                active[i] = t
            except ValueError:
                active.append(t)

        # Discard vacant positions from the end of the active list.

        while active and active[-1] is None:
            active.pop()

        # Add an entry for the time point before "instants".

        slots.append((Point(point), active[:]))

        # Discard events ending at the same time as they began.

        if instants:
            for t in instants:
                i = active.index(t)
                active[i] = None

            # Discard vacant positions from the end of the active list.

            while active and active[-1] is None:
                active.pop()

            # Add another entry for the time point after "instants".

            slots.append((Point(point, Point.REPEATED), active[:]))

    return slots

def add_day_start_points(slots, tzid):

    """
    Introduce into the 'slots' any day start points required by multi-day
    periods. The 'tzid' is required to make sure that appropriate time zones
    are chosen and not necessarily those provided by the existing time points.
    """

    new_slots = []
    current_date = None
    previously_active = []

    for point, active in slots:
        start_of_day = get_start_of_day(point.point, tzid)
        this_date = point.point.date()

        # For each new day, add a slot for the start of the day where periods
        # are active and where no such slot already exists.

        if this_date != current_date:

            # Fill in days where events remain active.

            if current_date:
                current_date += timedelta(1)
                while current_date < this_date:
                    new_slots.append((Point(get_start_of_day(current_date, tzid)), previously_active))
                    current_date += timedelta(1)
            else:
                current_date = this_date

            # Add any continuing periods.

            if point.point != start_of_day:
                new_slots.append((Point(start_of_day), previously_active))

        # Add the currently active periods at this point in time.

        previously_active = active

    for t in new_slots:
        insort_left(slots, t)

def remove_end_slot(slots, view_period):

    """
    Remove from 'slots' any slot situated at the end of the given 'view_period'.
    """

    end = view_period.get_end_point()
    if not end or not slots:
        return
    i = bisect_left(slots, (Point(end), None))
    if i < len(slots):
        del slots[i:]

def add_slots(slots, points):

    """
    Introduce into the 'slots' entries for those in 'points' that are not
    already present, propagating active periods from time points preceding
    those added.
    """

    new_slots = []

    for point in points:
        i = bisect_left(slots, (point,)) # slots is [(point, active)...]
        if i < len(slots) and slots[i][0] == point:
            continue

        new_slots.append((point, i > 0 and slots[i-1][1] or []))

    for t in new_slots:
        insort_left(slots, t)

def partition_by_day(slots):

    """
    Return a mapping from dates to time points provided by 'slots'.
    """

    d = {}

    for point, value in slots:
        day = point.point.date()
        if not d.has_key(day):
            d[day] = []
        d[day].append((point, value))

    return d

def add_empty_days(days, tzid, start=None, end=None):

    """
    Add empty days to 'days' between busy days, and optionally from the given
    'start' day and until the given 'end' day.
    """

    last_day = start - timedelta(1)
    all_days = days.keys()
    all_days.sort()

    for day in all_days:
        if last_day:
            empty_day = last_day + timedelta(1)
            while empty_day < day:
                days[empty_day] = [(Point(get_start_of_day(empty_day, tzid)), None)]
                empty_day += timedelta(1)
        last_day = day

    if end:
        empty_day = last_day + timedelta(1)
        while empty_day < end:
            days[empty_day] = [(Point(get_start_of_day(empty_day, tzid)), None)]
            empty_day += timedelta(1)

def get_spans(slots):

    "Inspect the given 'slots', returning a mapping of period keys to spans."

    points = [point for point, active in slots]
    spans = {}

    for _point, active in slots:
        for p in active:
            if p:
                key = p.get_key()
                start_slot = bisect_left(points, p.get_start())
                end_slot = bisect_left(points, p.get_end())
                spans[key] = end_slot - start_slot

    return spans

# vim: tabstop=4 expandtab shiftwidth=4
