{
  Copyright 2008-2022 Michalis Kamburelis.

  This file is part of "Castle Game Engine".

  "Castle Game Engine" is free software; see the file COPYING.txt,
  included in this distribution, for details about the copyright.

  "Castle Game Engine" is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ----------------------------------------------------------------------------
}

{$ifdef read_interface}
  { Base type for cubic environment map textures. }
  TAbstractEnvironmentTextureNode = class(TAbstractSingleTextureNode)
  {$I auto_generated_node_helpers/x3dnodes_x3denvironmenttexturenode.inc}
  end;

  { Cube environment map texture defined as a six individual 2D texture nodes. }
  TComposedCubeMapTextureNode = class(TAbstractEnvironmentTextureNode)
  strict private
    FAlphaChannelData: TAlphaChannel;
  strict protected
    function AlphaChannelData: TAlphaChannel; override;
  public
    { Make sure all 6 sides are loaded.
      Make sure that all 6 fields for each cube map side are assigned,
      are TAbstractTexture2DNode instance (X3D spec requires it),
      load them, and check are they TCastleImage (we cannot suport
      TGPUCompressedImage when loading cubemaps this way,
      as we have to rotate images from ComposedCubeMapTextureNode
      when passing them to GPU, which is only possible for TCastleImage).

      Also calculate AlphaChannel for the whole cube map.
      Our AlphaChannel method will reflect the state from
      last call of this. }
    function LoadSides: boolean;

  {$I auto_generated_node_helpers/x3dnodes_composedcubemaptexture.inc}
  end;

  { Cube environment map texture generated by rendering the 3D world,
    useful for real-time mirrors. }
  TGeneratedCubeMapTextureNode = class(TAbstractEnvironmentTextureNode)
  strict private
    type
      TGenCubeMapGenTex = class(TGeneratedTextureFunctionality)
      strict private
        FParent: TGeneratedCubeMapTextureNode;
      protected
        function GetUpdate: TTextureUpdate; override;
        procedure SetUpdate(const Value: TTextureUpdate); override;
      public
        constructor Create(const AParent: TGeneratedCubeMapTextureNode);
      end;
    var
      FGenTexFunctionality: TGenCubeMapGenTex;
  public
    constructor Create(const AX3DName: String = ''; const ABaseUrl: String = ''); override;
  {$I auto_generated_node_helpers/x3dnodes_generatedcubemaptexture.inc}
  end;

  { Cube environment map texture loaded from a single file, like DDS. }
  TImageCubeMapTextureNode = class(TAbstractEnvironmentTextureNode)
  strict private
    FAlphaChannelData: TAlphaChannel;
  strict protected
    function AlphaChannelData: TAlphaChannel; override;
  public
    constructor Create(const AX3DName: String = ''; const ABaseUrl: String = ''); override;

    { Load cube environment map from a composite (DDS, KTX...) image.

      In case of problems, will make WritelnWarning.
      This includes situations when url cannot be loaded for whatever reason.
      Also problems when url contains valid Composite image,
      but not describing cube map with all 6 sides.

      If all URLs failed, will return @nil.

      Although the loaded image is not saved here, we do save the AlphaChannel
      type. Our AlphaChannel method will reflect last loaded image. }
    function LoadImage: TCompositeImage;

  {$I auto_generated_node_helpers/x3dnodes_imagecubemaptexture.inc}
  end;

{$endif read_interface}

{$ifdef read_implementation}

{ TComposedCubeMapTextureNode ------------------------------------------------ }

function TComposedCubeMapTextureNode.LoadSides: boolean;

  { Checks is given side has non-nil valid node class,
    and then if image there can be loaded. }
  function SideLoaded(SideField: TSFNode): boolean;
  var
    SideTex: TAbstractTexture2DNode;
  begin
    Result :=
      (SideField.Value <> nil) and
      (SideField.Value is TAbstractTexture2DNode);
    if Result then
    begin
      SideTex := TAbstractTexture2DNode(SideField.Value);
      Result := SideTex.IsTextureImage;

      if Result and not (SideTex.TextureImage is TCastleImage) then
      begin
        WritelnWarning('VRML/X3D', 'ComposedCubeMapTexture cannot contain images compressed GPU-compression algorithms, as we have to rotate images within, and we cannot do this (fast) with compressed textures');
        Result := false;
      end;

      { If any slice has full-range alpha, then assume whole texture does. }
      AlphaMaxVar(FAlphaChannelData, SideTex.AlphaChannelFinal);
    end;
  end;

begin
  FAlphaChannelData := acNone;
  Result :=
    SideLoaded(FdBack) and
    SideLoaded(FdBottom) and
    SideLoaded(FdFront) and
    SideLoaded(FdLeft) and
    SideLoaded(FdRight) and
    SideLoaded(FdTop);
end;

function TComposedCubeMapTextureNode.AlphaChannelData: TAlphaChannel;
begin
  Result := FAlphaChannelData;
end;

{ TGeneratedCubeMapTextureNode.TGenCubeMapGenTex -------------------------------- }

constructor TGeneratedCubeMapTextureNode.TGenCubeMapGenTex.Create(const AParent: TGeneratedCubeMapTextureNode);
begin
  inherited Create(AParent);
  FParent := AParent;
end;

function TGeneratedCubeMapTextureNode.TGenCubeMapGenTex.GetUpdate: TTextureUpdate;
begin
  Result := FParent.Update;
end;

procedure TGeneratedCubeMapTextureNode.TGenCubeMapGenTex.SetUpdate(const Value: TTextureUpdate);
begin
  FParent.Update := Value;
end;

{ TGeneratedCubeMapTextureNode ----------------------------------------------- }

constructor TGeneratedCubeMapTextureNode.Create(const AX3DName, ABaseUrl: String);
begin
  inherited;
  FGenTexFunctionality := TGenCubeMapGenTex.Create(Self);
  AddFunctionality(FGenTexFunctionality);
end;

{ TImageCubeMapTextureNode --------------------------------------------------- }

constructor TImageCubeMapTextureNode.Create(const AX3DName, ABaseUrl: String);
begin
  inherited;
  AddFunctionality(TUrlFunctionality.Create(Self));
end;

function TImageCubeMapTextureNode.LoadImage: TCompositeImage;
var
  I: Integer;
  FullUrl: string;
begin
  Result := TCompositeImage.Create;
  try
    for I := 0 to FdUrl.Items.Count - 1 do
    begin
      FullUrl := PathFromBaseUrl(FdUrl.Items[I]);

      if not TCompositeImage.MatchesURL(FullUrl) then
      begin
        WritelnWarning('VRML/X3D', Format('Only composite (DDS, KTX...) format is supported for ImageCubeMapTexture node, but URL is "%s"', [FullUrl]));
        Continue;
      end;

      try
        Result.LoadFromFile(FullUrl);
      except
        on E: Exception do
        begin
          Result.Close;
          WritelnWarning('VRML/X3D', Format('Error when loading composite file "%s": %s', [FullUrl, E.Message]));
          Continue;
        end;
      end;

      FAlphaChannelData := Result.Images[0].AlphaChannel;

      if Result.CompositeType <> ctCubeMap then
      begin
        Result.Close;
        WritelnWarning('VRML/X3D', Format('Composite image "%s" given for ImageCubeMapTexture doesn''t describe a cube map texture', [FullUrl]));
        Continue;
      end;

      if Result.CubeMapSides <> AllCubeMapSides then
      begin
        Result.Close;
        WritelnWarning('VRML/X3D', Format('Composite image "%s" given for ImageCubeMapTexture doesn''t contain all cube map sides', [FullUrl]));
        Continue;
      end;

      Exit;
    end;

    { If we got here, then no URL was good. So set Result to @nil. }
    FreeAndNil(Result);
  except FreeAndNil(Result); raise end;
end;

function TImageCubeMapTextureNode.AlphaChannelData: TAlphaChannel;
begin
  Result := FAlphaChannelData;
end;

procedure RegisterCubeMapTexturingNodes;
begin
  NodesManager.RegisterNodeClasses([
    TComposedCubeMapTextureNode,
    TGeneratedCubeMapTextureNode,
    TImageCubeMapTextureNode
  ]);
end;

{$endif read_implementation}
