/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.internal.statusbar;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.graphics.Rect;
import android.os.Parcel;
import android.os.Parcelable;
import android.view.InsetsFlags;
import android.view.ViewDebug;
import android.view.WindowInsetsController.Appearance;

import com.android.internal.util.DataClass;

/**
 * Details about the letterbox state of an app.
 */
@DataClass(
        genParcelable = true,
        genAidl = true,
        genToString = true,
        genGetters = false,
        genEqualsHashCode = true
)
public class LetterboxDetails implements Parcelable {

    @NonNull
    private final Rect mLetterboxInnerBounds;
    @NonNull
    private final Rect mLetterboxFullBounds;
    private final int mAppAppearance;

    /**
     * Returns the bounds of the inner letterbox (app content).
     *
     * <p>When an app is letterboxed, it is not using the full bounds of its window. Here we return
     * the bounds that are being used for the app content.
     *
     * <pre>
     * +-------+---------+-------+
     * |       |         |       |
     * |       |         |       |
     * | Outer |  Inner  | Outer |
     * |       |         |       |
     * |       |         |       |
     * +-------+-------- +-------+
     * <pre>
     */
    @NonNull
    public Rect getLetterboxInnerBounds() {
        return mLetterboxInnerBounds;
    }

    /**
     * Returns the full bounds of the letterbox.
     *
     * <p>These are the entire bounds of the window where the app is placed. We cannot assume that
     * the full bounds are the bounds of the screen, as the app can be in split-screen, or can have
     * some margin due to display cutouts.
     *
     * <pre>
     * ---- Full bounds width ----
     * +-------+---------+-------+ |
     * |       |         |       | |
     * |       |         |       | |
     * | Outer |  Inner  | Outer | + Full bounds height
     * |       |         |       | |
     * |       |         |       | |
     * +-------+-------- +-------+ |
     * </pre>
     */
    @NonNull
    public Rect getLetterboxFullBounds() {
        return mLetterboxFullBounds;
    }

    /**
     * Returns the {@link Appearance} of the inner letterbox (app content).
     */
    @Appearance
    public int getAppAppearance() {
        return mAppAppearance;
    }

    /** Returns a string representation of the {@link #getAppAppearance()} property. */
    public String appAppearanceToString() {
        return ViewDebug.flagsToString(InsetsFlags.class, "appearance", mAppAppearance);
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/com/android/internal/statusbar/LetterboxDetails.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    @DataClass.Generated.Member
    public LetterboxDetails(
            @NonNull Rect letterboxInnerBounds,
            @NonNull Rect letterboxFullBounds,
            int appAppearance) {
        this.mLetterboxInnerBounds = letterboxInnerBounds;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mLetterboxInnerBounds);
        this.mLetterboxFullBounds = letterboxFullBounds;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mLetterboxFullBounds);
        this.mAppAppearance = appAppearance;

        // onConstructed(); // You can define this method to get a callback
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "LetterboxDetails { " +
                "letterboxInnerBounds = " + mLetterboxInnerBounds + ", " +
                "letterboxFullBounds = " + mLetterboxFullBounds + ", " +
                "appAppearance = " + appAppearanceToString() +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public boolean equals(@Nullable Object o) {
        // You can override field equality logic by defining either of the methods like:
        // boolean fieldNameEquals(LetterboxDetails other) { ... }
        // boolean fieldNameEquals(FieldType otherValue) { ... }

        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        @SuppressWarnings("unchecked")
        LetterboxDetails that = (LetterboxDetails) o;
        //noinspection PointlessBooleanExpression
        return true
                && java.util.Objects.equals(mLetterboxInnerBounds, that.mLetterboxInnerBounds)
                && java.util.Objects.equals(mLetterboxFullBounds, that.mLetterboxFullBounds)
                && mAppAppearance == that.mAppAppearance;
    }

    @Override
    @DataClass.Generated.Member
    public int hashCode() {
        // You can override field hashCode logic by defining methods like:
        // int fieldNameHashCode() { ... }

        int _hash = 1;
        _hash = 31 * _hash + java.util.Objects.hashCode(mLetterboxInnerBounds);
        _hash = 31 * _hash + java.util.Objects.hashCode(mLetterboxFullBounds);
        _hash = 31 * _hash + mAppAppearance;
        return _hash;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeTypedObject(mLetterboxInnerBounds, flags);
        dest.writeTypedObject(mLetterboxFullBounds, flags);
        dest.writeInt(mAppAppearance);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    protected LetterboxDetails(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        Rect letterboxInnerBounds = (Rect) in.readTypedObject(Rect.CREATOR);
        Rect letterboxFullBounds = (Rect) in.readTypedObject(Rect.CREATOR);
        int appAppearance = in.readInt();

        this.mLetterboxInnerBounds = letterboxInnerBounds;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mLetterboxInnerBounds);
        this.mLetterboxFullBounds = letterboxFullBounds;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mLetterboxFullBounds);
        this.mAppAppearance = appAppearance;

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<LetterboxDetails> CREATOR
            = new Parcelable.Creator<LetterboxDetails>() {
        @Override
        public LetterboxDetails[] newArray(int size) {
            return new LetterboxDetails[size];
        }

        @Override
        public LetterboxDetails createFromParcel(@NonNull Parcel in) {
            return new LetterboxDetails(in);
        }
    };

    @DataClass.Generated(
            time = 1656941109526L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/com/android/internal/statusbar/LetterboxDetails.java",
            inputSignatures = "private final @android.annotation.NonNull android.graphics.Rect mLetterboxInnerBounds\nprivate final @android.annotation.NonNull android.graphics.Rect mLetterboxFullBounds\nprivate final  int mAppAppearance\npublic @android.annotation.NonNull android.graphics.Rect getLetterboxInnerBounds()\npublic @android.annotation.NonNull android.graphics.Rect getLetterboxFullBounds()\npublic @android.view.WindowInsetsController.Appearance int getAppAppearance()\npublic  java.lang.String appAppearanceToString()\nclass LetterboxDetails extends java.lang.Object implements [android.os.Parcelable]\n@com.android.internal.util.DataClass(genParcelable=true, genAidl=true, genToString=true, genGetters=false, genEqualsHashCode=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
