/*-
 * Copyright (c) 2001 Dag-Erling Smrgrav
 * Copyright (c) 1993 Jan-Simon Pendry
 * Copyright (c) 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Jan-Simon Pendry.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)procfs_vfsops.c	8.7 (Berkeley) 5/10/95
 *
 * $FreeBSD: stable/6/sys/fs/procfs/procfs.c 166707 2007-02-14 10:24:50Z kib $
 */

#include <sys/param.h>
#include <sys/queue.h>
#include <sys/exec.h>
#include <sys/lock.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mount.h>
#include <sys/mutex.h>
#include <sys/proc.h>
#include <sys/sbuf.h>
#include <sys/sysproto.h>
#include <sys/systm.h>
#include <sys/vnode.h>

#include <vm/vm.h>
#include <vm/pmap.h>
#include <vm/vm_param.h>

#include <fs/pseudofs/pseudofs.h>
#include <fs/procfs/procfs.h>

/*
 * Filler function for proc/pid/self
 */
int
procfs_doprocfile(PFS_FILL_ARGS)
{
	char *fullpath = "unknown";
	char *freepath = NULL;
	struct vnode *textvp;
	int err;

	textvp = p->p_textvp;
	VI_LOCK(textvp);
	vholdl(textvp);
	err = vn_lock(textvp, LK_EXCLUSIVE | LK_INTERLOCK, td);
	vdrop(textvp);
	if (err)
		return (err);
	vn_fullpath(td, textvp, &fullpath, &freepath);
	VOP_UNLOCK(textvp, 0, td);
	sbuf_printf(sb, "%s", fullpath);
	if (freepath)
		free(freepath, M_TEMP);
	return (0);
}

/*
 * Filler function for proc/curproc
 */
int
procfs_docurproc(PFS_FILL_ARGS)
{
	sbuf_printf(sb, "%ld", (long)td->td_proc->p_pid);
	return (0);
}

/*
 * Adjust mode for some nodes that need it
 */
int
procfs_attr(PFS_ATTR_ARGS)
{
	PROC_LOCK_ASSERT(p, MA_OWNED);

	/* XXX inefficient, split into separate functions */
	if (strcmp(pn->pn_name, "ctl") == 0 ||
	    strcmp(pn->pn_name, "note") == 0 ||
	    strcmp(pn->pn_name, "notepg") == 0)
		vap->va_mode = 0200;
	else if (strcmp(pn->pn_name, "mem") == 0 ||
	    strcmp(pn->pn_name, "regs") == 0 ||
	    strcmp(pn->pn_name, "dbregs") == 0 ||
	    strcmp(pn->pn_name, "fpregs") == 0)
		vap->va_mode = 0600;

	if ((p->p_flag & P_SUGID) && pn->pn_type != pfstype_procdir)
		vap->va_mode = 0;

	vap->va_uid = p->p_ucred->cr_uid;
	vap->va_gid = p->p_ucred->cr_gid;

	return (0);
}

/*
 * Visibility: some files only exist for non-system processes
 * Non-static because linprocfs uses it.
 */
int
procfs_notsystem(PFS_VIS_ARGS)
{
	PROC_LOCK_ASSERT(p, MA_OWNED);
	return ((p->p_flag & P_SYSTEM) == 0);
}

/*
 * Visibility: some files are only visible to process that can debug
 * the target process.
 */
int
procfs_candebug(PFS_VIS_ARGS)
{
	PROC_LOCK_ASSERT(p, MA_OWNED);
	return ((p->p_flag & P_SYSTEM) == 0 && p_candebug(td, p) == 0);
}

/*
 * Constructor
 */
static int
procfs_init(PFS_INIT_ARGS)
{
	struct pfs_node *root;
	struct pfs_node *dir;
	struct pfs_node *node;

	root = pi->pi_root;

	pfs_create_link(root, "curproc", procfs_docurproc,
	    NULL, NULL, 0);

	dir = pfs_create_dir(root, "pid",
	    procfs_attr, NULL, PFS_PROCDEP);
	pfs_create_file(dir, "cmdline", procfs_doproccmdline,
	    NULL, NULL, PFS_RD);
	pfs_create_file(dir, "ctl", procfs_doprocctl,
	    procfs_attr, NULL, PFS_WR);
	pfs_create_file(dir, "dbregs", procfs_doprocdbregs,
	    procfs_attr, procfs_candebug, PFS_RDWR|PFS_RAW);
	pfs_create_file(dir, "etype", procfs_doproctype,
	    NULL, NULL, PFS_RD);
	pfs_create_file(dir, "fpregs", procfs_doprocfpregs,
	    procfs_attr, procfs_candebug, PFS_RDWR|PFS_RAW);
	pfs_create_file(dir, "map", procfs_doprocmap,
	    NULL, procfs_notsystem, PFS_RD);
	node = pfs_create_file(dir, "mem", procfs_doprocmem,
	    procfs_attr, procfs_candebug, PFS_RDWR|PFS_RAW);
	node->pn_ioctl = procfs_ioctl;
	node->pn_close = procfs_close;
	pfs_create_file(dir, "note", procfs_doprocnote,
	    procfs_attr, procfs_candebug, PFS_WR);
	pfs_create_file(dir, "notepg", procfs_doprocnote,
	    procfs_attr, procfs_candebug, PFS_WR);
	pfs_create_file(dir, "regs", procfs_doprocregs,
	    procfs_attr, procfs_candebug, PFS_RDWR|PFS_RAW);
	pfs_create_file(dir, "rlimit", procfs_doprocrlimit,
	    NULL, NULL, PFS_RD);
	pfs_create_file(dir, "status", procfs_doprocstatus,
	    NULL, NULL, PFS_RD);

	pfs_create_link(dir, "file", procfs_doprocfile,
	    NULL, procfs_notsystem, 0);

	return (0);
}

/*
 * Destructor
 */
static int
procfs_uninit(PFS_INIT_ARGS)
{
	/* nothing to do, pseudofs will GC */
	return (0);
}

PSEUDOFS(procfs, 1);
